/************************************************************
*   OWNER.C - Sets the owner of all files in a directory to
*   a specified command line parameter.
*
*   MUST be compiled with byte alignment of structures.
*       eg. Microsoft
*           CL /Zp1 OWNER.C
*
*   Chris Owen, QUT Library, Brisbane, Australia
*   4/8/89
*/

#include <stdio.h>
#include <process.h>
#include <dos.h>
#include <direct.h>
#include <stdlib.h>
#include <string.h>

#define FALSE 0
#define TRUE  !FALSE
#define MAX_PATH_LEN    128
#define MAX_NAME_LEN    48
#define BIND_TYPE_USER  256

#define dir_hndl(drv)   bdos (0xE9, (drive), 0)

union  REGS  regs;
struct SREGS sregs;

    /* Netware Packets */
struct {
    unsigned PacketLength;
    char     Function;
    char     FileAttributes;
    char     ExtAttributes;
    char     Reserved1[4];
    unsigned CreateDate;
    unsigned LastAccessDate;
    unsigned LastUpdateDate;
    unsigned LastUpdateTime;
    long     UniqueOwnerId;
    unsigned LastArchiveDate;
    unsigned LastArchiveTime;
    char     Reserved2[56];
    char     Base;
    char     SearchAttributes;
    char     PathLength;
    char     Path[MAX_PATH_LEN];
    }set_file_rq;

struct {
    unsigned PacketLength;
    char     Function;
    unsigned SeqNo;
    char     Base;
    char     FileAttributes;
    char     PathLength;
    char     Path[MAX_PATH_LEN];
    }scan_dir_rq;

struct {
    unsigned PacketLength;
    unsigned SeqNo;
    char     FileName[14];
    char     FileAttributes;
    char     ExtAttributes;
    long     FileSize;
    unsigned CreateDate;
    unsigned LastAccessDate;
    unsigned LastUpdateDate;
    unsigned LastUpdateTime;
    long     UniqueOwnerId;
    unsigned LastArchiveDate;
    unsigned LastArchiveTime;
    char     Reserved[56];
    }scan_dir_rp;

struct{
    unsigned PacketLength;
    char     Function;
    unsigned Type;
    char     NameLength;
    char     Name[MAX_NAME_LEN];
    }scan_obj_rq;

struct{
    unsigned PacketLength;
    long     ObjectID;
    unsigned Type;
    char     Name[MAX_NAME_LEN];
    }scan_obj_rp;

/*************************************************************************
*  usage () - display usage info and exit to DOS
*/
void usage (char *msg)
{
    if (msg && *msg)
        printf ("%s\n\n", msg);
    printf ("Usage: OWNER [?] <name> [switches]\n\n");
    printf ("    where      ? displays this message\n");
    printf ("          <name> is the username to set as the new owner\n");
    printf ("      [switches] are:\n");
    printf ("           /Ffilename  - default is *.*\n");
    printf ("           /S          - recursively process subdirectories\n");
    exit (1);
}

/*************************************************************************
*  tabs () - prints n TAB characters to stdout.
*/
void tabs (int n)
{
    int i;

    for (i=0; i<n; i++)
        printf ("\t");
}

/*************************************************************************
*  netware () - Makes a Netware call requiring only packets as parameters.
*  Returns value of AL after the call to Netware.
*/
int netware (char function, void *request, void *reply)
{
    char far *req, far *rep;

    /* set up request buffer */
    req = (char far *) request;
    rep = (char far *) reply;

    /* set up registers */
    regs.h.ah = function;       /* AH = Netware function */
    regs.x.si = FP_OFF (req);   /* set DS:SI to the request buffer */
    sregs.ds  = FP_SEG (req);
    regs.x.di = FP_OFF (rep);   /* set ES:DI to the reply buffer */
    sregs.es  = FP_SEG (rep);

    /* call Netware shell */
    return (intdosx (&regs, &regs, &sregs) & 0xFF);
}

/*************************************************************************
*  get_fileinfo () - gets information on filename via Netware function
*   E3h (0Fh) returned in scan_dir_rp.
*/
int get_fileinfo (unsigned dir_handle, char *filename)
{
    scan_dir_rq.PacketLength     = sizeof(scan_dir_rq) - sizeof(scan_dir_rq.PacketLength);
    scan_dir_rq.Function         = 0x0F;
    scan_dir_rq.SeqNo            = 0xFFFF;
    scan_dir_rq.Base             = dir_handle;
    scan_dir_rq.FileAttributes   = 0xFF;
    scan_dir_rq.PathLength       = strlen (filename);
    strcpy (scan_dir_rq.Path, filename);

    scan_dir_rp.PacketLength     = sizeof(scan_dir_rp) - sizeof(scan_dir_rp.PacketLength);

    netware (0xE3, &scan_dir_rq, &scan_dir_rp);
}

/*************************************************************************
*  set_fileinfo () - sets file information for filename to that received
*  from the last call to get_fileinfo (), with the exception of owner which
*  is set to the parameter owner.
*/
int set_fileinfo (unsigned dir_handle, char *filename, long owner)
{
    long reply = 0L;

    /* set up request buffer */
    set_file_rq.PacketLength     = sizeof (set_file_rq);
    set_file_rq.Function         = 0x10;
    set_file_rq.FileAttributes   = scan_dir_rp.FileAttributes;
    set_file_rq.ExtAttributes    = scan_dir_rp.ExtAttributes;
    set_file_rq.CreateDate       = scan_dir_rp.CreateDate;
    set_file_rq.LastAccessDate   = scan_dir_rp.LastAccessDate;
    set_file_rq.LastUpdateDate   = scan_dir_rp.LastUpdateDate;
    set_file_rq.LastUpdateTime   = scan_dir_rp.LastUpdateTime;
    set_file_rq.UniqueOwnerId    = owner;
    set_file_rq.LastArchiveDate  = scan_dir_rp.LastArchiveDate;
    set_file_rq.LastArchiveTime  = scan_dir_rp.LastArchiveTime;
    set_file_rq.Base             = dir_handle;
    set_file_rq.SearchAttributes = 0xFF;
    set_file_rq.PathLength       = strlen(filename);
    strncpy (set_file_rq.Path, filename, MAX_PATH_LEN);

    return (netware (0xE3, &set_file_rq, &reply));
}

/*************************************************************************
* set_owner () : sets the owner of files matching filename to the
*  parameter owner.
*/
int set_owner (unsigned dir_handle, char *filename, long owner)
{
    int err;

    if (!get_fileinfo (dir_handle, filename))
        set_fileinfo (dir_handle, filename, owner);
}

/*************************************************************************
* owner_id () - Searches the bindery for the USER object with name
*  owner_name.
* Returns the id of the object if found, otherwise 0.
*/
long owner_id (char *owner)
{
    /* set up request buffer */
    scan_obj_rq.NameLength   = (char) strlen (owner);
    if (scan_obj_rq.NameLength >= MAX_NAME_LEN)
        usage ("User name exceeds maximum length!");
    scan_obj_rq.PacketLength = sizeof (scan_obj_rq) - sizeof (scan_obj_rq.Name) +
                               scan_obj_rq.NameLength - sizeof (scan_obj_rq.PacketLength);
    scan_obj_rq.Function     = 0x35;
    scan_obj_rq.Type         = BIND_TYPE_USER;
    strncpy (scan_obj_rq.Name, owner, MAX_NAME_LEN);

    scan_obj_rp.PacketLength = sizeof (scan_obj_rp) - sizeof (scan_obj_rq.PacketLength);

    if (netware (0xE3, &scan_obj_rq, &scan_obj_rp))
        return (0L);
    else
        return (scan_obj_rp.ObjectID);
}

/*************************************************************************
*  do_files () - process all files matching filename, setting the owner
*   to owner.  If sub != 0, then all subdirectories are recursively
*   processed.
*/
void do_files (unsigned drive, char *filename, long owner, int sub, int level)
{
    struct find_t buff;
    int found;

    if (!owner)
        usage ("You must specify the owner!\7");

    /* do files */
    if (!(found = !_dos_findfirst (filename, 0xFF, &buff))){
        tabs (level);
        printf ("no files found matching %s\n", filename);
        }

    while (found){
        if (!(buff.attrib & _A_SUBDIR)){
            set_owner (dir_hndl (drive), buff.name, owner);
            tabs (level);
            printf ("%s\n",buff.name);
            }
        found = !_dos_findnext (&buff);
        }

    /* do sub directories */
    if (sub){
        if ((found = !_dos_findfirst ("*.*", 0xFF, &buff))){
            while (found){
                if (buff.attrib & _A_SUBDIR){
                    if (strcmp (buff.name, ".") && strcmp (buff.name, "..")){
                        chdir (buff.name);
                        tabs (level);
                        printf ("---> %s :\n", buff.name);
                        do_files (drive, filename, owner, sub, level+1);
                        chdir ("..");
                        }
                    }
                found = !_dos_findnext (&buff);
                }
            }
        }
}

main (int argc, char *argv[])
{
    char     attr = _A_NORMAL;
    int      sub = FALSE;
    unsigned drv, drv_tst, old_drv, drives;
    char     drive  [_MAX_DRIVE],
             dir    [_MAX_DIR],
             fname  [_MAX_FNAME + _MAX_EXT],
             ext    [_MAX_EXT],
             old_dir[_MAX_DIR];
    char     *filename = "*.*",
             *owner    = NULL;

    printf ("OWNER 1.0 - Changes the owner of files under Novell Netware.\n");
    printf ("Chris Owen, Queensland University of Technology, Australia.\n\n");

    if (argc < 2)
        usage ("arg count!");
    _dos_getdrive (&old_drv);
    getcwd (old_dir, _MAX_DIR);
    drv = old_drv - 1;                       /* Netware has drive A: = 0 */

    /* process command line */
    while (--argc){
        switch (argv[argc][0]){
            case '/':
            case '-':
                switch (argv[argc][1]){
                    case 'S':                /* do sub directories */
                    case 's':
                        sub = TRUE;
                        break;
                    case 'F':                /* match filespec */
                    case 'f':
                        _splitpath (&argv[argc][2], drive, dir, fname, ext);
                        if (!(*drive || *dir || *fname || *ext))
                            usage ("invalid filename");
                        if (*fname || *ext)
                            filename = strcat (fname, ext);
                        if (*drive){
                            drv = toupper (*drive) - 'A';
                            _dos_setdrive (drv+1, &drives);
                            _dos_getdrive (&drv_tst);
                            if (--drv_tst != drv)
                                return (perror (drive));
                            }
                        if (*dir)
                            if (chdir (dir))
                                return (perror (dir));
                        break;
                    default:
                        usage ("invalid switch!\7");
                    }
                break;
            case '?':
                usage (NULL);
            default:
                owner = argv[argc];
                break;
            }
        }

    /* do job */
    do_files (drv, filename, owner_id (owner), sub, 0);

    _dos_setdrive (old_drv, &drives);
    chdir (old_dir);
}
