 {ͻ
   unit : Novell queue                          Target Platforms         
   desc : Novell queue functions                  - DOS Real             
                                                  - DOS Protected        
                                                  - Windows              
   (C) 1994 by Christoph Weller 16/01/93 v1.1                            
  ͼ}

(*

  18.1.1995  : Inserted all possible error-results of a procedure/function
               into the description-header.

*)

unit NVQueue;

interface

{$I NVDEFINE.INC}

uses
  Strings,  { Borland strings unit                      }
  NvMain;   { Novell main unit                          }

{
              CONST declarations
 }

const

  { (q)ueue (s)tatus flags }
  qsOpDisabledNewJobs           = $01; { Operator disabled addition of
                                         new jobs                      }
  qsOpRefusesAddJobAttach       = $02; { Operators refused additional
                                         job servers attaching         }
  qsOpDisabledJobServicing      = $04; { Operator disabled Job
                                         servicing                     }

  { (Q)ueue services (j)ob control flags }
  qjReady                       = $10; { Job ready to be serviced            }
  qjServedAutomat               = $04; { Job will be serviced automatically
                                         if connection broken                }
  qjRemainInQueue               = $08; { Job remains in queue after server
                                         aborts job                          }
  qjNoFilledJobFile             = $10; { Client has not filled associated
                                         job File                            }
  qjUserHoldJob                 = $40; { User Hold Job advances, but cannot
                                         be serviced until this bit is
                                         cleared by User or Operator         }
  qjOperatorHoldJob             = $80; { Operator Hold Job advances, but
                                         cannot be serviced until this bit
                                         is cleared by an Operator           }

{
              TYPE declarations
 }

type

  { queue record }
  tQueue        =
  record
    QueueId    : LongInt; { Object-id of queue                         }
    Client     : Byte;    { Client station                             }
    ClientTask : Byte;    { Client task number                         }
    ClientId   : LongInt; { Client id                                  }
    TargetId   : LongInt; { Target server id
         	            FFFFFFFh if any server acceptable          }
    Execution  : array[1..6] of Byte;
                 { Execution time year, month, day, hour, minute, second
                   FFFFFFFFFFFF to execute as soon as possible         }
    JobTime    : array[1..6] of Byte;
                 { Job entry time, format see above                    }
    JobNo      : Word;    { Job-number                                 }
    JobType    : Word;    { Job-type                                   }
    JobPos     : Byte;    { Job-position                               }
    JobFlags   : Byte;    { Job control flags                          }
    JobName    : String[14]; { Job-filename                            }
    JobHandle  : array[1..6] of Byte;  { Job filehandle                }
    ServerSt   : Byte;    { Server station                             }
    ServerTask : Byte;    { Server tasknumber                          }
    ServerId   : LongInt; { Server object-id                           }
    JobDesc    : String[50]; { Job description string                  }
    { Client Record Area }
    Unused     : Byte;    { Unused ????                                }
    TabSize    : Byte;    { Tab-size                                   }
    Copies     : Word;    { Copies                                     }
    Unused2    : Byte;    { Unused ??                                  }
    PrintFlags : Byte;    { Printflags                                 }
    ClientRec  : array[1..39] of Char; { Client rec unused ???         }
    ClientName : String[13]; { Client-name                             }
    BannerName : String[13]; { Banner-name                             }
    JName      : String[14]; { Job-name                                }
    PathName   : String[80]; { Path-name                               }
  end;

  { queue status }
  tQueueStatus  =
  record
    QueueId        : LongInt; { Object id of queue                         }
    QStatus        : Byte;    { status of queue
	                        bit 0: operator disabled addition of new
                                       jobs
                                bit 1: operator refuses additional job
                                       servers attaching
	                        bit 2: operator disabled job servicing    }
    QueueJobs      : Byte;    { Number of jobs in queue                   }
    ServerAttached : Byte;    { Number of servers attached to queue       }
    AttServerList  : array[1..25] of LongInt;
                     { List of Objects-ids of attached servers            }
    AttStList      : array[1..25] of Byte;
                     { List of attached server stations                   }
  end;

  { queue list }
  tQueueList    =
  record
    JobCount   : Word;                  { Job count                        }
    Jobs       : array[1..249] of Word; { List of jobs numbers by position
                                          in queue                         }
    MaxJobs    : Word;                  { Maximum job numbers              }
  end;

  { Server status record }
  tServerStatusRecord = array[1..64] of Byte;

  { Server status record}
  tServerStatusReply     =
  record
    Size                 : Word;                 { size of reply      }
    ServerStatusRecord   : tServerStatusRecord;  { status record      }
  end;

{
        procedure and function headers
 }

{ QUEUE SERVICES Create queue }
function CreateQueue(var QueueId : LongInt;
                     QueueName,
                     Path : String;
                     Handle : Byte):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Destroy queue }
function DestroyQueue(QueueId : LongInt):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Read queue status }
function ReadQueueStatus(var QueueStat : tQueueStatus):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Set queue status }
function SetQueueStatus(QueueId : LongInt;
                        QueueStat :Byte):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Create queue job and file }
function CreateQueueJobAndFile(var Job : tQueue):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Close file and start job }
function CloseFileStartJob(QueueId : LongInt;
                           JobNumber : Word):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Remove job from queue }
function RemoveJobFromQueue(QueueId : LongInt;
                            JobNumber : Word):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Get queue job list }
function GetQueueJobList(QueueId : LongInt;
                         var QList : tQueueList):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Read queue job entry }
function ReadQueueJobEntry(var Job : tQueue):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Change queue job entry }
function ChangeQueueJobEntry(Job : tQueue):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Change queue job-position }
function ChangeQueueJobPos(QueueId : LongInt;
                          JobNo : Word;
                          NewPos : Byte):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Attach queue server to queue }
function AttachQServerToQueue(QueueId : LongInt):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Deattach queue server from queue }
function DeAttachQServerFromQueue(QueueId : LongInt):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Service queue job and open file }
function ServiceQueueJobOpenFile(QueueId : LongInt;
                                 TargetJobType : Word):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Finish servicing job and file }
function FinishServicingJob(QueueId : LongInt;
                            JobNo : Word;
                            Charge : LongInt):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Abort servicing queue job and file }
function AbortServicingQueue(QueueId : LongInt;
                             JobNo : Word):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Change to client rights }
function ChangeToClientRights(QueueId : LongInt;
                              JobNo : Word):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Restore queue server rights }
function RestoreQueueServerRights:Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Read queue server current status }
function ReadQueueServerCurrentStatus(QueueId : LongInt;
                                      ServerId : LongInt;
                                      ServerStation : Byte;
                                      var StatusRec : tServerStatusReply):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Set queue server current status }
function SetQueueServerCurrentStatus(QueueId : LongInt;
                                     StatusRec : tServerStatusRecord):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

{ QUEUE SERVICES Get queue job file size }
function GetQueueJobFileSize(var JobSize : LongInt;
                             QueueId : LongInt;
                             JobNo : Word):Byte;
{$IFDEF PROTECTED} export; {$ENDIF}

(* -------------------------- implemenation -------------------------- *)

implementation

{
  function        : CreateQueue
  service type    : queue functions
  Netware version : NW v2.1 - v4.x

  Desc : Creates an new queue with users,
         operators and servers.

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neDirectoryFull
           - neInvalidDirHandle1
           - neInvalidPath
           - nePropertyExists
           - neObjectExists
           - neInvalidName
           - neNoWildcardAllowed
           - neInvalidBindSecLevel
           - neNoCreateObjPrivilege
           - neNoCreatePropPriv
           - neNoSuchObject
           - neBinderyLocked
           - neBinderyError
 }

function CreateQueue(var QueueId : LongInt;
                     QueueName,
                     Path : String;
                     Handle : Byte):Byte;

var
  NovRegs       : Word;
  Marker        : Byte;

  Request       :
  record
    FLen        : Word; { Length of request         }
    SubF        : Byte; { Subfunction               }
    QueueT      : Word; { Queue rype  (big-endian)  }
    Queue       : array[1..168] of Char;
                  { 1   Byte Length of queue name
                    48  Bytes queue name
                    1   Byte directory handle
                    1   Byte length of pathname
                    118 Bytes path of directory
                        where to create the queue   }
  end;

  Reply         :
  record
    Size        : Word;     { Size of reply             }
    ObjId       : LongInt;  { Object id of queue        }
  end;

begin
  Reply.Size := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    SubF      := $64;
    QueueT    := Swap(3); { Object Typ = 3 for Queue }
    Marker    := 1;
    QueueName := StupCase(QueueName);
    Move(QueueName, Ptr(Seg(Queue[Marker]), Ofs(Queue[Marker]))^,
         Length(QueueName) + 1);
    Inc(Marker, Length(QueueName) + 1);
    Queue[Marker] := Chr(Handle);
    Inc(Marker);
    Path := StupCase(Path);
    Move(Path, Ptr(Seg(Queue[Marker]), Ofs(Queue[Marker]))^,
         Length(Path) + 1);
    Inc(Marker, Length(Path) + 1);
    FLen    := 3 + Marker;
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  CreateQueue := Lo(NovRegs);
  QueueId := SwapLongInt(Reply.ObjId)
end;

{
  function        : DestroyQueue
  service type    : queue functions
  Netware version : NW v2.1 - v4.x

  Desc : Destroys the queue and all system-files

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neInvalidPath
           - neQueueError
           - neNoSuchQueue
           - neHardwareError
 }

function DestroyQueue(QueueId : LongInt):Byte;

var
  NovRegs       : Word;
  Reply         : Word;    { Size of Reply                    }

  Request       :
  record
    FLen        : Word;    { Length of request                }
    SubF        : Byte;    { Subfunction                      }
    Queue       : LongInt; { Object id of queue (big-endian)  }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen    := SizeOf(Request) - 2;
    SubF    := $65;
    Queue   := SwapLongInt(QueueId);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  DestroyQueue := Lo(NovRegs);
end;

{
  function        : ReadQueueStatus
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Reads the current status of an queue

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neInvalidPath
           - neNoSuchQueue
           - neNoServerQueue
           - neNoQueueRights
           - neInvalidBindSecLevel
           - neNoSuchObject
           - neBinderyLocked
           - neBinderyError
 }

function ReadQueueStatus(var QueueStat : tQueueStatus):Byte;

var
  NovRegs       : Word;
  i             : Byte;

  Request       :
  record
    FLen        : Word;    { Length of request                }
    SubF        : Byte;    { Subfunction                      }
    Queue       : LongInt; { Object id of queue (big-endian)  }
  end;

  Reply         :
  record
    Size        : Word;    { Reply size                          }
    ObjId       : LongInt; { Object id of queue                  }
    QStat       : Byte;    { Queue status                        }
    QJobs       : Byte;    { Number of jobs in queue             }
    AServer     : Byte;    { Number of attached servers          }
    QList       : array[1..125] of Byte;
                  { 100 Bytes Object id list of attached servers
                    25  Bytes list of attached server stations   }
    MaxList     : Byte;    { Maximum number of servers to return }
  end;

begin
  Reply.Size    := SizeOf(Reply) -2;
  Reply.MaxList := 25;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen    := SizeOf(Request) - 2;
    SubF    := $66;
    Queue   := SwapLongInt(QueueStat.QueueId);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  ReadQueueStatus := Lo(NovRegs);
  with Reply, QueueStat do
  begin
    QueueId        := SwapLongInt(ObjId);
    QStatus        := QStat;
    QueueJobs      := QJobs;
    ServerAttached := AServer;
    FillChar(AttServerList, SizeOf(AttServerList), 0);
    FillChar(AttStList, SizeOf(AttStList), 0);
    Move(QList, AttServerList, (AServer * 4));
    Move(Ptr(Seg(QList), Ofs(QList) + (AServer * 4))^, AttStList, SizeOf(AttStList));
  end;
end;

{
  function        : SetQueueStatus
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Sets the status of the queue.
         Queue-operator rights required

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neInvalidPath
           - neQueueError
           - neNoSuchQueue
           - neNoQueueRights
           - neBinderyLocked
           - neBinderyError
 }

function SetQueueStatus(QueueId : LongInt;
                        QueueStat :Byte):Byte;

var
  NovRegs       : Word;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                }
    SubF        : Byte;    { Subfunction                      }
    Queue       : LongInt; { Object id of queue (big-endian)  }
    QStat       : Byte;    { Queue status                     }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen    := SizeOf(Request) - 2;
    SubF    := $67;
    Queue   := SwapLongInt(QueueId);
    QStat   := QueueStat;
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  SetQueueStatus := Lo(NovRegs);
end;

{
  function        : CreateQueueJobAndFile
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Creates an new job in the queue

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neDirectoryFull
           - neInvalidPath
           - neQueueError
           - neNoSuchQueue
           - neNoServerQueue
           - neNoQueueRights
           - neQueueFull
           - neQueueHalted
           - nePropertyExists
           - neInvalidName
           - neNoWildcardAllowed
           - neInvalidBindSecLevel
           - neNoCreatePropPriv
           - neNoSuchObject
           - neBinderyLocked
           - neBinderyError
 }

function CreateQueueJobAndFile(var Job : tQueue):Byte;

var
  NovRegs       : Word;

  Request       :
  record
    FLen       : Word;    { Length of request                       }
    SubF       : Byte;    { Subfunction                             }
    Queue      : LongInt; { Object id of queue (big-endian)         }
    CliSt      : Byte;    { Client station                          }
    CliTsk     : Byte;    { Client task number                      }
    CliId      : LongInt; { Object id of client (big-endian)        }
    TargetSv   : LongInt; { Object id of target server (big-endian) }
    ExecTime   : array[1..6] of Byte; { Execution time              }
    JEntryT    : array[1..6] of Byte; { Job Entry time              }
    JNumber    : Word;    { Job number (big-endian)                 }
    JTyp       : Word;    { Job type (big-endian)                   }
    JPos       : Byte;    { Job position                            }
    JCFlags    : Byte;    { Job control flags                       }
    JFName     : array[1..14] of Char; { ASCIIZ job file name       }
    JHandle    : array[1..6] of Byte;  { Job file handle            }
    ServSt     : Byte;    { Server station                          }
    ServTsk    : Byte;    { Server task number                      }
    ServId     : LongInt; { Object id of server                     }
    JDesc      : array[1..50] of Char; { ASCIIZ job desc. string    }
    CliRec     : array[1..152] of Byte;{ Client record area         }
  end;

  Reply        :
  record
    Size       : Word;    { Reply size                              }
    CliSt      : Byte;    { Client station                          }
    CliTsk     : Byte;    { Client task                             }
    CliId      : LongInt; { Object id of client (big-endian)        }
    Target     : LongInt; { Object id of target server (big-endian) }
    TargetExec : array[1..6] of Byte; { Target execution time       }
    JEntryT    : array[1..6] of Byte; { Job entry time              }
    JNumber    : Word;    { Job number (big-endian)                 }
    JType      : Word;    { Job type (big-endian)                   }
    JPos       : Byte;    { Job position                            }
    JCFlags    : Byte;    { Job control flags                       }
    JFName     : array[1..14] of Char; { ASCIIZ job filename        }
    JHandle    : array[1..6] of Byte;  { Job file handle            }
    ServSt     : Byte;    { Server station                          }
    ServTsk    : Byte;    { Server task number                      }
    ServId     : LongInt; { Object id of server (big-endian)        }
  end;

begin
  Reply.Size    := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request, Job do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $68;
    Queue    := QueueId;
    CliSt    := Client;
    CliTsk   := ClientTask;
    CliId    := ClientId;
    TargetSv := TargetId;
    Move(Execution, ExecTime, SizeOf(Execution));
    Move(JobTime, JEntryT, SizeOf(JobTime));
    JNumber  := JobNo;
    JTyp     := JobType;
    JPos     := JobPos;
    JCFlags  := JobFlags;
    Move(JobName, JFName, SizeOf(JobName));
    Move(JobHandle, JHandle, SizeOf(JobHandle));
    ServSt   := ServerSt;
    ServTsk  := ServerTask;
    ServId   := ServerId;
    Move(JobDesc, JDesc, SizeOf(JobDesc));
    Move(ClientRec, CliRec, SizeOf(ClientRec));
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  CreateQueueJobAndFile := Lo(NovRegs);
  with Job, Reply do
  begin
    Client     := CliSt;
    ClientTask := CliTsk;
    ClientId   := CliId;
    TargetId   := Target;
    Move(TargetExec, Execution, SizeOf(TargetExec));
    Move(JEntryT, JobTime, SizeOf(JEntryT));
    JobNo      := JNumber;
    JobType    := JType;
    JobPos     := JPos;
    JobFlags   := JCFlags;
    JobName    := JFName;
    Move(JHandle, JobHandle, SizeOf(JHandle));
    ServerSt   := ServSt;
    ServerTask := ServTsk;
    ServerId   := ServId;
  end;
end;

{
  function        : CloseFileStartJob
  service type    : queue functions
  Netware version : NW v2.1 - v4.x

  Desc : Close an new file written to the
         queue and start the job

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neQueueError
           - neNoSuchQueue
           - neNoQueueRights
           - neNoQueueJob
           - neNoJobRights
           - neBinderyLocked
           - neBinderyError
 }

function CloseFileStartJob(QueueId : LongInt;JobNumber : Word):Byte;

var
  NovRegs       : Word;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    Queue       : LongInt; { Object id of queue (big-endian)         }
    JNo         : Word;    { Job number (big-endian)                 }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $69;
    Queue    := SwapLongInt(QueueId);
    JNo      := Swap(JobNumber);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  CloseFileStartJob := Lo(NovRegs);
end;

{
  function        : RemoveJobFromQueue
  service type    : queue functions
  Netware version : NW v2.1 - v4.x

  Desc : Removes an job from the queue

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neQueueError
           - neNoSuchQueue
           - neNoQueueRights
           - neNoQueueJob
           - neNoJobRights
           - neBinderyLocked
           - neBinderyError
 }

function RemoveJobFromQueue(QueueId : LongInt;JobNumber : Word):Byte;

var
  NovRegs       : Word;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    Queue       : LongInt; { Object id of queue (big-endian)         }
    JNo         : Word;    { Job number (big-endian)                 }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $6A;
    Queue    := SwapLongInt(QueueId);
    JNo      := Swap(JobNumber);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  RemoveJobFromQueue := Lo(NovRegs);
end;

{
  function        : GetQueueJobList
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Gets all jobs of an queue

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neInvalidPath
           - neQueueError
           - neNoSuchQueue
           - neNoServerQueue
           - neNoQueueRights
           - neNoSuchObject
           - neBinderyLocked
           - neBinderyError
 }

function GetQueueJobList(QueueId : LongInt;
                         var QList : tQueueList):Byte;

var
  NovRegs       : Word;
  i             : Byte;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    Queue       : LongInt; { Object id of queue (big-endian)         }
  end;

  Reply         :
  record
    Size        : Word;    { Size of reply                           }
    QJobCount   : Word;    { Job count (big-endian)                  }
    QJobs       : array[1..249] of Word;
                  { List of jobs by position in queue                }
    QMaxJobs    : Word;    { Maximum job numbers                     }
  end;

begin
  Reply.Size := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $6B;
    Queue    := SwapLongInt(QueueId);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  GetQueueJobList := Lo(NovRegs);
  with QList, Reply do
  begin
    JobCount   := Swap(QJobCount);
    for i := 1 to 249 do Jobs[i] := Swap(QJobs[i]);
    MaxJobs    := Swap(QMaxJobs);
  end;
end;

{
  function        : ReadQueueJobEntry
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Reads all infos of an queue job-entry

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neQueueError
           - neNoSuchQueue
           - neNoServerQueue
           - neNoQueueRights
           - neNoQueueJob
           - neNoSuchObject
           - neBinderyLocked
           - neBinderyError
 }

function ReadQueueJobEntry(var Job : tQueue):Byte;

var
  NovRegs       : Word;
  i             : Byte;

  Request       :
  record
    FLen       : Word;    { Length of request                       }
    SubF       : Byte;    { Subfunction                             }
    Queue      : LongInt; { Object id of queue (big-endian)         }
    JNo        : Word;    { Job number (big-endian)                 }
  end;

  Reply        :
  record
    Size       : Word;    { Reply size                              }
    CliSt      : Byte;    { Client station                          }
    CliTsk     : Byte;    { Client task                             }
    CliId      : LongInt; { Object id of client (big-endian)        }
    Target     : LongInt; { Object id of target server (big-endian) }
    TargetExec : array[1..6] of Byte; { Target execution time       }
    JEntryT    : array[1..6] of Byte; { Job entry time              }
    JNumber    : Word;    { Job number (big-endian)                 }
    JType      : Word;    { Job type (big-endian)                   }
    JPos       : Byte;    { Job position                            }
    JCFlags    : Byte;    { Job control flags                       }
    JFName     : array[1..14] of Char; { ASCIIZ job filename        }
    JHandle    : array[1..6] of Byte;  { Job file handle            }
    ServSt     : Byte;    { Server station                          }
    ServTsk    : Byte;    { Server task number                      }
    ServId     : LongInt; { Object id of server (big-endian)        }
    JobD       : array[1..50] of Char; { ASCIZ job desc. string     }
    { Client Record Area }
    U          : Byte;    { unused ???                              }
    TabS       : Byte;    { Tab-size                                }
    Cop        : Word;    { Copies                                  }
    U2         : Byte;    { Unused ???                              }
    PrintF     : Byte;    { Printflags                              }
    ClientR    : array[1..26] of Char; { Client record unused ???   }
    ClientN    : array[1..13] of Char; { Client-name                }
    BannerN    : array[1..13] of Char; { Banner-name                }
    JobN       : array[1..14] of Char; { Job-name                   }
    PathN      : array[1..80] of Char; { Path-name                  }
  end;

begin
  Reply.Size := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $6C;
    Queue    := SwapLongInt(Job.QueueId);
    JNo      := Swap(Job.JobNo);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  ReadQueueJobEntry := Lo(NovRegs);
  FillChar(Job, SizeOf(Job), 0);
  with Job, Reply do
  begin
    Client     := CliSt;
    ClientTask := CliTsk;
    ClientId   := SwapLongInt(CliId);
    TargetId   := SwapLongInt(Target);
    Move(TargetExec, Execution, SizeOf(TargetExec));
    Move(JEntryT, JobTime, SizeOf(JEntryT));
    JobNo      := Swap(JNumber);
    JobType    := Swap(JType);
    JobPos     := JPos;
    JobFlags   := JCFlags;
    JobName    := StrPas(@JFName);
    Move(JHandle, JobHandle, SizeOf(JHandle));
    ServerSt   := ServSt;
    ServerTask := ServTsk;
    ServerId   := SwapLongInt(ServId);
    JobDesc    := StrPas(@JobD);
    Unused     := U;
    TabSize    := TabS;
    Copies     := Swap(Cop);
    Unused2    := U2;
    PrintFlags := PrintF;
    Move(ClientR, ClientRec, SizeOf(ClientR));
    ClientName := StrPas(@ClientN);
    BannerName := StrPas(@BannerN);
    JName      := StrPas(@JobN);
    PathName   := StrPas(@PathN);
  end;
end;

{
  function        : ChangeQueueJobEntry
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Changes an job-entry in the queue

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neQueueError
           - neNoSuchQueue
           - neNoQueueJob
           - neQueueServicing
           - neBinderyLocked
           - neBinderyError
 }

function ChangeQueueJobEntry(Job : tQueue):Byte;

var
  NovRegs       : Word;
  i             : Byte;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    Queue       : LongInt; { Object id of queue                      }
    CliSt       : Byte;    { Client station                          }
    CliTsk      : Byte;    { Client task                             }
    CliId       : LongInt; { Object id of client (big-endian)        }
    Target      : LongInt; { Object id of target server (big-endian) }
    TargetExec  : array[1..6] of Byte; { Target execution time       }
    JEntryT     : array[1..6] of Byte; { Job entry time              }
    JNumber     : Word;    { Job number (big-endian)                 }
    JType       : Word;    { Job type (big-endian)                   }
    JPos        : Byte;    { Job position                            }
    JCFlags     : Byte;    { Job control flags                       }
    JFName      : array[1..14] of Char; { ASCIIZ job filename        }
    JHandle     : array[1..6] of Byte;  { Job file handle            }
    ServSt      : Byte;    { Server station                          }
    ServTsk     : Byte;    { Server task number                      }
    ServId      : LongInt; { Object id of server (big-endian)        }
    JobD        : array[1..50] of Char; { ASCIZ job desc. string     }
    { Client Record Area }
    U           : Byte;    { Unused ???                              }
    TabS        : Byte;    { Tab-size                                }
    Cop         : Word;    { Copies                                  }
    U2          : Byte;    { Unused ???                              }
    PrintF      : Byte;    { Printflags                              }
    ClientR     : array[1..26] of Char; { Client record unused ???   }
    ClientN     : array[1..13] of Char; { Client-name                }
    BannerN     : array[1..13] of Char; { Banner-name                }
    JobN        : array[1..14] of Char; { Job-name                   }
    PathN       : array[1..80] of Char; { Path-name                  }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request, Job do
  begin
    FLen       := SizeOf(Request) - 2;
    SubF       := $6D;
    Queue      := SwapLongInt(QueueId);
    CliSt      := Client;
    CliTsk     := ClientTask;
    CliId      := SwapLongInt(ClientId);
    Target     := SwapLongInt(TargetId);
    Move(Execution, TargetExec, SizeOf(Execution));
    Move(JobTime, JEntryT, SizeOf(JobTime));
    JNumber    := Swap(JobNo);
    JType      := Swap(JobType);
    JPos       := JobPos;
    JCFlags    := JobFlags;
    StrPCopy(@JFName, StupCase(JobName));
    Move(JobHandle, JHandle, SizeOf(JobHandle));
    ServSt     := ServerSt;
    ServTsk    := ServerTask;
    ServId     := ServerId;
    StrPCopy(@JobD, JobDesc);
    TabS       := TabSize;
    Cop        := Copies;
    PrintF     := PrintFlags;
    Move(ClientRec, ClientR, SizeOf(ClientRec));
    StrPCopy(@ClientN, StupCase(ClientName));
    StrPCopy(@BannerN, BannerName);
    StrPCopy(@JobN, StupCase(JobName));
    StrPCopy(@PathN, StupCase(PathName));
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  ChangeQueueJobEntry := Lo(NovRegs);
end;

{
  function        : ChangeQueueJobPos
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Changes the position of an job-entry
         in the queue

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neQueueError
           - neNoSuchQueue
           - neNoQueueJob
           - neNoJobRights
           - neBinderyLocked
           - neBinderyError
 }

function ChangeQueueJobPos(QueueId : LongInt;
                           JobNo : Word;
                           NewPos : Byte):Byte;

var
  NovRegs       : Word;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    Queue       : LongInt; { Object id of queue (big-endian)         }
    JNo         : Word;    { Job number (big-endian)                 }
    NewP        : Byte;    { New position in queue                   }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $6E;
    Queue    := SwapLongInt(QueueId);
    JNo      := Swap(JobNo);
    NewP     := NewPos;
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  ChangeQueueJobPos := Lo(NovRegs);
end;

{
  function        : AttachQServer
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Attaches an queue-server to an queue

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neInvalidPath
           - neQueueError
           - neNoSuchQueue
           - neNoQueueRights
           - neQueueHalted
           - neMaxQueueServer
           - neBinderyLocked
           - neBinderyError
 }

function AttachQServerToQueue(QueueId : LongInt):Byte;

var
  NovRegs       : Word;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    Queue       : LongInt; { Object id of queue (big-endian)         }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $6F;
    Queue    := SwapLongInt(QueueId);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  AttachQServerToQueue := Lo(NovRegs);
end;

{
  function        : DeattachQServerFromQueue
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Deattaches an queue-server from an queue

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neInvalidPath
           - neQueueError
           - neNoSuchQueue
           - neNoServerQueue
           - neBinderyLocked
           - neBinderyError
 }

function DeAttachQServerFromQueue(QueueId : LongInt):Byte;

var
  NovRegs       : Word;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    Queue       : LongInt; { Object id of queue (big-endian)         }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $70;
    Queue    := SwapLongInt(QueueId);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  DeAttachQServerFromQueue := Lo(NovRegs);
end;

{
  function        : ServiceQueueJobOpenFile
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : With this function the queue-server
         tries to service the next job in
         the queue

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neInvalidPath
           - neQueueError
           - neNoSuchQueue
           - neNoQueueRights
           - neNoQueueJob
           - neStationIsNotServer
           - neQueueHalted
           - neBinderyLocked
           - neBinderyError
 }

function ServiceQueueJobOpenFile(QueueId : LongInt;
                                 TargetJobType : Word):Byte;

var
  NovRegs       : Word;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    Queue       : LongInt; { Object id of queue (big-endian)         }
    TargetJ     : Word;    { Target job type (big-endian)            }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $71;
    Queue    := SwapLongInt(QueueId);
    TargetJ  := Swap(TargetJobType);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  ServiceQueueJobOpenFile := Lo(NovRegs);
end;

{
  function        : FinishServicingJob
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Servicing of an job succesfull finished

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neQueueError
           - neNoSuchQueue
           - neNoJobRights
 }

function FinishServicingJob(QueueId : LongInt;
                            JobNo : Word;
                            Charge : LongInt):Byte;

var
  NovRegs       : Word;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    Queue       : LongInt; { Object id of queue (big-endian)         }
    JNo         : Word;    { Job number                              }
    QCharge     : LongInt; { Charge                                  }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $72;
    Queue    := SwapLongInt(QueueId);
    JNo      := Swap(JobNo);
    QCharge  := SwapLongInt(Charge);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  FinishServicingJob := Lo(NovRegs);
end;

{
  function        : AbortServicingQueue
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Servicing of an job not successfull
         finished

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neQueueError
           - neNoSuchQueue
           - neNoJobRights
           - neStationIsNotServer
 }

function AbortServicingQueue(QueueId : LongInt;
                             JobNo : Word):Byte;

var
  NovRegs       : Word;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    Queue       : LongInt; { Object id of queue (big-endian)         }
    JNo         : Word;    { Job number                              }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $73;
    Queue    := SwapLongInt(QueueId);
    JNo      := Swap(JobNo);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  AbortServicingQueue := Lo(NovRegs);
end;

{
  function        : ChangeToClientRights
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Change to the rights of the client who
         has put the job in the queue

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neQueueError
 }

function ChangeToClientRights(QueueId : LongInt;
                              JobNo : Word):Byte;

var
  NovRegs       : Word;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    Queue       : LongInt; { Object id of queue (big-endian)         }
    JNo         : Word;    { Job number                              }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $74;
    Queue    := SwapLongInt(QueueId);
    JNo      := Swap(JobNo);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  ChangeToClientRights := Lo(NovRegs);
end;

{
  function        : RestoreQueueServerRights
  service type    : queue functions
  Netware version : NW v2.1x - v4.x

  Desc : Restores the rights after an
         'Change to client rights'

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neInvalidPath
           - neQueueError
           - neNoSuchQueue
           - neNoQueueRights
           - neNoQueueJob
           - neStationIsNotServer
           - neQueueHalted
           - neBinderyLocked
           - neBinderyError
 }

function RestoreQueueServerRights:Byte;

var
  NovRegs       : Word;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
  end;

begin
  Reply := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $75;
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  RestoreQueueServerRights := Lo(NovRegs);
end;

{
  function        : ReadQueueServerCurrentStatus
  service type    : queue functions
  Netware version : NW v2.1 - v4.x

  Desc : Gets the current status of an
         queue server

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neInvalidPath
           - neNoSuchQueue
           - neNoServerQueue
           - neNoQueueRights
           - neInvalidBindSecLevel
           - neNoSuchObject
           - neBinderyLocked
           - neBinderyError
 }

function ReadQueueServerCurrentStatus(QueueId : LongInt;
                                      ServerId : LongInt;
                                      ServerStation : Byte;
                                      var StatusRec : tServerStatusReply):Byte;

var
  NovRegs       : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    Queue       : LongInt; { Object id of queue (big-endian)         }
    Sever       : LongInt; { Server id                               }
    ServerSt    : Byte;    { Server station                          }
  end;

  Reply         : tServerStatusReply absolute StatusRec;

begin
  Reply.Size := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen      := SizeOf(Request) - 2;
    SubF      := $76;
    Queue     := SwapLongInt(QueueId);
    Sever     := SwapLongInt(ServerId);
    ServerSt  := ServerStation;
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  ReadQueueServerCurrentStatus := Lo(NovRegs);
end;

{
  function        : SetQueueServerCurrentStatus
  service type    : queue functions
  Netware version : NW v2.1 - v4.x

  Desc : Sets the current status of an
         queue server

  Errors : - neSuccessfull
           - neServerOutOfMem
           - neInvalidPath
           - neQueueError
           - neNoSuchQueue
           - neBinderyLocked
           - neBinderyError
 }

function SetQueueServerCurrentStatus(QueueId : LongInt;
                                     StatusRec : tServerStatusRecord):Byte;

var
  NovRegs       : Word;
  Reply         : Word;

  Request       :
  record
    FLen        : Word;                 { Length of request          }
    SubF        : Byte;                 { Subfunction                }
    Queue       : LongInt;              { Object id of queue         }
    StRecord    : tServerStatusRecord;  { server status record       }
  end;

begin
  Reply := 0;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen      := SizeOf(Request) - 2;
    SubF      := $77;
    Queue     := SwapLongInt(QueueId);
    StRecord  := StatusRec;
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  SetQueueServerCurrentStatus := Lo(NovRegs);
end;

{
  function        : GetQueueJobFileSize
  service type    : queue functions
  Netware version : NW v2.1 - v4.x

  Desc : Gets the size of an queue-job
         connected file

  Errors : - neSuccessfull
 }

function GetQueueJobFileSize(var JobSize : LongInt;
                             QueueId : LongInt;
                             JobNo : Word):Byte;

var
  NovRegs       : Word;

  Request       :
  record
    FLen        : Word;    { Length of request                       }
    SubF        : Byte;    { Subfunction                             }
    QId         : LongInt; { Queue object id (big-endian)            }
    JNo         : Word;    { Job number                              }
  end;

  Reply         :
  record
    Size        : Word;    { Reply-size                              }
    QId         : LongInt; { Object id of queue (big-endian)         }
    JNo         : Word;    { Job number (big-endian)                 }
    JSize       : LongInt; { Size of job file in bytes (big-endian)  }
  end;

begin
  Reply.Size := SizeOf(Reply) -2;
  FillChar(Request, SizeOf(Request), 0);
  with Request do
  begin
    FLen     := SizeOf(Request) - 2;
    SubF     := $78;
    QId      := SwapLongInt(QueueId);
    JNo      := Swap(JobNo);
  end;
  NovRegs := $E300;
  ReqCallIntr(Request, Reply, SizeOf(Request), SizeOf(Reply), NovRegs);
  GetQueueJobFileSize := Lo(NovRegs);
  JobSize := SwapLongInt(Reply.JSize);
end;

{------------------------ end of unit ------------------------}

end.
