#!/usr/local/bin/perl

# getusers.pl
#
> # This script collects data on all of the users in the NDS and
> # compiles them into a CSV document for reading into Access or Excel.
# Use UsrAttr.pl to get a list of attributes you can get for each user.
#
# Copyright (c) 1998 By David E. Wheeler
# Computer Systems Engineer
# Medical Center Computing
# University of Virginia Health Sciences Center
# Wheeler@Virginia.edu

$version = "1.0";

use NDSm::NDSContext;
use NDSm::Buf_T;

###############################################
# Define variables, initiate memory buffers, and start session.
$C = new NDSm::NDSContext;	# Start a new NDS context.
$InBuf = new NDSm::Buf_T;	# Buffer for query
$OutBuf = new NDSm::Buf_T;	# buffer for reply
$CHandle = $C->GetHandle();	# Get the handle of the next context we've created.
@Attributes = ("Surname", "Given Name", "Initials", "OU", "Telephone Number", "Login Time", "Message Server");	# The info we want.
$Start = "[ROOT]";	# Set the context we're getting objects from.
$ndsdataloc = "ndsdata.csv";	#Location of final document for Access.
die("You're not logged on, bailing out!!") if (!$C->IsDSAuthenticated());	# Make sure we're logged on.

###############################################
# Standard Unicode init - no longer required.
($code_page, $country_id) = $C->LocaleConv();
if (defined($code_page) && defined($country_id)){
	$C->InitUnicodeTables($code_page, $country_id);
}

###############################################
# Alloc and initiate buffer for NDS read operation.
$Result = $InBuf->AllocBuf();						# Allocate the memory needed to get data into the NDS.
&MyDie ($InBuf, "AllocBuf()") if ($Result != 0);	# Error-Checking.
$Result = $InBuf->InitBuf($CHandle, "DSV_READ");	# Initialize the memory for the operation.
&MyDie ($InBuf, "InitBuf()") if ($Result != 0);		# Error-Checking.
$Result = $OutBuf->AllocBuf();						# Allocate the memory needed to get data from the NDS.
&MyDie ($OutBuf, "AllocBuf()") if ($Result != 0);	# Error-Checking.


###############################################
# In this section, we're getting the names of all the objects in the NDS. See the StatTree subroutine below.
$Flags = $C->GetContext("DCK_FLAGS");					# Make sure we always get full name and context...
MyDie($C, "GetContext()") if  (!defined($Flags));		# Error-checking.
$Flags |= $C->GetConst("DCV_CANONICALIZE_NAMES", 0);	# ...when we call CanonicalizeName() in StatTree 
MyDie($C, "GetConst()") if  (!defined($Flags));			# Error-checking.
$Result = $C->SetContext("DCK_FLAGS", $Flags);			# Set the flags.
MyDie($C, "SetContext()") if  (!defined($Result));		# Error-checking.
&StatTree($Start, $C);									# Get all the objects in the NDS.


###############################################
# In this section, we're getting the attributes of all the objects we collected above.
$Result = $C->SetContext("DCK_NAME_CONTEXT", "[ROOT]");									# Reset the context to [ROOT].
MyDie($C, "SetContext()") if  (!defined($Result));										# Error-checking.
foreach $Object (@o) {
	foreach (@Attributes){ ($InBuf->PutAttrName($CHandle, $_)); }						# Here we put each of the attributes we want into the input buffer.
	$Iter = $InBuf->Read($CHandle, $Object, "DS_ATTRIBUTE_VALUES", 0, $Iter, $OutBuf);	# Here we read in the data returned by the NDS via the output buffer.
	MyDie($InBuf, "Read()") if  (!defined($Iter));										# Error-checking.
	$AttrCount = $OutBuf->GetAttrCount($CHandle);										# Now we're finding out how many attributes were returned by the NDS for this object.
	MyDie($OutBuf, "GetAttrCount()") if  (!defined($AttrCount));						# Error-checking.
	$outer = 0;
	while ($outer < $AttrCount) {
		($AttrName, $ValCount, $SyntaxId) = $OutBuf->GetAttrName($CHandle);				# Now we're getting the Attribute name, the number of values associated with it, and the SyntaxID, which is needed to get the value.
		MyDie($OutBuf, "GetAttrName()") if  (!defined($AttrName));						# Error-checking.
		$inner = 0;
		while ($inner < $ValCount) {
			$AttrVal = $OutBuf->GetAttrVal($CHandle, $SyntaxId);						# Now we're getting the actual attribute value. This is what we want!
			MyDie($OutBuf, "GetAttrVal()") if  (!defined($AttrVal));					# Error-checking.
			if (defined($AttrVal)) {
				$d{$AttrName} = $AttrVal;												# Here's where we output the attribute, putting it into a hash.
			}	
			$inner++;
	  }
	  $outer++;
	}
	$Object =~ s/\./,/;			# Separate the ID from the context.
	$d{'Initials'} =~ s/\.//g;	# Remove any periods from the Initials field.

	# Convert the date to human-readable format.
	if ($d{'Login Time'} != "") {
		($sec, $min, $hour, $mday, $mon, $year) = localtime($d{'Login Time'});
		$mon = $mon +1;
		if ($mon < 10) { $offmon = "0" }	#Set Month offset.
		else { undef $offmon; }
		if ($mday < 10) { $offmday = "0" }	#Set day offset.
		else { undef $offmday; }
		$year = $year + 1900;
		$date = "$offmon$mon/$offmday$mday/$year";
	}
	else { $date = ""; }

	$d{'Login Time'} = localtime($d{'Login Time'});	# Convert the login time to human-readable format.
	push (@f, "\U$Object,\E\u$d{'Surname'},\u$d{'Given Name'},\u$d{'Initials'},$d{'OU'},$d{'Telephone Number'},$date,$d{'Message Server'}\n");	# Put all the variables into the final array.
}

# Output this thing to ndsdata.csv.
print STDOUT ("\nSaving NDS Data to ndsdata.csv...\n");
@f = sort (@f); # Sort the data
unshift (@f, "NetID,Context,LName,FName,Initial,Dept,Phone,LastLogin,DefServer\n"); #Add Column headings.
open (FILE, ">$ndsdataloc") ||  die ("Cannot open file: $!.\n");	#Open file.
print FILE (@f);	#Output the data.
close (FILE);	#Close the file.


##########################################
# Subroutines.
sub StatTree {
	my $StartContainer = shift;
	my $Context = shift;
	my ($i, $iter, $ObjectName, $ObjectCount, $AttrCount, @ObjInfo, $ObjType, $key, $value);
	my $Buf = new NDSm::Buf_T;
	$Context->SetContext("DCK_NAME_CONTEXT", "$StartContainer");
	$Buf->AllocBuf(32768);
	$iter = $Buf->GetConst("NO_MORE_ITERATIONS");
	do {
		$iter = $Buf->List($CHandle, "", $iter);
		MyDie($Buf, "List()") if (!defined($iter));
		$ObjectCount = $Buf->GetObjectCount($CHandle);
		MyDie($Buf, "GetObjectCount()") if  (!defined($ObjectCount));

		#Now we can list all objects in this context
		for($i=0; $i < $ObjectCount; $i++) {
			($ObjectName, $AttrCount, @ObjInfo) = $Buf->GetObjectName($CHandle);
			MyDie($Buf, "GetObjectName()") if (!defined($ObjectName));
			$ObjectName = $Context->CanonicalizeName($ObjectName);
			if ($ObjInfo[3] =~ /User/i and $ObjectName !~ /ADMIN|AHEIM|APOT|TRAIN|CLINK/) {		# Make sure each object is a user, and ignore IDs that we don't want to collect.
				push (@o, $ObjectName);
			}
			if ($ObjInfo[0] & $Buf->GetConst("DS_CONTAINER_ENTRY")) {
				StatTree("$ObjectName", $Context);
				$Context->SetContext("DCK_NAME_CONTEXT", "$StartContainer");
			}
		}
	} while ($iter != -1);
}

sub MyDie {
   my ($Error, $Msg) = @_;
   my $Err = $Error->LastErr();
   die("$Msg, error $Err");
}

__END__



