
package NDSm::Buf_T;
use strict;
use IO::Handle;
use NDSm;


# This variable set the debug level in the C-extension.

*AUTOLOAD = \&NDSm::AUTOLOAD;
*BIN_DEBUG = \&NDSm::BIN_DEBUG;
*VERSION = \$NDSm::VERSION;

=head1 NAME

Buf_T - module in the package NDSm


=head1 SYNOPSIS

  use NDSm::Buf_T;
  use NDSm::NDSContext;

  $Context = new NDSm::NDSContext;
  $Buf = new NDSm::Buf_T;

  Then you should use the methods outlined below.
  Look in the t/ directory for examples.

=head1 DESCRIPTION

  This extension enables you to modify the data stored in Novell NDS.
  from perl 5.004 or higher.
  This module have all the methods for adding, modify and delete
  data in NDS. You must also use NDSContext.pm to get a handle.
  You should get the NWSDK with same version as this module from
  http://developer.novell.com to understand what each method do.


=head1 METHODS

=over 4

=item * $Buf = new NDSm::Buf_T;

    Creates a new Buf_T.

=cut
sub new {
	my $proto = shift;
	my $class = ref($proto) || $proto;
	my $self  = {};
	$self->{PBUF} = undef;
	$self->{LAST_ERR} = 0;
		  $self->{DEBUG} = 0;
	bless ($self, $class);
	$self;
}


sub DESTROY {
	my $self = shift;
	$self->FreeBuf();
}



=item * $iter = $Buf->AddObject($Context, $ObjectName, $iter, $more);

    Create a object called $ObjectName (You'll have to
    use methods for adding data to the buf before calling AddObject)
    $iters value is returned.
    Returns undef on failure.

=cut

sub AddObject {
	my $self = shift;
	if (@_ < 4){
	  warn("Not enough parameters to AddObject\n") if $self->{DEBUG};
     return undef;
	  $self->{LAST_ERR} = -1;
   }
	my ($Context, $ObjName, $iter, $more) = @_;
   $iter = -1 if (!defined($iter) || !$iter);
	$iter = &NDSm::NWDSAddObject($Context, $ObjName, $iter, $more, $self->{PBUF});
   if (!defined($iter)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
     warn "AddObject returning <undef>\n" if $self->{DEBUG};
   }
   $iter;
}


=item * $iter = AddPartition($context, $server, $partitionRoot, $iter, $more);

    Creates the root object of a new NDS partition but is now obsolete.
    The server  parameter identifies the server where the master
    replica of the new partition is to be stored.
    Call AddReplica() and SplitPartition() instead.
    Returns $iter which can be used in sequensiell calls.
    Returns undef on failure.

=cut
sub AddPartition {
	my $self = shift;
	my ($context, $server, $partitionRoot, $iter, $more) = @_;
   $iter = -1 if (!defined($iter) || !$iter);
	$iter =  &NDSm::NWDSAddPartition($context, $server, $partitionRoot, $iter, $more, $self->{PBUF});
   if (!defined($iter)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
     warn "AddPartition returning <undef>\n" if $self->{DEBUG};
   }
   $iter;
}


=item * $Retval = $Buf->AllocBuf(); or $Buf->AllocBuf($Size);

    Alloc memory to hold values you specify with Put* methods
    With no args the alloced space is a default defined in NWSDK.
    You can also feed it $Size (Integer value).
    Returns 0 on success.

=cut
sub AllocBuf {
	my $self = shift;
	my $Retval;
	my ($Size) = @_;
		  if (defined($self->{PBUF})){$self->FreeBuf();}
		  if (!$Size){$Size = &NDSm::constant("DEFAULT_MESSAGE_LEN", 0);}
	$Retval = &NDSm::NWDSAllocBuf($Size, $self->{PBUF});
	if ($Retval){$self->{PBUF} = undef;}
	return $Retval;
}


=item * $match = $Buf->Compare($Context, $ObjectName);

    Compare the attributes stored in $Buf with the attributes
    associated with object $ObjectName.
    The value of $match give the result of the compare.
    0 : No match.
    non-zero : Match.
    returns undef on internal failure.

=cut
sub Compare {
	my $self = shift;
	my ($Context, $Object) = @_;
	my $match = &NDSm::NWDSCompare($Context, $Object, $self->{PBUF});
	if (!defined($match)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "Compare returning <undef>\n" if $self->{DEBUG};
   }
   $match;
}


=item * $Buf->Debug($DebugLevel);

    This method set the internal debugging of a class
    to a given level. uhmmm, Only 0 and and 1 are in use
    in version 0.13.03.

=cut
sub Debug {
	my $self = shift;
   $self->{DEBUG} = shift;
}



=item * $Buf->FreeBuf();

    Frees the memory allocated for internal structs, giving you a
    chance to make the buf larger or smaller.
    Use InitBuf() to initiate already allocated memory.
    return 0 on success.

=cut
sub FreeBuf {
	my $self = shift;
	if (defined($self->{PBUF})){
	  &NDSm::NWDSFreeBuf($self->{PBUF});
	  $self->{PBUF} = undef;
	}
}




=item * $AttrCount = $Buf->GetAttrCount($Context);

    Get the number of attributes stored in $Buf. The number of attributes
    are returned in $AttrCount.
    return undef on success.

=cut
sub GetAttrCount {
	my $self = shift;
	my $Context = shift;
	my $Count = &NDSm::NWDSGetAttrCount($Context, $self->{PBUF});
	if (!defined($Count)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
     warn "GetAttrCount returning <undef>\n" if $self->{DEBUG};
	}
   $Count;
}





=item * ($AttrName, $ValCount, $SyntaxId) = $Buf->GetAttrName($Context);

    Get the name of the attribute ($AttrName) stored in buf and the number of values 
    associated with the attribute ($Valcount). The $syntaxID associated with this
    attribute is also returned.

=cut
sub GetAttrName {
	my $self = shift;
	my $Context = shift;
	my @Vals = &NDSm::NWDSGetAttrName($Context, $self->{PBUF});
	if (!defined(@Vals)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "GetAttrName returning <undef>\n" if $self->{DEBUG};
   }
   @Vals;
}




=item * $AttrVal = $Buf->GetAttrVal($Context, $SyntaxId);

    Return the 'human-readable' data stored in $Buf. This method require you
    to know what kind of data is returned: Read more about this in datatypes.txt.
    Returns the value as string or integer.
    returns undef on failure.

=cut
sub GetAttrVal {
	my $self = shift;
	my ($Context, $SyntaxId) = @_;
	my $Val = &NDSm::NWDSGetAttrVal($Context, $self->{PBUF}, $SyntaxId);
	if (!defined($Val)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "GetAttrVal returning <undef>\n" if $self->{DEBUG};
   }
	$Val;
}


=item * $BufAddr = $Buf->_GetBuf();

    Internal use only!!!, if you use this it returns the memory-address of the actual 
    buffer. Please don't use this!!.

=cut
sub _GetBuf {
	my $self = shift;
	return $self->{PBUF};   
}


=item * $Value = $Buf->GetConst($ConstName);

    Returns the numeric value of a named constant, mostly for internal use.

=cut
# Call with ($ConstName)
# returns the value of a given constantname
sub GetConst {
      my ($self, $ConstName)  = @_;
	   return &NDSm::constant($ConstName, 0);
}


=item * $ObjectCount = $Buf->GetObjectCount($Context);

    Returns the number of object stored in the buffer after a Read() in parameter
    $ObjectCount.
    Returns undef on failure.

=cut
sub GetObjectCount {
	my $self = shift;
	my ($Context) = @_;
	my $Count = &NDSm::NWDSGetObjectCount($Context, $self->{PBUF});
	if (!defined($Context)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
     warn "GetObjectCount returning <undef>\n" if $self->{DEBUG};
   }
   $Count;
}


=item * ($ObjectName, $AttrCount, %objectInfo) = $Buf->GetObjectName($Context);

    Returns the $ObjectName, number of attributes ($AttrCount), and a hash
    giving more info about the object in $Buf. The hash has the same keys as the
    names of the field in the struct Object_Info_T defined in NWSDK.

=cut
sub GetObjectName {
	my $self = shift;
	my ($Context) = @_;
	my @Vals = &NDSm::NWDSGetObjectName($Context, $self->{PBUF});
	if (!defined(@Vals)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
     warn "GetObjectName returning <undef>\n" if $self->{DEBUG};
   }
   @Vals;
}


=item * $FilHandle = $Buf->OpenStream($Context, $ObjectName, $AttrName, $flag);

    This method open a SYN_STREAM attribute (login-scripts and other),
    and associates a file handle with it.
    The $flag value determines if the file is to opened in read only or
    write only. "r" or "w".
    Returns a filehandle that can be used in normal file operations if
    successful, undef if not.
    NB! Remember to call $FilHandle->truncate after you're done writing. If
    you don't some 'old' lines may included.

=cut
sub OpenStream($Context, $ObjectName, $AttrName, $flag) {
	my $self = shift;
	my ($Context, $ObjectName, $AttrName, $flag) = @_;
	my ($nwHandle, $fh, $Retval);
	undef $fh;
# First we need to delete the existing value if we are going to write
	$Retval = 0;
	if ($flag eq "w")
	{
		my $Buf = new NDSm::Buf_T;
		if (!defined($Buf)){ return undef;}
		$Retval |= $Buf->AllocBuf();
		$Retval |= $Buf->InitBuf($Context, "DSV_MODIFY_ENTRY");
		$Retval |= $Buf->PutChange($Context, "DS_OVERWRITE_VALUE", $AttrName);
		$Retval |= $Buf->PutAttrVal($Context, "SYN_STREAM", 0);
		$Buf->ModifyObject($Context, $ObjectName, 0, 0);
		undef $Buf;
	}
	if (!($Retval))
	{
	  $nwHandle = &NDSm::NWDSOpenStream ($Context, $ObjectName, $AttrName, ($flag eq "w") ? 2 : 1);
	  if (defined($nwHandle)) {
		 $fh = new IO::Handle;
		 if (!($fh->fdopen($nwHandle, $flag))) {
			if ($self->{DEBUG}) {
			  warn("Could not open SYN_STREAM attribute\n");
			  warn("Maybe attribute never used before or broken C _open_osfhandle()\n");
			}
		 }
	  }else {
     	$self->{LAST_ERR} = &NDSm::GetLastErr();
     	warn "NWDSOpenStream returning <undef>\n" if $self->{DEBUG};
     }
	}
	$fh;
}




=item * $Buf->InitBuf($Context, $OperationName);

    Initiates the buffer for use with a certain operation, 
    Return 0 on success.
	 operations are snipped out of the Novell SDK:
    
    Name             Related Function(s)
    DSV_READ         ExtSyncRead, ListAttrsEffectiveRights, Read, ReadReferences
    DSVCOMPARE      Compare
    DSV_SEARCH       ExtSyncList, ExtSyncSearch, ListByClassAndName, ListContainers, PutFilter, Search
    DSV_ADD_ENTRY           AddObject
    DSV_MODIFY_ENTRY        ModifyObject
    DSV_READ_ATTR_DEF       ReadAttrDef
    DSV_DEFINE_CLASS        DefineClass
    DSV_READ_CLASS_DEF      ReadClassDef
    DSV_MODIFY_CLASS_DEF    ModifyClassDef
    DSV_LIST_CONTAINABLE_CLASSES	ListContainableClasses
    DSV_READ_SYNTAXES	GetSyntaxDef, PutSyntaxName, and ReadSyntaxes

=cut
sub InitBuf {
	my $self = shift;
	my ($Context, $Op) = @_;
	my $Retval = -1;
	if (defined($self->{PBUF})){
	  $Retval = &NDSm::NWDSInitBuf($Context, &NDSm::constant($Op, 0), $self->{PBUF});
	}
	return $Retval;
}



=item * $Buf->IsNumType($syntaxId);

    Returns TRUE (1) if attribute with syntaxId can be read with integer.

=cut
sub IsNumType {
      my ($self, $syntaxId) = @_;
      return &NDSm::IsNumType($syntaxId);
}



=item * $Buf->IsStringType($syntaxId);

    Returns TRUE (1) if attribute with syntaxId can be read with string.

=cut
sub IsStringType {
      my ($self, $syntaxId) = @_;
      return &NDSm::IsStringType($syntaxId);
}



=item * $Buf->IsMLType($syntaxId);

    Returns TRUE (1) if attribute with $syntaxId is a multiline string.

=cut
sub IsMLType {
      my ($self, $syntaxId) = @_;
      return &NDSm::IsMLType($syntaxId);
}


=item * $Err = $Buf->LastErr;

    Return the last error-number from $Buf. The error-number
    is often an NDS error which is described if you press
    <F1> in NWadmin (MS Windows) and search for 'Error'.

=cut
sub LastErr {
	   my $self = shift;
      return $self->{LAST_ERR};
}


=item * $Iter = $Buf->List($Context, $ObjectName);

    Query for the subordiantes of $ObjectName (typically a O or OU). $Buf
    will receive all objects. They can be retrieved by $Buf->GetObjectName().
    & $Buf->GetAttrVal.
    See also t/Browse.t
    Returns undef on failure and -1 when no more iterations are needed.

=cut
sub List {
	my $self = shift;
	my($Context, $ObjectName, $iter) = @_;
   $iter = -1 if (!defined($iter) || !$iter);
	$iter = &NDSm::NWDSList($Context, $ObjectName, $iter, $self->{PBUF});
	if (!defined($iter)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "List returning <undef>\n" if $self->{DEBUG};
   }
   $iter;
}



=item * $Iter = $Buf->ModifyObject($Context, $ObjName, $Iter, $more);

    Execute a modify object operation on the object named in $ObjName. If iterations are
    needed the handle is returned in $Iter.
    See also t/ChangeAttr.t
    return undef on failure.

=cut
sub ModifyObject {
	my $self = shift;
	if (@_ < 4){warn("Not enough parameters to ModifyObject\n");return -1;}
	my ($Context, $ObjName, $iter, $more) = @_;
   $iter = -1 if (!defined($iter) || !$iter);
	$iter = &NDSm::NWDSModifyObject($Context, $ObjName, $iter, $more, $self->{PBUF});
	if (!defined($iter)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "ModifyObject returning <undef>\n" if $self->{DEBUG};
	}
   $iter;
}


=item * $Retval = $Buf->PutAttrName($Context, $AttrName);

    Put the attribute name of the attribute to delete, modify, add ....
    See also t/ChangeAttr.t and t/*
    Returns 0 on success.

=cut
sub PutAttrName {
	my $self = shift;
	my ($Context, $AttrName) = @_;
	return &NDSm::NWDSPutAttrName($Context, $self->{PBUF}, $AttrName);
}


=item * $Retval = $Buf->PutAttrVal($Context, $SyntaxName, $Data);

    Insert the value of a attribute to delete, modify, add ....
    This method require that you understand what's outlined in datatypes.txt
    included with this package.
    Examples are included in the t/ directory. You unziped with -d didn't you?
    Returns 0 on success.

=cut
sub PutAttrVal {
	my $self = shift;
	if (@_ < 3){warn("Not enough parameters to PutAttrVal\n");return -1;}
	my ($Context, $Type, $Data) = @_;
	return &NDSm::NWDSPutAttrVal($Context, $self->{PBUF}, &NDSm::constant($Type, 0), $Data);
}                       


=item * $Buf->PutChange($Context, $OperationName, $AttributeName);

    Tells the buffer that a change is to be made on a given attribute. ($AttributeName)
    $OperationName is one of: "DS_OVERWRITE_VALUE", "DS_REMOVE_VALUE", "DS_ADD_VALUE".
    See also t/ChangeAttr.t
    Returns 0 on success

=cut
# Call with ($Context, $OperationName, $AttributeName)
sub PutChange{
	my $self = shift;
	if (@_ < 3){warn("Not enough parameters to PutChange\n"); return -1;}
	my ($Context, $Op, $AttrName) = @_;
	return &NDSm::NWDSPutChange($Context, $self->{PBUF}, &NDSm::constant($Op,0), $AttrName);
}


=item * $Iter = $Buf->Read($Context, $ObjectName, $InfoType, $AllAttrs, $Iter, $ResultBuf);

    Read data from NDS. See t/GetAttr.t
    Returns undef on failure.

=cut
sub Read {
	my $self = shift;
	die ("Read needs 6 args\n") if (@_ != 6);
	my ($Context, $ObjName, $InfoType, $AllAttrs, $Iter, $ResultBuf) = @_;
	$InfoType = $self->GetConst($InfoType);
   $Iter = -1 if (!defined($Iter) || !$Iter);
	$Iter = &NDSm::NWDSRead($Context, $ObjName, $InfoType, $AllAttrs, $self->{PBUF}, $Iter, $ResultBuf->_GetBuf());
	if (!defined($Iter)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "Read returning <undef>\n" if $self->{DEBUG};
   }
   $Iter;
}





END {;}

1;

__END__

=item * More to come..... 

=back

=head1 SEE ALSO

L<NDSContext>

L<Misc>

L<NWServer>

L<NDSm> 

http://www.ahs.hist.no/distr/NDSm/      

http://developer.novell.com/


=head1 COPYRIGHT

  Copyright (c) Steinar Kleven 1997.
  All rights reserved.


=head1 AUTHOR

  Steinar Kleven, 
  mailto:Steinar.Kleven@ahs.hist.no


=cut
