
package NDSm::NDSContext;
use strict;
use NDSm;

*AUTOLOAD = \&NDSm::AUTOLOAD;
*BIN_DEBUG = \$NDSm::BIN_DEBUG;
*VERSION = \$NDSm::VERSION;


my $InitStatus = &NDSm::NWCallsInit(0, 0);
die("Init failed, error: #$InitStatus\n") if ($InitStatus);


END {
	&NDSm::NWCallsTerm(0);
	&NDSm::NWFreeUnicodeTables();
}


=head1 NAME

NDSContext - module in the package NDSm.


=head1 SYNOPSIS

  use NDSm::NDSContext;
  use NDSm::Buf_T;

  $Context = new NDSm::NDSContext;
  $Buf = new NDSm::Buf_T;





  Then you should use the methods outlined below.
  Look in the t/ directory for examples.

=head1 DESCRIPTION

  This extension enables you to modify the data stored in Novell NDS.
  with perl 5.000 or higher.
  This module also includes methods for initalizing and
  misc methods, like WhoAmI and others.
  You should get the NWSDK with same version as this module from
  http://developer.novell.com to understand what each method do.


=head1 METHODS

=over 4

=item * $C = new NDSm::NDSContext;

	 Creates a new NDSContext.
	 Don't make to many context-objects.
	 (8 for DOS, Windows NT has only 16, OS/2 Windows has 48).
	 Maybe we should start using the new NWDSCreateContextHandle()
	 which supports unlimited handles. Well, if anyone reach the limit
	 in their script, please let me know.

=cut
sub new {
			my $proto = shift;
			my $class = ref($proto) || $proto;
      	my $self  = {};
	$self->{CONTEXT} = undef;
	$self->{LAST_ERR} = 0;
	$self->{DEBUG} = 0;
	bless ($self, $class);
	$self->_Init();
        return $self;
}


sub DESTROY {
	my $self = shift;
   &NDSm::FreeContext($self->{CONTEXT});
}



=item * $Handle = $C->GetHandle();

    Returns the handle to a context. A contaxt can have many properties
    set, but there are no support for changing these values yet.
    The value returned is just a number, so you can put it directly
    into an scalar and feed it to all methods that require the
    contexthandle.

=cut
sub GetHandle{
        my $self = shift;
	return $self->{CONTEXT};
}


=item * $C->_Init();

    This method is used internally to initialize things.
    You should normaly never use this directly.
    This method is called by new().

=cut
sub _Init {
	my $self = shift;
	if (!defined($self->{CONTEXT})){
	  $self->{CONTEXT} = &NDSm::NWDSCreateContext();
     if ($self->{CONTEXT} == &NDSm::constant("ERR_CONTEXT_CREATION", 0)){
     undef $self->{CONTEXT};
	  }else{
# Turn typeless naming on and canonicalize names off as default.
       my $Flags = $self->GetContext("DCK_FLAGS");
       $Flags |= &NDSm::constant("DCV_TYPELESS_NAMES", 0);
       $Flags &= ~&NDSm::constant("DCV_CANONICALIZE_NAMES", 0);
       my $Status = $self->SetContext("DCK_FLAGS", $Flags);
       if ($Status) {$self->{LAST_ERR} = &NDSm::LastErr();}
	  }
	}
}



=item * $Val = $C->IsInit();

    You can use IsInit() to find out if the NDSContext is created
    and initialized as it should be.

=cut
sub IsInit {
	my $self=shift;
	return defined($self->{CONTEXT});
}



=item * $AbbreviatedName = $C->AbbreviateName($InName);

    This method abbreviates the $Name. Converts a NDS name
    (including the naming attributes) to its shortest form relative
    to a specified name context.
    The abbreviated name is returned in $AbbreviatedName.
    The method return undef on failure.

=cut
sub AbbreviateName {
	my $self = shift;
	if (@_ < 1){ warn("AbbreviateName needs 1 param\n");return undef;}
   my ($InName) = @_;
	my $abbrev = &NDSm::NWDSAbbreviateName($self->{CONTEXT}, $InName);
   if (!defined($abbrev)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
     warn "AbbreviateName returning <undef>\n" if $self->{DEBUG};
	  undef;
	}else { $abbrev; }
        
}


=item * $C->AbortPartitionOperation($partitionRoot);

    Aborts a partition operation in progress.
    Returns 0 on success.

=cut
sub AbortPartitionOperation {
	my $self = shift;
	my($partitionRoot) = @_;
        return &NDSm::NWDSAbortPartitionOperation($partitionRoot);
}



=item * $C->AddReplica($server, $partitionRoot, $replicaType);

    Adds a replica of an existing NDS partition to a server.
    $server is the name of the server where the replica is to be stored.
    $partitionRoot is the name of the root object of the NDS partition to be replicated.
    $replicaType specifies the type of the new replica (1:secondary or 2:read-only).
    Returns 0 on success.

=cut
sub AddReplica {
	my $self = shift;
	my ($server, $partitionRoot, $replicaType) = @_;
	return &NDSm::NWDSAddReplica($self->{CONTEXT}, $server, $partitionRoot, $replicaType);    
}



=item * $C->AddSecurityEquiv($equalFrom, $equalTo);

    Adds to the specified object's security equivalence.
    $equalTo Points to the name to be added to the Security Equivalence 
    attribute of the object specified by equalFrom .
    This is handy when adding users to a group!.
    Returns 0 on success.

=cut
sub AddSecurityEquiv {
	my $self = shift;
	my ($equalFrom, $equalTo) = @_;
	return &NDSm::NWDSAddSecurityEquiv($self->{CONTEXT}, $equalFrom, $equalTo);
}


=item * $C->CanDSAuthenticate();

   If you can authenticate to the NDS this returns != 0. 

=cut
sub CanDSAuthenticate {
	my $self = shift;
	return &NDSm::NWDSCanDSAuthenticate($self->{CONTEXT});
}




=item * $Outname = $C->CanonicalizeName(InName)

    Converts an abbreviated name to the canonical form.
    The canonicalized name is returned in $Outname.
    For example, if the input is
    CN=Steinar Kleven
    and the name context is
    OU=Bar.O=Acme
    the canonicalized name is
    CN=Steinar kleven.OU=Bar.O=Acme

    returns undef on failure.

=cut
sub CanonicalizeName {
	my $self = shift;
	my $InName = shift;
	my $Canon = &NDSm::NWDSCanonicalizeName($self->{CONTEXT}, $InName);
        if (!defined($Canon)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
          warn "CanonicalizeName returning <undef>\n" if $self->{DEBUG};
	  undef;
	}else { $Canon; }
}



=item * $C->ChangeObjectPassword($Opt, $ObjName, $oldPass, $newpass)

    This method set the password for a given object 
    once a public/private key pair has been assigned.
    See GenerateObjectKeyPair.
    The $Opt flag is not used... hmmm, maybe we can set the password
    without giving a old password using the right code. (NWAdmin can).
    Anyway, returns 0 on success.

=cut
# Opt is not used in NSDK(13)
sub ChangeObjectPassword {
	my $self = shift;
	my ($foo, $Obj, $Old, $New) = @_;
	return &NDSm::NWDSChangeObjectPassword($self->{CONTEXT}, $foo, $Obj, $Old, $New);
}





=item * $C->ChangeReplicaType($replicaName, $ServerName, $NewType);

    Changes the replica type of a given replica on a given server.
    The available $NewType values are:
        0     RT_MASTER     Master replica
        1     RT_SECONDARY  Secondary replica
        2     RT_READONLY   Read-only replica
    Returns 0, on success.

=cut
sub ChangeReplicaType {
	my $self = shift;
	my($Name, $Server, $Type) = @_;
	$Type = &NDSm::constant($Type, 0);
	return &NDSm::NWDSChangeReplicaType($self->{CONTEXT}, $Name, $Server, $Type); 
}



=item * $NewHandle = $C->CreateContextHandle();

    This method is for internal use ONLY.

=cut
sub CreateContextHandle {
	my $self = shift;
	my $Handle = &NDSm::NWDSCreateContextHandle();
	if (!defined($Handle)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "CreateContextHandle returning <undef>\n" if $self->{DEBUG};
	  undef;
	}else { $Handle; }
}


=item * $C->Debug($DebugLevel);

    This method set the internal debugging of a class
    to a given level. uhmmm, Only 0 and and 1 are in use
    in version 0.13.03.

=cut
sub Debug {
	my $self = shift;
        $self->{DEBUG} = shift;
}



=item * $C->GenerateObjectKeyPair($ObjName, $newpass, $flag)

    This method creates or changes a public/private key pair for a specified object
    Has to be called before the object can log in.
    Returns 0 on success.

=cut
sub GenerateObjectKeyPair {
	my $self = shift;
	my ($ObjName, $newpass, $flag) = @_;
	return &NDSm::NWDSGenerateObjectKeyPair($self->{CONTEXT}, $ObjName, $newpass, $flag);
}


=item * $Value = $Buf->GetConst($ConstName);

    Returns the numeric value of a named constant, mostly for internal use.

=cut
# Call with ($ConstName)
# returns the value of a given constantname
sub GetConst {
      my ($self, $ConstName)  = @_;
	   return &NDSm::constant($ConstName, 0);
}



=item * $Value = $C->GetContext($Key);

    Return the NDS Context-variable associated with $Key.
    Possible values for $Key includes (Copied from SDK-manuals):
    1	DCK_FLAGS		Bit definitions
    2	DCK_CONFIDENCE		Definitions: 0 DCV_LOW_CONF 1 DCV_MED_CONF 2 DCV_HIGH_CONF
    3	DCK_NAME_CONTEXT	Character string array
    4	DCK_TRANSPORT_TYPE	nuint32[2]	Not currently in use
    5	DCK_REFERRAL_SCOPE	nuint32	Definitions: 0 DCV_ANY_SCOPE1 DCV_COUNTRY_SCOPE2DCV_ORGANIZATION_SCOPE3 DCV_LOCAL_SCOPE
    8	DCK_LAST_CONNECTION	Returns NWCONN_HANDLE
   11	DCK_TREE_NAME		Character string array of at most NW_MAX_TREE_NAME_LEN (includes NULL) ASCII or UNICODE characters
   This method returns undef on failure.

=cut
sub GetContext {
	my $self = shift;
	my $Key = shift;
	my $Val = &NDSm::NWDSGetContext($self->{CONTEXT}, &NDSm::constant($Key,0));
	if (!defined($Val)) {
     $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "GetContext returning <undef>\n" if $self->{DEBUG};
     undef;
	}else { $Val; }
}


=item * $NameContext = $C->GetDefNameContext()

    GetDefNameContext() returns the default name context 
    for the logged in object.

=cut
sub GetDefNameContext {
	my $self = shift;
	my $Context =  &NDSm::NWDSGetDefNameContext($self->{CONTEXT});	
	if (!defined($Context)) {
     $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "GetDefNameContext returning <undef>\n" if $self->{DEBUG};
     undef;
	}else { $Context; }
}






=item * $partitionRoot = $C->GetPartitionRoot($ObjectName);

    Returns the partition root name of the given object ($ObjectName).
    The partition root name is returned in $partitionRoot.
    Returns undef on failure.

=cut
sub GetPartitionRoot {
	my $self = shift;
	my $ObjectName = shift;
	my $PartRoot =  &NDSm::NWDSGetPartitionRoot($self->{CONTEXT}, $ObjectName);
	if (!defined($PartRoot)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "GetDefNameContext returning <undef>\n" if $self->{DEBUG};
          undef;
	}else { $PartRoot; }
}



#
#
# Add GetSyntaxID here
#
#





=item * $Err = $C->LastErr;

    Return the last error-number from $Buf. The error-number
    is often an NDS error which is described if you press
    <F1> in NWadm* and search for 'Error'.

=cut
sub LastErr {
	my $self = shift;
        return $self->{LAST_ERR};
}



=item * $C->MoveObject($objectName, $destParentDN, $destRDN);

    Moves an NDS object from one container to another and/or renames 
    the object.
    $objectName is the name of the object to move. $destParentDN is
    the container where the object is to be put. $destRDN is 
    the new name of the object.
    Example:
    CN=Steinar.OU=Devel.O=Makers
    and you want to move Steinar to Sales, for objectName  pass in
    CN=Steinar.OU=Devel.O=Makers
    for destParentDN  pass in
    OU=Sales.O=Makers
    and for destRDN  pass in
    CN=Steinar
    Returns 0 on success

=cut    
sub MoveObject {
	my $self = shift;
	my ($Obj, $destParentDN, $destRDN) = @_;
	return &NDSm::NWDSMoveObject($self->{CONTEXT}, $destParentDN, $destRDN); 
}



=item * $C->PartitionReceiveAllUpdates($partitionRoot, $serverName);

    This method changes the state of the partition so all servers 
    holding a partition replica will send entire partition information
    to the original partition.
    $partitionRoot holds the name root object name for the partition.
    $serverName holds the name of the server who has the partition
    witch should receive all updates.
    Returns 0 on success.

=cut
sub PartitionReceiveAllUpdates {
	my $self = shift;
	my ($pRoot, $sn) = @_;
	return &NDSm::NWDSPartitionReceiveAllUpdates($self->{CONTEXT}, $pRoot, $sn);
}


=item * $C->PartitionSendAllUpdates($partitionRoot, $serverName);

   Like above, but tells the partition to SEND full updates
   Resturn 0 on success.

=cut
sub PartitionSendAllUpdates {
	my $self = shift;
	my ($pRoot, $sn) = @_;
	return &NDSm::NWDSPartitionSendAllUpdates($self->{CONTEXT}, $pRoot, $sn);
}



=item * $C->ReloadDS($ServerName);

    Tells the server named in $ServerName to reload the DS.NLM.
    Returns 0 on success.

=cut
sub ReloadDS {
	my $self = shift;
	my ($SName) = @_;
	return &NDSm::NWDSReloadDS($self->{CONTEXT}, $SName);
}


=item * $typeLess = $C->RemoveAllTypes($Name):

    Returns the object name as a typeless in $typeLess.
    Example:
    "CN=Steinar.OU=Devel.O=Makers" in $Name will return 
    "Steinar.Devel.Makers" in $typeLess.
    Returns undef on failure.

=cut
sub RemoveAllTypes {
	my $self = shift;
	my $Name = shift;
   my $TypeLess = &NDSm::NWDSRemoveAllTypes($self->{CONTEXT}, $Name);
	if (!defined($TypeLess)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "RemoveAllTypes returning <undef>\n" if $self->{DEBUG};
          undef;
	}else { $TypeLess; }
}


=item * $C->RemoveAttrDef($AttrName);

    Deletes an attribute definition ($AttrName) from the NDS Schema.
    Clients cannot subtract from the standard set of attribute
    definitions defined by the NDS Base Schema. 
    (these attributes are flagged nonremovable).
    Clients can, however, add and remove non-standard definitions.
    (if not in use). 
    Returns 0 on success.

=cut
sub RemoveAttrDef {
	my $self = shift;
	my ($AttrName) = @_;
	return &NDSm::NWDSRemoveAttrDef($self->{CONTEXT}, $AttrName);
}


=item * $C->RemoveClassDef($ClassName);

    Removes a class definition from the NDS Schema.
    The same rules as for RemoveAttrDef() apply when deleteing
    classdefinitions.
    Return 0 on success.

=cut
sub RemoveClassDef {
	my $self = shift;
	my($ClassName) = @_;
	return &NDSm::NWDSRemoveClassDef($self->{CONTEXT}, $ClassName);
}


=item * $C->RemoveObject($ObjectName);

    Removes the named object from NDS. 
    The object can not have any subordinates.
    Returns 0 on success.

=cut
sub RemoveObject {
	my $self = shift;
	my ($ObjectName) = @_;
	return &NDSm::NWDSRemoveObject($self->{CONTEXT}, $ObjectName);
}


=item * $C->RemovePartition($partitionRoot);

    Removes the named partition from NDS by deleting its master replica.

    *** THIS METHOD IF NOT A GOOD IDEA TO USE IF YOU'RE
    *** NOT ABSOLUTELY SURE WHAT YOU'RE DOING.

    If this method return 0 you've toasted a partition. Did your NSD-aware
    backup system run last night ??

=cut
sub RemovePartition {
	my $self = shift;
	my ($PartitionName) = @_;
	return &NDSm::NWDSRemovePartition($self->{CONTEXT}, $PartitionName);
}


=item * $C->RemoveReplica($ServerName, $partitionRoot);

    This method removes a replica from the replica set of an
    NDS partition.
    You can't remove the master replica with this method. If that's what
    you want you should use RemovePartition() instead. Read the man about
    RemovePartition() before using it !!!.
    Return 0 on success.

=cut
sub RemoveReplica {
	my $self = shift;
	my ($ServerName, $PartitionRoot) = @_;
	return &NDSm::NWDSRemoveReplica($self->{CONTEXT}, $ServerName, $PartitionRoot);
}


=item * $C->RemSecurityEquiv($equalFrom, $equalTo);

    Removes a security equivalence from the specified object.
    $equalFrom is the name of the object whose Security Equivalence 
    attribute is to be modified.
    $equalTo is the object name to be removed from the list.
    Also return 0 on success.

=cut
sub RemSecurityEquiv {
        my $self = shift;
	my ($equalFrom, $equalTo) = @_;
	return &NDSm::NWDSRemSecurityEquiv($self->{CONTEXT}, $equalFrom, $equalTo);
}



=item * $C->RepairTimeStamps($partitionRoot);

   Sets the time stamps for all of a partition's objects and object 
   attributes to the current time on the NetWare server where the master 
   replica is located.
   $partitionRoot is the name of the partition's root object.
   Return 0 on success.

=cut
sub RepairTimeStamps {
	my $self = shift;
	my ($partitionRoot) = @_;
	return &NDSm::NWDSRepairTimeStamps($self->{CONTEXT}, $partitionRoot);
}



=item * $longName = $C->ReplaceAttrNameAbbrev($inName);

    Replaces the abbreviated attribute name with its unabbreviated name.
    Example:
    If $inName is "CN", the value of $longName will be "Common Name".
    Returns undef on failure.

=cut 
sub ReplaceAttrNameAbbrev {
	my $self = shift;
	my $inStr = shift;
	my $Full = &NDSm::NWDSReplaceAttrNameAbbrev($self->{CONTEXT}, $inStr);
	if (!defined($Full)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "ReplaceAttrNameAbbrev returning <undef>\n" if $self->{DEBUG};
     undef;
	}else { $Full; }
}


	
=item * $C->SetContext($key, $Value)

    Sets the value of an internal variable in this 
    NDS-context. See NWSDK for options.

=cut
sub  SetContext {
	my $self = shift;
        my ($key, $value) = @_;
	return &NDSm::NWDSSetContext($self->{CONTEXT}, NDSm::constant($key, 0), $value);
}




=item * $C->SetDefNameContext($NameContext);

    Sets the default name context to $NameContext;
    Return 0 on success.

=cut
sub SetDefNameContext {
	my $self = @_;
	my ($NameContext) = @_;
	my $Len = strlen($NameContext);
        return &NDSm::NWDSSetDefNameContext($self->{CONTEXT}, $Len, $NameContext);
}


=item * ($code_page, $country_id) = $C->LocaleConv();

  This method returns the values in a list. The values are 
  according to the current locale setting.
  Returns code_page, country_id as numbers.


=cut
sub LocaleConv {
   my $self = shift;
   return &NDSm::NWLlocaleconv();
}


=item * $C->InitUnicodeTables($CountryId, $CodePage); or ();

    This method sets the current CountryId and Codepage as defined by IBM.
    If called with () it uses the current setting for your machine.
    Return 0 on success.

=cut
sub InitUnicodeTables {
   my $self = shift;
   my ($CountryId, $CodePage) = @_;
   if (!$CountryId || !$CodePage){
     ($CodePage, $CountryId) = $self->LocaleConv();
   }
   return &NDSm::NWInitUnicodeTables($CountryId, $CodePage);
}



=item * $C->IsDSAuthenticated();

    Returns nonzero value if you are authenticated to NDS.

=cut
sub IsDSAuthenticated {
	my $self = shift;
	return &NDSm::NWIsDSAuthenticated();
}



=item * $C->Login($options, $UserName, $passwd, $validperiod);

    Performs all authentication operations needed to establish a client's 
    connection to the network and to the network's authentication service.
    As of NWSDK(13) validperiod and options are not used. (Use 0)
    Returns 0 on success.

=cut
sub Login {
	my $self = shift;
	my ($Opt, $User, $Passwd, $Period) = @_;
        return &NDSm::NWDSLogin($self->{CONTEXT}, 0, $User, $Passwd, 0);
}



=item * $C->Logout();

    Guess!

=cut
sub Logout {
	my $self = shift;
	return &NDSm::NWDSLogout($self->{CONTEXT});
}


=item * $C->VerifyObjectPassword($opt, $objectName, $Passwd);

    Verifies the password of an object.
    $Passwd can be any length and all characters are significant. 
    Upper- and lowercase letters are distinct.
    Can we use this to hack passwd's ?? Weee, I don't know if this
    get caught by the intruder lockout mechanism. If it passes through,
    Novell should plug the hole anyway, so I leave this method for anyone
    to know.
    Returns 0 on success.

=cut 
sub VerifyObjectPassword {
	my $self = shift;
	my ($opt, $objectName, $Passwd) = @_;
	return NWDSVerifyObjectPassword($self->{CONTEXT}, $opt, $objectName, $Passwd);
}


=item * $MyName = $C->WhoAmI();

    This method is Novells answer to unix 'whoami'.
    Your loginname will be returned in $MyName.
    Return undef on failure. 

=cut
sub WhoAmI {
        my $self = shift;
        my $MyCN = &NDSm::NWDSWhoAmI($self->{CONTEXT});
	if (!defined($MyCN)) {
	  $self->{LAST_ERR} = &NDSm::GetLastErr();
	  warn "WhoAmI returning <undef>\n" if $self->{DEBUG};
          undef;
	}else { $MyCN; }
}





1;

__END__

=item * More to come..... 

=back

=head1 SEE ALSO


L<Buf_T>      

L<NWServer>

L<Misc>

L<NDSm> 

http://www.ahs.hist.no/distr/NDSm/      

http://developer.novell.com 


=head1 COPYRIGHT

  Copyright (c) Steinar Kleven 1997.
  All rights reserved.


=head1 AUTHOR

  Steinar Kleven, 
  mailto:Steinar.Kleven@ahs.hist.no


=cut
