package NDSm::NWServer;

use strict;
use vars qw($BIN_DEBUG $VERSION @ISA @EXPORT @EXPORT_OK);

require Exporter;
require DynaLoader;

@ISA = qw(Exporter DynaLoader);
# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.
@EXPORT = qw(

);

$VERSION = '0.13.05';
bootstrap NDSm::NWServer $VERSION;

# Preloaded methods go here.

my $InitStatus = &NWCallsInit(0, 0);
die("Init failed, error: #$InitStatus\n") if ($InitStatus);


END {
	&NWCallsTerm(0);
}


sub new {
	my $proto = shift;
	my $class = ref($proto) || $proto;
	my $self  = {};
	$self->{CONN} = undef;
	$self->{SERVER_NAME} = shift;
	$self->{IS_AUTH} = undef;
   $self->{UNLIC} = 0;   # Use in later versions to unlicense conn if we licensed it
   $self->{CLOSE} = 0;  # Use in later versions to detach is we attached.
	$self->{DEBUG} = 0;
	die("Usage: new NWServer(ServerName);\n") if (!defined($self->{SERVER_NAME}));
	bless ($self, $class);
	$self->_Init();
	$self;
}


sub DESTROY {
	my $self = shift;
	&NWCLXTerm();
}


sub GetHandle{
	my $self = shift;
	return $self->{CONN};
}


sub Debug {
	my $self = shift;
	$self->{DEBUG} = shift;
}


#Tries several ways of getting a connection to the server
sub _Init {
	my $self = shift;
   &NWCLXInit(0, 0);
   $self->{CLOSE} = 1; # Defaults to close the connection after use
	if(!defined($self->{CONN})){
	# First see if we already have an connection to this server
   # GetConnectionHandle is obsolete, but I use it for now.
  		$self->{CONN} = &NWGetConnectionHandle($self->{SERVER_NAME}, 0, 0);
	   if (defined($self->{CONN}))
      {
        $self->{CLOSE} = 0; # This is opened by another app.
		}
	}
   if (!defined($self->{CONN})){
	# We wasn't connected, and try to open a new connection.
   	$self->{CONN} = &NWAttachToFileServer($self->{SERVER_NAME}, 0);
	}
   if (!defined($self->{CONN})){
	# We wasn't connected, and try bindery server name format.
   	$self->{CONN} = &NWCCOpenConnByName(0, $self->{SERVER_NAME},
      	&constant("NWCC_NAME_FORMAT_BIND", 0),
         &constant("NWCC_OPEN_UNLICENSED", 0),
         &constant("NWCC_TRAN_TYPE_WILD", 0));
	}
   if (!defined($self->{CONN})){
  	# We wasn't connected, and try nds server name format.
   	$self->{CONN} = &NWCCOpenConnByName(0, $self->{SERVER_NAME},
      	&constant("NWCC_NAME_FORMAT_NDS", 0),
         &constant("NWCC_OPEN_UNLICENSED", 0),
         &constant("NWCC_TRAN_TYPE_WILD", 0));
	}
   if (!defined($self->{CONN})){
  	# We wasn't connected, and try nds server name format.
   	$self->{CONN} = &NWCCOpenConnByName(0, $self->{SERVER_NAME},
      	&constant("NWCC_NAME_FORMAT_NDS_TREE", 0),
         &constant("NWCC_OPEN_UNLICENSED", 0),
         &constant("NWCC_TRAN_TYPE_WILD", 0));
	}
	if (!defined($self->{CONN})) {
	  $self->{LAST_ERR} = &GetLastErr();
	  warn "_Init could not make a connection\n" if $^W;
	}
}


sub IsInit {
	my $self=shift;
	return defined($self->{CONN});
}



sub AddTrustee {
   my $self = shift;
   my ($not_used, $Path, $ObjectID, $MaskStr) = @_;
   my $Mask = &constant("TR_NONE", 0);
   if (defined($MaskStr)) {
      $Mask |= &constant("TR_SUPERVISOR", 0) if ($MaskStr =~ /S/i);
      $Mask |= &constant("TR_READ", 0) if ($MaskStr =~ /R/i);
      $Mask |= &constant("TR_WRITE", 0) if ($MaskStr =~ /W/i);
      $Mask |= &constant("TR_ERASE", 0) if ($MaskStr =~ /E/i);
      $Mask |= &constant("TR_CREATE", 0) if ($MaskStr =~ /C/i);
      $Mask |= &constant("TR_MODIFY", 0) if ($MaskStr =~ /M/i);
      $Mask |= &constant("TR_FILE_SCAN", 0) if ($MaskStr =~ /F/i);
      $Mask |= &constant("TR_ACCESS_CTRL", 0) if ($MaskStr =~ /A/i);
   }
	print "Mask: $MaskStr = $Mask\n" if $self->{DEBUG};
   return &NWAddTrustee($self->{CONN}, 0, $Path, $ObjectID, $Mask);
}


sub AuthenticateConn {
	my $self = shift;
	my $Context = shift;
	if (!defined($self->{IS_AUTH})) {
	  $self->{IS_AUTH} = &NWDSAuthenticateConn($Context, $self->{CONN}) ? undef : 1;
   }
   (defined($self->{IS_AUTH})) ? 0 : 1;
}




sub CheckConsolePrivileges {
	  my $self = shift;
	  return &NWCheckConsolePrivileges($self->{CONN});
}


sub CreateDirectory {
      my $self = shift;
      my ($dirHandle, $dirPath, $MaskStr) = @_;
		my $Mask = &constant("TA_NONE", 0);
   	if (defined($MaskStr)) {
			$Mask |= &constant("TA_READ", 0) if ($MaskStr =~ /READ/i);
   	   $Mask |= &constant("TA_WRITE", 0) if ($MaskStr =~ /WRITE/i);
      	$Mask |= &constant("TA_OPEN", 0) if ($MaskStr =~ /OPEN/i);
      	$Mask |= &constant("TA_CREATE", 0) if ($MaskStr =~ /CREATE/i);
	      $Mask |= &constant("TA_DELETE", 0) if ($MaskStr =~ /DELETE/i);
   	   $Mask |= &constant("TA_OWNERSHIP", 0) if ($MaskStr =~ /OWNER/i);
      	$Mask |= &constant("TA_SEARCH", 0) if ($MaskStr =~ /SEARCH/i);
      	$Mask |= &constant("TA_MODIFY", 0) if ($MaskStr =~ /MODIFY/i);
			$Mask |= &constant("TA_ALL", 0) if ($MaskStr =~ /ALL/i);
		}
      return &NWCreateDirectory($self->{CONN}, $dirHandle, $dirPath, $Mask);
}


sub DeleteDirectory {
      my $self = shift;
      my ($dirHandle, $dirPath) = @_;
      return &NWDeleteDirectory($self->{CONN}, $dirHandle, $dirPath);
}


sub DeleteTrustee {
      my $self = shift;
      my ($dirHandle, $dirPath, $ObjectID) = @_;
      return &NWDeleteTrustee($self->{CONN}, $dirHandle, $dirPath, $ObjectID);
}



sub DisableFileServerLogin {
      my $self = shift;
	   return &NWDisableFileServerLogin($self->{CONN});
}


sub DownFileServer {
	  my $self = shift;
	  my ($forceFlag) = @_;
	  return &NWDownFileServer($self->{CONN}, $forceFlag);
}


sub EnableFileServerLogin {
		my $self = shift;
		return &NWEnableFileServerLogin($self->{CONN});
}



sub ExecuteNCFFile {
		my $self = shift;
		my $NCFFile = shift;
		return &NWSMExecuteNCFFile($self->{CONN}, $NCFFile);
}



sub GetExtendedVolumeInfo {
		my $self = shift;
      my $volNum = shift;
		my @volinfo = &NWGetExtendedVolumeInfo($self->{CONN}, $volNum);
      if (!defined(@volinfo)) {
	     $self->{LAST_ERR} = &GetLastErr();
	     warn "GetExtendedVolumeInfo returning <undef>\n" if $self->{DEBUG};
      }
      @volinfo;
}


sub GetObjDiskRestrictions {
		my $self = shift;
      my ($volNum, $objID) = @_;
		my @DiskRestriction = &NWGetObjDiskRestrictions($self->{CONN}, $volNum, $objID);
      if (!defined(@DiskRestriction)) {
	     $self->{LAST_ERR} = &GetLastErr();
	     warn "GetObjDiskRestrictions returning <undef>\n" if $self->{DEBUG};
      }
      @DiskRestriction;
}



sub GetVolumeName {
		my $self = shift;
		my $VolumeNum = shift;
		my $VolName &NWGetVolumeName($self->{CONN}, $VolumeNum);
		if (!defined($VolName)) {
	     $self->{LAST_ERR} = &GetLastErr();
	     warn "GetVolumeName returning <undef>\n" if $self->{DEBUG};
      }
      $VolName;
}


sub GetVolumeNumber {
		my $self = shift;
      my $VolumeName = shift;
      my $VolNum = NWGetVolumeNumber($self->{CONN}, $VolumeName);
      if (!defined($VolNum)) {
	     $self->{LAST_ERR} = &GetLastErr();
	     warn "GetVolumeNumber returning <undef>\n" if $self->{DEBUG};
      }
      $VolNum;
}


sub LicenseConn {
		my $self = shift;
		return &NWCCLicenseConn($self->{CONN});
}


sub LoadNLM {
		my $self = shift;
		my $LoadCmd = shift;
		return &NWSMLoadNLM($self->{CONN}, $LoadCmd);
}


sub MapNameToID {
      my $self = shift;
      my ($Context, $ObjectName) = @_;
      my $IdNum = &NWDSMapNameToID($Context, $self->{CONN}, $ObjectName);
      if (!defined($IdNum)) {
	     $self->{LAST_ERR} = &GetLastErr();
	     warn "MapNameToID returning <undef>\n" if $self->{DEBUG};
      }
      $IdNum;
}



sub SetObjectVolSpaceLimit {
      my $self = shift;
      my ($volNum, $ObjectID, $Max) = @_;
      my @volinfo = &NWGetExtendedVolumeInfo($self->{CONN}, $volNum);
      if (!defined(@volinfo)){ return &GetLastErr(); }
      my $Units = ($Max*1000)/4;
      return &NWSetObjectVolSpaceLimit($self->{CONN}, $volNum, $ObjectID, $Units);
}



sub UnloadNLM {
		my $self = shift;
		my $Nlm = shift;
		return &NWSMUnloadNLM($self->{CONN}, $Nlm);
}


1;
=cut
__END__


# Autoload methods go after =cut, and are processed by the autosplit program.


=head1 NAME

NWServer - module in the package NDSm.


=head1 SYNOPSIS

  use NDSm::NWServer;

  $Server = new NDSm::NWServer($ServerName);

  Then you should use the methods outlined below.
  One of the features includes downing a server.
  This is NOT an Netware compatible SERVER, but an interface
  to an existing Netware server. Do I have to say that???
  If I could write a Netware server in less than 100 Kb (Loadable's size)
  I would be rich.
  Look in the t/ directory for examples.

=head1 DESCRIPTION

  This extension enables you to work with Novell NetWare servers
  from perl 5.004 or higher.
  You should get the NWSDK with same version as this module from
  http://developer.novell.com to understand what each method do.


=head1 METHODS

=over 4

=item * $Server = new NDSm::NWServer($ServerName);

    Creates a new Server object.
    If you aren't connected, then this object tries to make a new connection
    to the file server. You shoul call the method IsInit() to find out if
    the connection was made successfully.
    If no previous connection to the server was made this class set up a
    UNLICENSED connection because a lot of the methods don't require you to be licensed.
    If you need to be licensed you should call the member method LicenseConn() which
    will enable you to read/write files and print documents.
    (You can create directories (4.11) without beeing licensed.)


=item * $Handle = $Server->GetHandle();

    Returns the handle to a server connection.
    The value returned is just a number, so you can put it directly
    into an scalar and feed it to all methods that require the
    connection-handle.



=item * $Server->Debug($DebugLevel);

    This method set the internal debugging of a class
    to a given level. uhmmm, Only 0 and and 1 are in use
    in version 0.13.03.



=item * $Server->_Init();

    This method is used internally to initialize things.
    You should normaly never use this directly.
    This method is called by new().



=item * $Val = $Server->IsInit();

    You can use IsInit() to find out if the connection to the server is created
    and initialized as it should be.


=item * $Val = $Server->AddTrustee($not_used, $Path, $ObjectID, @RightsMask)

    Add a trustee to a file or directory.
    $Path is full path with volumename: SYS:SYSTEM/LOG.
    $ObjectId is the Id returned from $Server->MapNameToID().
    @RightsMask is a array of string constants values which is ORed together.

    example: ("TR_ERASE", "TR_READ")
    Possible values for rightsMask is:
    TR_NONE, TR_READ, TR_WRITE, TR_OPEN, TR_DIRECTORY, TR_CREATE, TR_DELETE
    TR_ERASE, TR_OWNERSHIP, TR_ACCESS_CTRL, TR_FILE_SCAN, TR_SEARCH
    TR_FILE_ACCESS, TR_MODIFY, TR_ALL, TR_SUPERVISOR, TR_NORMAL
    Returns 0 on success.


=item * $Val = $Server->AuthenticateConn($Context)

    This method authenticates the server connection.
    Maybe we should authenticate as part of initializing this
    class?. For now we have to call it manualy.
    If the connection already is authenticated we still return success.
    returns 0 on success.



=item * $Retval = $Server->CheckConsolePrivileges();

    CheckConsolePrivileges determines if the logged-in user is a console operator.
    Some return values include:
        0	SUCCESSFUL
    34817	INVALID_CONNECTION
    35270	NO_CONSOLE_PRIVILEGES



=item * $Retval = $Server->CreateDirectory($dirHandle, $dirPath, $accessMask);

    Create a directory on a Netware server. set $dirHandle to 0.
    $dirPath is the full path to a existing directory: eg. SYS:ETC\
    $accessMask is a string containing one or more of the following strings:
    READ, WRITE, OPEN, CREATE, DELETE, OWNER, SEARCH, MODIFY, ALL.
    Returns 0 on success;



=item * $Retval = $Server->DeleteDirectory($dirHandle, $dirPath);

    Delete a directory on a Netware server. $dirHandle should be 0.
    $dirPath is the full name w/volume.
    Returns 0 on success.



=item * $Retval = $Server->DeleteTrustee($dirHandle, $dirPath, $ObjectID);

    Delete a object from the trustee list for a file/directory.
    $dirHandle should be 0.
    $dirPath is the full name w/volume.
    $ObjectID is a server supplied number. Use $Server->MapNameToID()
    Returns 0 on success.


=item * $Retval = $Server->DisableFileServerLogin();

    Tells the server to stop accepting logins, workstations can
    connect though.
    You must have console operator rights. to use this method.
    See $Server->CheckConsolePrivileges().
    returns 0 on success.



=item * $Retval = $Server->DownFileServer($forceFlag)

    This very handy methods downs the server which name was used
    to create this class.
    If $forceFlag is 0 the server will shut down even if files are open.
    You must have console operator rights to use this method.
    You should use $Server->CheckConsolePrivileges() check this.
    Returns 0 on success.



=item * $Retval = $Server->EnableFileServerLogin();

    Instructs the server to begin accepting new login
    requests from clients.
    You must have console operator rights.
    returns 0 on succcess.


=item * $Retval = $Server->ExecuteNCFFile($FileName);

    Tell the server to execute the .NCF (batch) file given in
    $FileName. The file must reside on one of the fileservers volumes.
    returns 0 on success.



=item * @volInfo = $Server->GetExtendedVolumeInfo($volNum);

    This metod get all info about a given volumenumber on this server.
    $volNum can be retieved by calling $Server->GetVolumeNumber(<VOLNAME>).
    Returns undef on failure.
    Returns a table with the following values (All vaulues are numbers):
    index:	Value:
    0 			volType
    1			statusFlag
    2			sectorSize
    3			sectorsPerCluster
    4			volSizeInClusters
    5			freeClusters
    6			subAllocFreeableClusters
    7			freeableLimboSectors
    8			nonfreeableLimboSectors
    9			availSubAllocSectors
    10      nonuseableSubAllocSectors
    11			subAllocClusters
    12			numDataStreams
    13			numLimboDataStreams
    14			oldestDelFileAgeInTicks
    15			numCompressedDataStreams
    16			numCompressedLimboDataStreams
    17			numNoncompressibleDataStreams
    18			precompressedSectors
    19         compressedSectors
    20			numMigratedDataStreams
    21			migratedSectors
    22			clustersUsedByFAT
    23			clustersUsedByDirs
    24			clustersUsedByExtDirs
    25			totalDirEntries
    26			unusedDirEntries
    27			totalExtDirExtants
    28			unusedExtDirExtants
    29			extAttrsDefined
    30			extAttrExtantsUsed
    31			DirectoryServicesObjectID
    32			volLastModifiedDateAndTime



=item * @DiskRestriction = $Server->GetObjDiskRestrictions($volNum, $objID);

    Get the diskrestriction for a given object on a volume.
    The resturned array includes two fields:
    0 : Restriction set on volume for this object.
    1 : Currently used by this object.
    Returns undef on failure.



=item * $VolName = $Server->GetVolumeName($VolumeNumber);

    Retieve the volumename for a given volumenumber. This method
    return undef if the given volumenumber don't exists or isn't mounted or
    if the call failed.


=item * $VolNumber = $Server->GetVolumeNumber($VolumeName);

    Retieve the volumeunmber for a given volumename. This method
    return undef if an error accured. If a volume with given name
    don't exists on this server the $server->LastErr() will
    give 0x8998	as error-number.


=item * $Retval = $Server->LicenseConn();

    License the current server connection.
    (Removes the * in front of connection in monitor.)
    I manage to create directories without using this though.
    Returns 0 on success.


=item * $Retval = $Server->LoadNLM($Loadcmd);

    Load a NLM on this server.
    The $LoadCmd has the format:
    {VOLUME NAME:}{PATH\...}NLMname{.ext}{parameters}
    Returns 0 on success.


=item * $Id = $Server->MapNameToID($Context, $ObjectName);

    Map a NDS object name to a objectId which can be used in server-centered
    methods, like adding trustees to a volume directory.
    returns undef on failure.


=item * $Retval = SetObjectVolSpaceLimit($volNum, $ObjectID, $Max);

    Set the maximum space a object can use on a Netware server volume.
    $VolNum is a number you can get from $Server->NWGetVolumeNumber().
    $ObjectID is a value returned from $Server->MapNameToID().
    $Max is the maximum space this object can use on given volume, in MB.
    If the restriction is greater than 160 000 on a 3.1 server or
    320 000 on a 4.x server, the object has no restrictions.
    Returns 0 on success.


=item * $Retval = $Server->UnloadNLM($NLMName);

    Unloads the NLM given in $NLMName.
    The $NLMName has the format: NLMname{.ext}
    ext defaults to NLM.
    Returns 0 on success.



=item * More to come.....

=back

=head1 SEE ALSO

L<NDSContext>

L<Misc>

L<Buf_T>

L<NDSm>

http://www.ahs.hist.no/distr/NDSm/

http://developer.novell.com


=head1 COPYRIGHT

  Copyright (c) Steinar Kleven 1997.
  All rights reserved.

=head1 AUTHOR

  Steinar Kleven,
  mailto:Steinar.Kleven@ahs.hist.no


=cut
