unit nwout;

interface

uses
  SysUtils, 
  WinTypes, 
  WinProcs, 
  Messages, 
  Classes, 
  Graphics, 
  Controls,
  ExtCtrls,
  Outline,
  menus,
  Forms, 
  Dialogs, 
  grids,
  StdCtrls,
  nwBase,
  nwTools,
  nwLib,
  nwServer,
  nwNDS,
  nwPrint ;


{$IFDEF Ver110}  
  {$ObjExportAll On}
{$ENDIF}

type
  TNWOutline = class(TCustomOutline)
  private
    FCustomImages,
    FSorted        : boolean ;
    FIndentWidth   : byte    ;
    FBmp           : TBitmap ;
    FNWBase        : TNWBase ;
    FObjType       : TNWObjListType ;
    FTreeExpand,
    FOthersExpand,
    FMembersExpand,
    FDirectoryExpand,
    FFileExpand,
    FUserExpand,
    FGroupExpand,
    FPrintServerExpand,
    FPrinterExpand,
    FPrintQueueExpand,
    FPrintQueueJobExpand,
    FServerExpand,
    FVolumeExpand,
    FOrgExpand,
    FOrgUnitExpand : TNotifyEvent ;
    procedure _populate ;
    procedure _setObjType(value : TNWObjListType) ;
    procedure _setNWBase(value : TNWBase) ;
    procedure _setCustomImages(value : boolean) ;
    function  _getText : string ;
    procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
    procedure NWMBaseServerChange(var Message : TMessage) ; message NWM_BASESERVERCHANGE ;
    function _getFullPath(selectedItem : integer) : string ;
  protected
    procedure Notification(AComponent: TComponent;
                           Operation: TOperation); override;
    procedure loaded ; override ;                       
    procedure DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState); override ;
    procedure selectOutlineItem ; virtual ;
  public      
    constructor create(AOwner : TComponent) ; override ;
    destructor  destroy ; override ;
  published     
    property Lines;
    property ItemHeight;
    property OnDrawItem;
    property Align;
    property Enabled;
    property Font;
    property Color;
    property ParentColor;
    property ParentCtl3D;
    property Ctl3D;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property DragMode;
    property DragCursor;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    {$IFDEF Win32}
    property OnStartDrag;
    {$ENDIF}
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnDblClick;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property BorderStyle;
    property ItemSeparator;
    property ParentFont;
    property ParentShowHint;
    property ShowHint;
    property PopupMenu;
    property ScrollBars;
    property OnExpandPrinter       : TNotifyEvent read FPrinterExpand       write FPrinterExpand  ;
    property OnExpandPrintQueue    : TNotifyEvent read FPrintQueueExpand    write FPrintQueueExpand  ;
    property OnExpandPrintQueueJob : TNotifyEvent read FPrintQueueJobExpand write FPrintQueueJobExpand ;
    property OnExpandPrintServer   : TNotifyEvent read FPrintServerExpand   write FPrintServerExpand ;
    property OnExpandDirectory     : TNotifyEvent read FDirectoryExpand     write FDirectoryExpand  ;
    property OnExpandFile          : TNotifyEvent read FFileExpand          write FFileExpand ;
    property OnExpandUser          : TNotifyEvent read FUserExpand          write FUserExpand ;
    property OnExpandGroup         : TNotifyEvent read FGroupExpand         write FGroupExpand ;
    property OnExpandOrg           : TNotifyEvent read FOrgExpand           write FOrgExpand   ;
    property OnExpandOrgUnit       : TNotifyEvent read FOrgUnitExpand       write FOrgUnitExpand ;
    property OnExpandMembers       : TNotifyEvent read FMembersExpand       write FMembersExpand ;
    property OnExpandOthers        : TNotifyEvent read FOthersExpand        write FOthersExpand ;
    property OnExpandTree          : TNotifyEvent read FTreeExpand          write FTreeExpand   ;
    property OnExpandVolume        : TNotifyEvent read FVolumeExpand        write FVolumeExpand ;
    property ShowObjType      : TNWObjListType read FObjType       write _setObjType default nwlNone ;
    property NWBase           : TNWBase        read FNWBase        write _setNWBase ;
    property Sorted           : boolean        read FSorted        write FSorted default true ;
    property IndentWidth      : byte           read FIndentWidth   write FIndentWidth default 10 ;
    property text             : string         read _getText  ;
    property UseCustomImages  : boolean        read FCustomImages  write _setCustomImages default true ;
  end;

{$IFDEF Win32}
  {$R nwout.res}  {outline glyphs & messages}
{$ELSE}
  {$r nwout16.res}
{$ENDIF}
  
implementation

{************************* Component Implementation ****************************}  
constructor TNWOutline.create(AOwner : TComponent) ;
  begin
    inherited create(AOwner) ;
    sorted        := true ;
    style         := otOwnerDraw ;
    FCustomImages := true ;
    FIndentWidth  := 10 ;
    FBmp          := TBitMap.create ;
  end;

destructor TNWOutline.destroy ;
  begin
    FBmp.free ;
    inherited destroy ;
  end;  

procedure TNWOutline.NWMBaseServerChange(var Message : TMessage) ; 
  begin
    _populate ;
  end;

procedure TNWOutline.Notification(AComponent: TComponent;
                                  Operation: TOperation);
  begin
    inherited Notification(AComponent, Operation);
    if (Operation = opRemove) then
      begin
        if (NWBase <> nil) and (AComponent = NWBase) then
          NWBase := nil ;
      end;
  end;

procedure TNWOutline.loaded ;
  begin
    inherited loaded ;
    _populate ;  
  end;  

procedure TNWOutline.DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
  var
    tWidth,
    offset : integer ;
    node   : TOutlineNode ;
  begin
    if FCUstomImages and (lines.count > 0) and (showObjType in [nwlNDSObjList,nwlNDSTree]) then
      begin
        {$IFDEF Win32}
        node := getVisibleNode(aRow) ;
        {$ELSE}
        node := items[aRow+1] ; {this is not correct!}
        {$ENDIF}
        if (node.data <> nil) then
          case TNWDSObjType(node.data) of   
            dsoTree                : FBmp.handle := loadBitmap(hInstance,'NW_TREE') ;
            dsoUser                : FBmp.handle := loadBitmap(hInstance,'NW_USER') ;
            dsoGroup               : FBmp.handle := loadBitmap(hInstance,'NW_GROUP') ;
            dsoOrganization        : FBmp.handle := loadBitmap(hInstance,'NW_ORG') ;
            dsoOrganizationalUnit  : FBmp.handle := loadBitmap(hInstance,'NW_ORGUNIT') ;
            dsoPrinter             : FBmp.handle := loadBitmap(hInstance,'NW_PRINTER') ;
            dsoPrintServer         : FBmp.handle := loadBitmap(hInstance,'NW_PRINTSERVER') ;
            dsoQueue               : FBmp.handle := loadBitmap(hInstance,'NW_QUEUE') ;
            dsoQueueJob            : FBmp.handle := loadBitmap(hInstance,'NW_QUEUEJOB') ;
            dsoVolume              : FBmp.handle := loadBitmap(hInstance,'NW_VOLUME') ;
            dsoFile                : FBmp.handle := loadBitmap(hInstance,'NW_FILE') ;
            dsoDirectory           : FBmp.handle := loadBitmap(hInstance,'NW_DIRECTORY') ;
            dsoServer,
               dsoNCPServer        : FBmp.handle := loadBitmap(hInstance,'NW_SERVER') ;
          else
            FBmp.handle := loadBitmap(hInstance,'NW_OTHER') ;
          end;
        if (FBmp.empty) then
          begin
             inherited DrawCell(ACol, ARow, ARect, AState);
            exit ;
          end;
        offset := ((node.level-1) * FIndentWidth) ;
        with canvas do begin
          twidth      := textWidth(node.text) ;
          font        := self.font ;
          copyMode    := cmSrcCopy ;
          brush.color := self.color ;
          fillRect(aRect) ;
          if (FBmp <> nil) then
            begin
              brushCopy(rect(arect.left+offset,
                             arect.top,
                             arect.left+FBmp.width+offset,
                             arect.top+FBmp.height),
                        FBmp,
                        rect(0, 0, FBmp.width, FBmp.height), 
                        FBmp.canvas.pixels[0,0]) ;
                        
              offset := offset + (FBmp.width + 6);
            end;
          {draw text}
          if (row = aRow) then
            begin
              brush.color := clHighlight  ;
              font.color  := clHighlightText ;
            end  
          else
            begin
              brush.color := self.color ;
              font.color  := self.font.color ;
            end;
          textOut(arect.Left + offset, arect.Top, node.text) ;
          {draw focus}
          if (aRow = row) then
            drawFocusRect(rect((aRect.left -1) + offset, 
                               aRect.top, 
                               aRect.left + offset + twidth + 1 , 
                               aRect.bottom)) ;
        end;
        {$IFDEF Win32}  {do onDrawItem Event}
        if (@OnDrawItem <> nil) then
          OnDrawItem(Self, ARow, ARect, [odFocused, odSelected]);
        {$ENDIF}
      end
    else
      inherited DrawCell(ACol, ARow, ARect, AState);
  end;

procedure TNWOutline.WMLButtonDown(var Message: TWMLButtonDown);
  var
    i,
    ntemp      : integer ;
    ttList     : TList   ;
    ncursor    : TCursor ;
    fileList,
    tempList   : TStringList ;
    ctemp,
    cDN,
    cObjName    : string ;
    expandEvent : TNotifyEvent ;
    serverInfo  : TNWServerInfo ;
  begin
    inherited ;
    ncursor       := screen.cursor ;
    screen.cursor := crHourglass ;
    expandEvent   := nil ;
    tempList      := nil ;
    try
      if (itemCount < 1) or (NWBase = nil) then
        exit ;
      with items[selectedItem] do begin
        if hasItems and
           (TNWDSObjType(items[selectedItem].data) <> dsoQueue) then
          begin
            if expanded then
              collapse
            else
              expand;
          end
        else 
          begin  
            cObjName := items[selectedItem].text ;
            cDN := _getFullPath(selectedItem) ;
            case TNWDSObjType(items[selectedItem].data) of
              dsoTree :
                begin
                  tempList := ndsList(cObjName,'') ;
                  expandEvent := FTreeExpand ;
                end;
              dsoOrganization :           {show object subordinates}
                begin
                  tempList := ndsList('',cDN) ; 
                  expandEvent := FOrgExpand ;
                end;  
              dsoOrganizationalUnit :     {show object subordinates}
                begin
                  tempList := ndsList('',cDN) ; 
                  expandEvent := FOrgUnitExpand ;
                end;     
              dsoGroup :                  {show members}
                begin
                 tempList := ndsGetGroupMembers('',cDN) ;
                 expandEvent := FMembersExpand ;
                end;   
              dsoQueue,dsoBinderyQueue :  {show queue jobs}
                 begin
                   tempList := TStringList.create ;
                   while items[selectedItem].hasItems do
                     delete(items[selectedItem].getFirstChild);
                   if (pos('.',cObjName) > 0) then  
                     getQueueJobList(0,cObjName,tempList) 
                   else  
                     getQueueJobList(0,cDN,tempList) ;
                   for i := 0 to (tempList.count -1) do
                     tempList.objects[i] := TObject(dsoQueueJob) ;
                   expandEvent := FPrintQueueJobExpand ;
                 end; 
              dsoPrinter :                {show queues}
                begin
                  tempList := TStringList.create ;
                  ttList := TList.create ;
                  try
                    if ndsGetMultiAttr('',cDN, dsaQueue, ttList) then
                      for i := 0 to (ttList.count-1) do
                        tempList.addObject(strPas(PChar(ttList[i]^)),TObject(dsoQueue)) ;
                  finally
                    ttList.free ;
                  end;
                  expandEvent := FPrinterExpand ;
                end;  
              dsoPrintServer :            {show print server host device}    
                begin
                  tempList := TStringList.create ;
                  if ndsGetStringAttr('',cDN,dsaHostDevice,ctemp) then
                    tempList.addObject(ctemp,TObject(ndsGetObjType('',ctemp))) ;
                  expandEvent := FPrintServerExpand ;  
                end;
              dsoServer,dsoNCPServer :    {show server info}
                begin
                  tempList := TStringList.create ;
                  {if getServerStats(openConnByName(cDN,true,true),serverInfo) then <-safer/slower}
                  if getServerStats(getServerHandle(ndsGetTypelessName(cObjName)),serverInfo) then
                     tempList.addObject('Netware Version: ' + serverInfo.version,TObject(0)) ;
                  expandEvent := FServerExpand ;   
                end;  
              dsoVolume :                 {show volume's dirs/files}  
                begin
                  if ndsGetStringAttr('',cDN,dsaHostResourceName,cObjName) then
                    begin
                      ctemp := getServerName(ndsAuthenticateObject('',cDN)) ;
                      if (ctemp <> '') then
                        begin
                          tempList := getVolDirList(ctemp + '/' + cObjName + ':\*') ;
                          for i := 0 to (tempList.count-1) do
                            tempList.objects[i] := TObject(dsoDirectory) ;
                          
                          fileList := getVolFileList(ctemp + '/' + cObjName + ':\*.*') ;
                          for i := 0 to (fileList.count-1) do 
                            tempList.addObject(fileList[i],TObject(dsoFile)) ;
                          fileList.free ;
                        end
                      else
                        tempList := TStringList.create ;
                    end
                  else
                    tempList := TStringList.create ;
                  expandEvent := FVolumeExpand ;  
                end;
              dsoDirectory :              {show directory files}  
                begin
                  expandEvent := FDirectoryExpand ;
                  tempList := TStringList.create ;
                end;
              dsoFile      :              {show file details}
                begin
                  expandEvent := FFileExpand ;
                  tempList := TStringList.create ;
                end;  
              dsoUser :                   {show user information}
                begin
                  tempList := TStringList.create ;
                  if (@FUserExpand <> nil) then
                    expandEvent := FUserExpand 
                  else
                    begin  {provide default expansion if not event}
                      if ndsGetStringAttr('', cDN, dsaFullName,ctemp) then
                        tempList.add('Full Name: ' + ctemp) ;
                      ttList := TList.create ;  
                      try  {get email addresses}
                        if ndsGetMultiAttr('', cDN, dsaEmailAddress, ttList) then
                          for i := 0 to (ttList.count-1) do
                            tempList.add('E-Mail: ' + strPas(TNWDSEMailInfo(ttList[i]^).emailAddress)) ;
                      finally
                        ttList.free ;
                      end;
                      ttList := TList.create ;
                      try  {get telephone numbers}
                        if ndsGetMultiAttr('',cDN, dsaTelephoneNumber,ttList) then
                          for i := 0 to (ttList.count-1) do
                            tempList.add('Telephone: ' + strPas(ttList[i])) ;
                      finally
                        ttList.free ;
                      end;
                      (*
                      ttList := TList.create ;
                      try  {get network address}
                        if ndsGetMultiAttr('',cDN, dsaNetworkAddress,ttList) then
                          for i := 0 to (ttList.count-1) do
                            tempList.add('NetAddress: ' + strPas(@TNWDSNetworkAddress(ttList[i]^).address) ) ;
                      finally
                        ttList.free ;
                      end; *)
                  end;
                end;  
              else
                begin
                  expandEvent := FOthersExpand ;
                  tempList := TStringList.create ;
                end;  
            end;    
            {-- add stringlist elements to selected outline item --}
            try
              ntemp := selectedItem ;
              if sorted then
                tempList.sort ;
              for i := 0 to (tempList.count-1) do 
                if (i = 0) then
                  ntemp := addChildObject(ntemp,strTran(tempList[i],char(VK_TAB),space(5)),tempList.objects[i]) 
                else
                  ntemp := addObject(ntemp,strTran(tempList[i],char(VK_TAB),space(5)),tempList.objects[i]) ;
              if hasItems then
                expand 
              else if (items[selectedItem].data <> nil) then
                selectOutlineItem ;
            finally
              tempList.free ;
            end;    
              
            {--- run 'after expand' event --- }  
            if (@expandEvent <> nil) then
              expandEvent(self) ;
          end;
      end;  
    finally
      screen.cursor := ncursor ;
    end;    
  end;  

procedure TNWOutline.selectOutlineItem ;
  begin
    {virtual}
  end;
  
{*********************** Property Read Implementation **************************}  
function TNWOutline._getText : string ;
  begin
    result := _getFullPath(selectedItem) ;
  end;

{*********************** Property Write Implementation *************************}  
procedure TNWOutline. _setNWBase(value : TNWBase) ;
  begin
    if (value <> FNWBase) then
      FNWBase := value ;
     _populate ;
  end;

procedure TNWOutline._setObjType(value : TNWObjListType) ;
  begin
    if (value <> FObjType) then
      begin
        FObjType := value ;  
        _populate ;
      end;  
  end;

procedure TNWOutline._setCustomImages(value : boolean) ;
  begin
    FCustomImages := value ;
    _populate ;  
  end;  
  
{*************************** Internal Functions ********************************}  
procedure TNWOutline._populate ;
  {populate items list}
  var
    i,
    index    : integer ;
    tempList : TStringList ;
  begin
    index := 0 ;
    if (csLoading in componentState) then
      {nothing}
    else if (showObjType = nwlNone) then
      begin
        clear ;
        exit ;
      end  
    else if (FNWBase <> nil) then
      begin
        tempList := FNWBase.getObjectList(showObjType) ; 
        try
          if sorted then
            tempList.sort ;
          beginUpdate ;
          clear ;
          for i := 0 to (tempList.count-1) do 
            index := addObject(index,tempList[i],tempList.objects[i]) ;
          endUpdate ;
        finally
          tempList.free ;
        end;  
      end;  
  end;  
  
  
function TNWOutline._getFullPath(selectedItem : integer) : string ;
  var
    tempNode : TOutlineNode ;
  begin
    result := '' ;
    if (FNWBase = nil) then
      exit ;
    tempNode := items[selectedItem] ;
    result := tempNode.text ;
    while true do begin
      tempNode := tempNode.parent ;
      if (tempNode = nil) then
        exit ;
      if (length(tempNode.text) > 0) then
        result := result + '.' + tempNode.text ;  
    end;
  end;

end.
 