unit nwbase;

interface

uses
  sysUtils,
  winTypes,
  WinProcs,
  Classes,
  Forms,
  Controls,
  messages ;

const
  NWM_BASE               = (CM_BASE + 2112) ;
  NWM_BASESERVERCHANGE   = NWM_BASE + 1 ;
  NWM_BASEUSERCHANGE     = NWM_BASE + 2 ;
  NWM_BASECONTEXTCHANGE  = NWM_BASE + 3 ;

type  
  {$I nwtypes.inc}
  {$I nwlib.inc}

type  
  TNWObjError = class(Exception) ;
  
  TNWObjType = (nwObjTypeNone,nwObjTypeUser,nwObjTypeGroup,nwObjTypePrintQ,
                nwObjTypeFileServer,nwObjTypeJobServer,nwObjTypeGateway,
                nwObjTypePrintServer,nwObjTypeArchiveQueue,nwObjTypeArchiveServer,
                nwObjTypeJobQueue,nwObjTypeAdministration,nwObjTypeNASSNAServer,
                nwObjTypeRemoteBridge,nwObjTypeTCPIPGateway,nwObjTypeNDSTree,
                nwObjTypeConnectedServer) ;
                
  TNWObjListType = (nwlNone,nwlUserList,nwlUser,nwlGroup,nwlPrintQ,
                    nwlConnectedFileServer,nwlFileServer,
                    nwlPrintServer,nwlArchiveServer,nwlJobQueue,
                    nwlAdministration,nwlNASSNAServer,
                    nwlRemoteBridge,nwlTCPIPGateway,nwlNDSTree,nwlNDSObjList) ;
                    
  TNWShellErrorMode = (nwErrInteractive,nwErrNetware,nwErrDOS) ; 

  TNWBroadcastMode = (nwBroadcastOn, nwBroadcastServerOnly, nwBroadcastOff, nwBroadcastStored) ;

  {-- Netware Error Display --}                    
  TNWLastError = class(TPersistent) 
  private
    FErrNWLib,
    FErrNWServer,
    FErrNWProp,
    FErrNWPrint,
    FErrNWNDS      : string ;
  published
    property NWLib       : string read FErrNWLib    write FErrNWLib    stored false ; 
    property NWServer    : string read FErrNWServer write FErrNWServer stored false ; 
    property NWProp      : string read FErrNWProp   write FErrNWProp   stored false ; 
    property NWNDS       : string read FErrNWNDS    write FErrNWNDS    stored false ; 
    property NWPrint     : string read FErrNWPrint  write FErrNWPrint  stored false ;
  end;

  {-- context flags component --}
  TNWDSContextFlags = class(TPersistent) 
  private
    FDereferenceAliases,
    FTranslateStrings,
    FTypeLessNames ,
    FAsyncMode,
    FCanonicalizeNames,
    FDereferenceBaseClass,
    FDisallowReferrals    : boolean ;
    procedure _setTypeLessNames(value : boolean) ;
    procedure _setDereferenceAliases(value : boolean) ;
    procedure _setTranslateStrings(value : boolean) ;
    procedure _setAsyncMode(value : boolean) ;
    procedure _setCanonicalizeNames(value : boolean) ;
    procedure _setDereferenceBaseClass(value : boolean) ;
    procedure _setDisallowReferrals(value : boolean) ;
  public 
    constructor create ; 
  published
    property DereferenceAliases   : boolean read FDereferenceAliases   write _setDereferenceAliases default true ;
    property TranslateStrings     : boolean read FTranslateStrings     write _setTranslateStrings default true ;
    property TypelessNames        : boolean read FTypelessNames        write _setTypelessNames default false ;
    property AsyncMode            : boolean read FAsyncMode            write _setAsyncMode default false ;
    property CanonicalizeNames    : boolean read FCanonicalizeNames    write _setCanonicalizeNames default true ;
    property DereferenceBaseClass : boolean read FDereferenceBaseClass write _setDereferenceBaseClass default false;
    property DisallowReferrals    : boolean read FDisallowReferrals    write _setDisallowReferrals default false;
  end;    

  {-- master control unit --}  
  TNWBase = class(TComponent) 
  private
    FLoginID,
    FServerName,
    FDefaultServer,
    FContextName      : string  ;
    FNWInit,
    FIsNDS,
    FSilent           : boolean ;
    FBroadcastMode    : TNWBroadcastMode ;
    FShellErrorMode   : TNWShellErrorMode ;
    FServerHandle     : TNWConnHandle     ;
    FLastError        : TNWLastError      ;
    FContextFlags     : TNWDSContextFlags ;
    FOnServerChange,
    FOnUserChange     : TNotifyEvent ;
    function  _getServerHandle      : TNWConnHandle ;
    function  _getServerName        : string ;
    function  _getVersion           : string ;
    function  _getFullName          : string ;
    function  _getConnectID         : TNWConnNumber ;
    function  _getObjType           : TNWObjType    ;
    function  _getNWObjType         : TObjType      ;
    function  _getLastError         : TNWLastError  ;
    function  _getMaxServerConns    : word   ;
    function  _getDefaultServer     : string ;
    function  _getObjectID          : string ;
    function  _getLoginID           : string ;
    function  _getIsConsoleOperator : boolean ;
    function  _getIsManager         : boolean ;
    function  _getFirstNetDrive     : char ;
    function  _getNextNetDrive      : char ;
    function  _getBroadcastMode     : TNWBroadcastMode ;
    function  _getServerDate        : TDateTime ;
    function  _getIsNDS             : boolean ;
    function  _getServerVersion     : string ;
    function  _getContextHandle     : TNWDSContextHandle ;
    function  _getNDSPreferredTree  : string ;
    procedure _setVersion(value : string) ;
    procedure _setLoginID(value : string) ;
    procedure _setServerName(value : string) ;
    procedure _setContextName(value : string) ;
    procedure _setDefaultServer(value : string) ;
    procedure _setShellErrorMode(value : TNWShellErrorMode) ;
    procedure _setBroadcastMode(value : TNWBroadCastMode) ;
    procedure _setServerDate(value : TDateTime) ;
    procedure _setNDSPreferredTree(value : string) ;
    procedure runException(errorMsg : string) ;
    procedure notifyChange(msgID : longint) ;
  public                       
    constructor create(AOwner : TComponent) ; override ;
    destructor  destroy ; override ;
    function objTypeToNWObjType(value : TNWObjType) : TObjType ;
    function getObjectList(objListType : TNWObjListType) : TStringList ;
    function getQJobList(queueName : string) : TStringList ;
    function send(msgText : string) : boolean ;
    function login(serverName, password : string) : boolean ;
    function logout(serverHandle : TNWConnHandle) : boolean ;
    function checkPassword(password : string) : boolean ;
    function changePassword(oldPassword, newPassword : string) : boolean ;
    function isGroupMember(groupName : string) : boolean ;
    function equivalentToObject(objName : string) : boolean ;
    function abbreviateName(objName : string) : string ;
    function expandName(objName : string) : string ;
    
    { Windows 32 Basic Networking Functions }
    {$IFDEF Win32}
    function winNetGetUserName : string ;
    function winOpenNetConnectDialog(aOwner : TWinControl) : boolean ;  
    function winCloseNetPrintDialog(aOwner : TWinControl) : boolean ;
    function winCloseNetConnectDialog(aOwner : TWinControl) : boolean ;
    function winMapDrive(drive : char ; 
                         path, userName, password : string ;
                         rememberConnection : boolean) : boolean ;
    function winMapDelete(drive : char ; permanent : boolean) : boolean ;
    {$ENDIF}

    {properties}
    property NDSContextHandle  : TNWDSContextHandle read _getContextHandle ;
    function getConnList : TConnList ;
  published
    { events }
    property OnServerChange    : TNotifyEvent      read FOnServerChange   write FOnServerChange ;
    property OnUserChange      : TNotifyEvent      read FOnUserChange     write FOnUserChange ;
    { read only properties }
    property ServerVersion     : string            read _getServerVersion stored false ;
    { misc. properties }
    property NetAvailable      : boolean           read FNWInit           stored false ;
    property NDSContextName    : string            read FContextName      write _setContextName  ;
    property NDSContextFlags   : TNWDSContextFlags read FContextFlags     write FContextFlags ;
    property NDSAvailable      : boolean           read _getIsNDS         stored false ; 
    property NDSPreferredTree  : string            read _getNDSPreferredTree write _setNDSPreferredTree stored false ;
    property DefaultServer     : string            read _getDefaultServer    write _setDefaultServer stored false ;
    property LastError         : TNWLastError      read _getLastError        write FLastError stored false ;
    property SilentExceptions  : boolean           read FSilent              write FSilent default true ; 
    property Version           : string            read _getVersion          write _setVersion stored false ;
    property LoginID           : string            read _getLoginID          write _setLoginID stored false ;
    property ServerName        : string            read _getServerName       write _setServerName stored false ;
    property ServerDate        : TDateTime         read _getServerDate       write _setServerDate stored false ;
    property ShellErrorMode    : TNWShellErrorMode read FShellErrorMode
                                                   write _setShellErrorMode stored false ;
    property BroadcastMode     : TNWBroadcastMode  read FBroadcastMode 
                                                   write _setBroadcastMode  stored false ;
    property ServerHandle      : TNWConnHandle     read _getServerHandle      stored false ;
    property LoginFullName     : string            read _getFullName          stored false ;
    property ObjectID          : string            read _getObjectID          stored false ;
    property ConnectionID      : TNWConnNumber     read _getConnectID         stored false ;
    property ObjectType        : TNWObjType        read _getObjType           stored false ;
    property NWObjectType      : TObjType          read _getNWObjType         stored false ;
    property MaxServerConns    : word              read _getMaxServerConns    stored false ;
    property ConsoleOperator   : boolean           read _getIsConsoleOperator stored false ;
    property Manager           : boolean           read _getIsManager         stored false ;
    property FirstNetworkDrive : char              read _getFirstNetDrive     stored false ;
    property NextNetworkDrive  : char              read _getNextNetDrive      stored false ;
  end;


implementation

uses
  nwLib,
  nwServer,
  nwProp,
  nwPrint,
  nwNDS,
  nwTools ;

const
  version_ID = '1.50' ;
                          
var
  FNWLib      : TNWLib            ;
  FNWNDS      : TNWNDS            ;
  FNWServer   : TNWServer         ;
  FNWProp     : TNWProp           ;
  FNWPrint    : TNWPrint          ;
  serverStats : TNWServerInfo     ;  {shared server information}

{*********************** TNWBase Object Initialization **************************}  
constructor TNWBase.create(AOwner : TComponent) ;
  begin
    inherited create(AOwner) ;
    FNWInit         := false ;
    FNWLib          := TNWLib.create ;
    FNWNDS          := TNWNDS.create ;
    FNWServer       := TNWServer.create ;
    FNWPrint        := TNWPrint.create ;
    FNWProp         := TNWProp.create ;
    FLastError      := TNWLastError.create ;
    FContextFlags   := TNWDSContextFlags.create ;
    FBroadcastMode  := TNWBroadcastMode(getBroadcastMode(0)) ;
    FSilent         := true  ;
    FServerHandle   := 0     ;
    FLoginID        := ''    ;
    FServerName     := ''    ;
    FIsNDS          := false ;
    FShellErrorMode := nwErrInteractive ;
    FNWInit         := NWInit ;
    if FNWInit and ndsInit then
      begin
        setAutoNDS(true) ;
        FIsNDS         := ndsIsDSServer(serverHandle) ;
        ndsContextName := '' ;  {set to root}
        {set initial context flags}
        if ndsContextFlags.TypelessNames then
          ndsSetContextFlags(DCV_TYPELESS_NAMES) 
        else  
        if ndsContextFlags.DereferenceAliases then  
          ndsSetContextFlags(DCV_DEREF_ALIASES) ;
        if ndsContextFlags.TranslateStrings then    
          ndsSetContextFlags(DCV_XLATE_STRINGS) ;
        if ndsContextFlags.AsyncMode then           
          ndsSetContextFlags(DCV_ASYNC_MODE) ;
        if ndsContextFlags.CanonicalizeNames then   
          ndsSetContextFlags(DCV_CANONICALIZE_NAMES) ;
        if ndsContextFlags.DereferenceBaseClass then
          ndsSetContextFlags(DCV_DEREF_BASE_CLASS) ;  
        if ndsContextFlags.DisallowReferrals then   
          ndsSetContextFlags(DCV_DISALLOW_REFERRALS) ;
      end;  
    {initial login ID}  
    if FIsNDS then
      FLoginID := ndsExpandName(ndsWhoAmI)
    else  
      FLoginID := whoAmI(serverHandle) ;
  end;

destructor TNWBase.destroy ;
  begin
    if ndsAvailable then
      ndsClose ;
    if assigned(FNWNDS) then 
      FNWNDS.free ;
    if assigned(FNWLib) then
      FNWLib.free ;
    if assigned(FNWServer) then
      FNWServer.free ;
    if assigned(FNWPrint) then
      FNWPrint.free ;
    if assigned(FNWProp) then
      FNWProp.free ;  
    if assigned(FLastError) then
      FLastError.free ;  
    if assigned(FContextFlags) then
      FContextFlags.free ;  
    FNWLib        := nil ;
    FNWNDS        := nil ;
    FNWServer     := nil ;
    FNWPrint      := nil ;
    FNWProp       := nil ;
    FContextFlags := nil ;
    FLastError    := nil ;
    inherited destroy ;
  end;  

procedure TNWBase.runException(errorMsg : string) ;
  begin
    if (not (csDesigning in componentState)) and (not FSilent) then
      raise TNWObjError.create(errorMsg) ;
  end;  

procedure TNWBase.notifyChange(msgID : longint) ;
  var
    i : integer ;
  begin
    if (not (csLoading in componentState)) and
       (owner is TForm) then
      with owner as TWinControl do 
        for i := 0 to (componentCount-1) do 
          if components[i] is TWinControl then
            sendMessage(TWinControl(components[i]).handle,msgID,0,0) ;
  end;  
  

{************************* Published Functions **********************************}  
function TNWBase.objTypeToNWObjType(value : TNWObjType) : TObjType ;
  begin
    case value of
      nwObjTypeUser            : result := $0100 ;
      nwObjTypeGroup           : result := $0200 ;
      nwObjTypePrintQ          : result := $0300 ;
      nwObjTypeFileServer      : result := $0400 ;
      nwObjTypeJobServer       : result := $0500 ;
      nwObjTypeGateway         : result := $0600 ;
      nwObjTypePrintServer     : result := $0700 ;
      nwObjTypeArchiveQueue    : result := $0800 ;
      nwObjTypeArchiveServer   : result := $0900 ;
      nwObjTypeJobQueue        : result := $0A00 ;
      nwObjTypeAdministration  : result := $0B00 ;
      nwObjTypeNASSNAServer    : result := $2100 ;
      nwObjTypeRemoteBridge    : result := $2600 ;
      nwObjTypeTCPIPGateway    : result := $2700 ;
    else
      result := 0 ;
    end;
  end;

function TNWBase.checkPassword(password : string) : boolean ;
  begin
    if ndsAvailable then
      result := ndsPassCheck('',loginID,password)
    else
      result := nwPassCheck(serverHandle,loginID,password) ;
  end;
  
function TNWBase.changePassword(oldPassword, newPassword : string) : boolean ;
  begin
    if ndsAvailable then
      result := ndsChangePassword('',loginID,newPassword,oldPassword) 
    else
      result := changeNWPassword(serverHandle,loginID,newPassword,oldPassword) ;
  end;
  
function TNWBase.isGroupMember(groupName : string) : boolean ;
  var
    i        : integer ;
    tempList : TStringList ;
  begin
    result := false ;
    if ndsAvailable then
      begin
        tempList := ndsGetMyGroups(ndsContextName,loginID) ;
        try
          for i := 0 to (tempList.count-1) do 
            if (lowerCase(ndsAbbreviateName(tempList[i])) = lowerCase(groupName)) then
              begin
                result := true ;
                break ;
              end;
        finally
          tempList.free ;
        end;  
      end
    else
      result := isMember(serverHandle,loginID,groupName) ;
  end;

  
function TNWBase.equivalentToObject(objName : string) : boolean ;
  begin
    result := secureEquiv(serverHandle,loginID,objName) ;
  end;  


{******************* method implementatation **************************}  
function TNWBase.expandName(objName : string) : string ;
  begin
    result := ndsExpandName(objName) ;
  end;
  
function TNWBase.abbreviateName(objName : string) : string ;
  begin
    result := ndsAbbreviateName(objName) ;
  end;
  
function TNWBase.getQJobList(queueName : string) : TStringList ;
  begin
    result := TStringList.create ;
    getQueueJobList(serverHandle,queueName,result) ;
  end;
    
function TNWBase.getObjectList(objListType : TNWObjListType) : TStringList ;
  begin
    case objListType of
      nwlUser                : result := getBinderyList(serverHandle,nw_user) ;
      nwlUserList            : result := getUserList(serverHandle,false) ;
      nwlGroup               : result := getBinderyList(serverHandle,nw_group) ;
      nwlPrintQ              : result := getBinderyList(serverHandle,nw_printQ) ;
      nwlFileServer          : result := getBinderyList(serverHandle,nw_server) ;
      nwlConnectedFileServer : result := getConnectedServerList ;
      nwlPrintServer         : result := getBinderyList(serverHandle,nw_printServer) ;
      nwlArchiveServer       : result := getBinderyList(serverHandle,nw_archiveServer) ;
      nwlJobQueue            : result := getBinderyList(serverHandle,nw_jobServer) ;
      nwlAdministration      : result := getBinderyList(serverHandle,nw_administration) ;
      nwlNASSNAServer        : result := getBinderyList(serverHandle,nw_nasSNAServer) ;
      nwlRemoteBridge        : result := getBinderyList(serverHandle,nw_remoteBridge) ;
      nwlTCPIPGateway        : result := getBinderyList(serverHandle,nw_tcpIPGateway) ;
      nwlNDSObjList          : if ndsAvailable then
                                 result := ndsList(NDSContextName,'') 
                               else
                                 result := TStringList.create ;  
      nwlNDSTree             : if ndsAvailable then 
                                 result := ndsGetTreeList 
                               else
                                 result := TStringList.create ;  
    else
      result := TStringList.create ;                           
    end;                                 
  end;

function TNWBase.send(msgText : string) : boolean ;
  begin
    result := sendLineMessage(serverHandle,loginID,msgText) ;
  end;  

function TNWBase.login(serverName, password : string) : boolean ;
  begin
    if ndsAvailable then
      result := ndsLogin('',loginID,password)
    else
      result := nwLogin(serverName,loginID,password,objTypeToNWObjType(objectType)) ;
  end; 

function TNWBase.logout(serverHandle : TNWConnHandle) : boolean ;
  begin
    if (serverHandle <> 0) then
      result := closeConnection(serverHandle)
    else if ndsAvailable then
      result := ndsLogout
    else
      result := nwLogout(serverHandle) ;  
  end;

{*************** Windows 32 Networking Functions ******************}  
{$IFDEF Win32}
function TNWBase.winNetGetUserName : string ;
  var
    bufSize  : DWORD ;
    cpObject : array[0..128] of char ;
  begin
    result  := '' ;
    bufsize := sizeOf(cpObject) ;
    if (WNetGetUser(nil,@cpObject,bufSize) = NO_ERROR) then
      result := strPas(cpObject) ;
  end;

function TNWBase.winOpenNetConnectDialog(aOwner : TWinControl) : boolean ;  
  begin
    result := (WNetConnectionDialog(aOwner.handle,RESOURCETYPE_DISK) = NO_ERROR) ;
  end;

function TNWBase.winCloseNetConnectDialog(aOwner : TWinControl) : boolean ;
  begin
    result := (WNetDisconnectDialog(aOwner.handle, RESOURCETYPE_DISK) = NO_ERROR) ;
  end;

function TNWBase.winCloseNetPrintDialog(aOwner : TWinControl) : boolean ;
  begin
    result := (WNetDisconnectDialog(aOwner.handle, RESOURCETYPE_PRINT) = NO_ERROR) ;
  end;
  
function TNWBase.winMapDelete(drive : char ; permanent : boolean) : boolean ;
  var
    cpDrive : array[0..128] of char ;
    flags   : integer ;
  begin
    if permanent then
      flags := CONNECT_UPDATE_PROFILE	
    else
      flags := 0 ;  
    strPCopy(cpDrive,drive + ':') ;  
    result := (WNetCancelConnection2(cpDrive,flags,true) = NO_ERROR) ;
  end;
  
function TNWBase.winMapDrive(drive : char ; 
                             path, userName, password : string ;
                             rememberConnection : boolean) : boolean ;
  var
    flags        : integer ;
    nError       : longint ;
    netRes       : TNETRESOURCE ; 
    cpUser,
    cpPassword,
    cpLocalPath  : array[0..128] of char ;
  begin                                
    result := false ;
    flags  := 0 ;
    with netRes do begin
      dwType := RESOURCETYPE_DISK ;
      strPCopy(@lpLocalName,drive + ':') ;
      strPCopy(@lpRemoteName,path) ;
      strPCopy(@lpProvider,'') ;
    end;
    if rememberConnection then          
      flags := (flags or CONNECT_UPDATE_PROFILE) ;
    strPCopy(cpUser,userName) ;
    strPCopy(cpPassword,password) ;  
    nError := WNetAddConnection2(netRes,@cpPassword,@cpUser,0) ;
    if (nError = NO_ERROR) then
      result := true 
    else  
      begin
        // fall back on older function
        strPCopy(cpLocalPath,drive + ':') ;
        result := (WNetAddConnection(PChar(path),cpPassword,cpLocalPath) = NO_ERROR) ;
      end;
  end;  
  

{$ENDIF}
  
  
{****************** read property implementation **********************}

function TNWBase._getNDSPreferredTree : string ; 
  begin
    result := '' ;
    if ndsAvailable then
      result := ndsGetServerTreeName(ndsGetPreferredDSServer) ;
  end;

function TNWBase._getContextHandle : TNWDSContextHandle ;
  begin
    result := getInternalContextHandle ;
  end;
  
function TNWBase._getServerVersion : string ;
  begin
    result := '0.0' ;
    if getServerStats(serverHandle,serverStats) then
      result := serverStats.version ;
  end;
  
function TNWBase._getIsNDS : boolean ;
  begin
    result := (FIsNDS and ndsIsDSServer(serverHandle)) ;
  end;

function TNWBase._getServerDate : TDateTime ;
  begin
    result := getServerDate(serverHandle) + getServerTime(serverHandle) ;
  end;                         

function TNWBase._getLoginID : string ;
  begin
    result := '' ;
    if (FLoginID <> '') then
      result := FLoginID 
    else if ndsAvailable then
      FLoginID := ndsWhoAmI
    else
      FLoginID := whoAmI(serverHandle) ;
    result := FLoginID ;  
  end;  

function TNWBase._getObjType : TNWObjType ;
  begin
    result := nwObjTypeUser ;
    case getObjType(serverHandle,FLoginID) of
      nw_user           : result := nwObjTypeUser ;
      nw_group          : result := nwObjTypeGroup ;
      nw_printQ         : result := nwObjTypePrintQ ;
      nw_server         : result := nwObjTypeFileServer ;
      nw_jobServer      : result := nwObjTypeJobServer ;
      nw_gateway        : result := nwObjTypeGateway ;
      nw_printServer    : result := nwObjTypePrintServer ;
      nw_archiveQueue   : result := nwObjTypeArchiveQueue ;
      nw_archiveServer  : result := nwObjTypeArchiveServer ;
      nw_jobQueue       : result := nwObjTypeJobQueue ;
      nw_administration : result := nwObjTypeAdministration ;
      nw_nasSNAServer   : result := nwObjTypeNASSNAServer ;
      nw_remoteBridge   : result := nwObjTypeRemoteBridge ;
      nw_TCPIPGateway   : result := nwObjTypeTCPIPGateway;
    else
      runException('Invalid Netware Object Type') ;  
    end;  
  end;

function TNWBase._getNWObjType : TObjType      ;
  begin
    result := getObjType(serverHandle,FLoginID) ;
  end;
  
function TNWBase._getFullName : string ;
  begin
    result := fullName(serverHandle,FLoginID) ;
  end;

function TNWBase._getConnectID : TNWConnNumber ;
  begin
    result := getConnectID(serverHandle) ;
  end;
    
function TNWBase._getVersion : string ;
  begin
    result := version_id + ' / ' + getNWLibVersion ;
  end;

function TNWBase._getServerHandle : TNWConnHandle ;
  begin
    result := 0 ;
    if (FServerHandle = 0) then
      FServerHandle := getPrimaryServerID ;
    result := FServerHandle ;  
  end;
  
function TNWBase._getServerName : string ;
  begin
    result := '' ;
    if (length(FServerName) < 1) then
      FServerName := getServerName(serverHandle) ;
    result := FServerName ;  
  end;

function TNWBase._getLastError : TNWLastError ;
  begin
    result          := FLastError ;
    result.NWLib    := FNWLib.getLastError ;
    result.NWServer := FNWServer.getLastError ;
    result.NWProp   := FNWProp.getLastError ;
    result.NWNDS    := FNWNDS.getLastError ;
    result.NWPrint  := FNWPrint.getLastError ;
  end;

function TNWBase._getDefaultServer : string ;
  begin
    result := getServerName(getPrimaryServerID) ;
  end;  

function TNWBase._getObjectID : string ;
  var
    nServer : TNWConnHandle ;
  begin
    nServer := serverHandle ;
    if ndsAvailable then
      result := ndsGetObjID(serverHandle,loginID) 
    else
      result := getObjID(nServer,loginID,objTypeToNWObjType(objectType)) ;
  end;  

function TNWBase._getMaxServerConns : word ;
  begin
    result := maxConns ;
  end;  

function TNWBase._getIsConsoleOperator : boolean ;
  begin
    result := isConsoleOperator(serverHandle) ;
  end;
  
function TNWBase._getIsManager : boolean ;
  begin
    result := isNWManager(serverHandle) ;
  end;

function TNWBase._getFirstNetDrive     : char ;
  begin
    result := getFirstNetDrive ;
  end;
  
function TNWBase._getNextNetDrive      : char ;
  begin
    result := getNextNetDrive ;
  end;

function TNWBase._getBroadcastMode : TNWBroadcastMode ;
  begin
    result := TNWBroadcastMode(getBroadcastMode(serverHandle)) ;
  end;  

{****************** write property implementation **********************}

procedure TNWBase._setVersion(value : string) ;
  begin
    if (csDesigning in componentState) and 
       (not (csLoading in componentState)) then
      okBox('NWLib Version: ' + getNWLibVersion + ';;' +
            'Component Version: ' + version_ID + ';;' +
            'Copyright 1995-2000 Devont Software Inc.;;http://www.devont.com;' +
            'sales@devont.com;fax (281) 754-4203') ;
  end;

procedure TNWBase._setServerDate(value : TDateTime) ;
  begin
    setServerDateTime(serverHandle,value) ;
  end;

procedure TNWBase._setLoginID(value : string) ;
  begin
    if (value <> '') then
      FLoginID := upperCase(value) 
    else if ndsAvailable then
      FLoginID := ndsWhoAmI
    else
      FLoginID := whoAmI(serverHandle) ;
    notifyChange(NWM_BASEUSERCHANGE) ;
    if assigned(FOnUserChange) then
      FOnUserChange(self) ;
  end;  

procedure TNWBase._setServerName(value : string) ;
  begin
    value := upperCase(value) ;
    if (length(value) < 1) then
      begin
        FServerHandle := getPrimaryServerID  ;
        FServerName   := getServerName(FServerHandle) ;
      end
    else 
      begin
        if (getServerHandle(value) <> 0) then
          begin
            FServerHandle := getServerHandle(value) ;
            FServerName   := value ;
          end
        else 
          runException('Invalid Server Handle') ;
      end;   
    notifyChange(NWM_BASESERVERCHANGE) ;
    if assigned(FOnServerChange) then
      FOnServerChange(self) ;
  end;
  
procedure TNWBase._setContextName(value : string) ;
  begin
    if ndsAvailable then 
      begin
        value := upperCase(value) ;
        if (value = '') or (value = '[ROOT]') then  
          begin
            resetNWLibContext ;
            value := ndsGetContextName ;
          end
        else  
           ndsSetContext(value);
        FContextName := value ;
        {notifyChange(NWM_BASECONTEXTCHANGE) ;}
        notifyChange(NWM_BASESERVERCHANGE) ;
      end;                             
  end;

procedure TNWBase._setDefaultServer(value : string) ;  

  var
    nServer : TNWConnHandle ;
  begin
    if (value <> FDefaultServer) then
      if setPreferredServer(value,nServer) then
        FDefaultServer := value ;
  end;

procedure TNWBase._setShellErrorMode(value : TNWShellErrorMode) ;
  begin
    if (value <> FShellErrorMode) then
      FShellErrorMode := value ;
    setShellErrorMode(ord(value)) ;  
  end;  

procedure TNWBase._setBroadcastMode(value : TNWBroadcastMode) ;
  begin
    FBroadcastMode := value ;
    setBroadcastMode(serverHandle,word(ord(value))) ;
  end;  

procedure TNWBase._setNDSPreferredTree(value : string) ;
  begin
    if (not ndsAvailable) then
      exit ;
    ndsSetPreferredTreeName(value) ;
  end;

{**************** TNWDSContextFlags Set/Get Methods *********************}  

constructor TNWDSContextFlags.create ;
  begin
    inherited create ;
    {netware context defaults}
    FDereferenceAliases   := true  ;
    FTranslateStrings     := true  ;
    FTypelessNames        := false ;
    FAsyncMode            := false ;
    FCanonicalizeNames    := true  ;
    FDereferenceBaseClass := false ;
    FDisallowReferrals    := false ;
  end;
    
procedure TNWDSContextFlags._setTypelessNames(value : boolean) ;
  var
    flags : longint ;
  begin
    FTypelessNames := value ;
    flags := ndsGetContextFlags ;
    if value then
      flags := (flags or DCV_TYPELESS_NAMES)
    else
      flags := (flags - DCV_TYPELESS_NAMES) ;  
    ndsSetContextFlags(flags) ;  
  end;
  
procedure TNWDSContextFlags._setDereferenceAliases(value : boolean) ;
  var
    flags : longint ;
  begin
    FDereferenceAliases := value ;
    flags := ndsGetContextFlags ;
    if value then
      flags := (flags or DCV_DEREF_ALIASES) 
    else
      flags := (flags - DCV_DEREF_ALIASES) ;  
    ndsSetContextFlags(flags) ;  
  end;

procedure TNWDSContextFlags._setTranslateStrings(value : boolean) ;
  var
    flags : longint ;
  begin
    FTranslateStrings := value ;
    flags := ndsGetContextFlags ;
    if value then
      flags := (flags or DCV_XLATE_STRINGS) 
    else
      flags := (flags - DCV_XLATE_STRINGS) ;  
    ndsSetContextFlags(flags) ;  
  end;

procedure TNWDSContextFlags._setAsyncMode(value : boolean) ;
  var
    flags : longint ;
  begin
    FAsyncMode := value ;
    flags := ndsGetContextFlags ;
    if value then
      flags := (flags or DCV_ASYNC_MODE) 
    else
      flags := (flags - DCV_ASYNC_MODE) ;  
    ndsSetContextFlags(flags) ;  
  end;

procedure TNWDSContextFlags._setCanonicalizeNames(value : boolean) ;
  var
    flags : longint ;
  begin
    FCanonicalizeNames := value ;
    flags := ndsGetContextFlags ;
    if value then
      flags := (flags or DCV_CANONICALIZE_NAMES) 
    else
      flags := (flags - DCV_CANONICALIZE_NAMES) ;  
    ndsSetContextFlags(flags) ;  
  end;

procedure TNWDSContextFlags._setDereferenceBaseClass(value : boolean) ;
  var
    flags : longint ;
  begin
    FDereferenceBaseClass := value ;
    flags := ndsGetContextFlags ;
    if value then
      flags := (flags or DCV_DEREF_BASE_CLASS) 
    else
      flags := (flags - DCV_DEREF_BASE_CLASS) ;  
    ndsSetContextFlags(flags) ;  
  end;

procedure TNWDSContextFlags._setDisallowReferrals(value : boolean) ;
  var
    flags : longint ;
  begin
    FDisallowReferrals := value ;
    flags := ndsGetContextFlags ;
    if value then
      flags := (flags or DCV_DISALLOW_REFERRALS) 
    else
      flags := (flags - DCV_DISALLOW_REFERRALS) ;  
    ndsSetContextFlags(flags) ;  
  end;

function TNWBase.getConnList : TConnList ;
var
  nConns : integer ;
begin
  if (not getUserConnList(FServerHandle,FLoginID,nConns,@result)) then
    alertBox('Error Getting ConnList') ;
end;  


  

end.
 