{ *************** General Netware Functions ****************

  Developed with Novell Inc.'s Client SDK for C SDK

  Copyright 1995-1999 by Devont Software Inc. and Jim Tyson.  All
  rights reserved.

}
unit nwlib;

interface

uses
  SysUtils,
  winTypes,
  WinProcs,
  Classes,
  Forms,
  Controls ;

{$IFDef Ver110}  
  {$ObjExportAll On}
{$ENDIF}  

  {$I nwlib.inc}

  type
    {$I nwtypes.inc}

  type
    ExNWLib = class(Exception) ;

  type
    TNWCCVersion = Record
      major : nuint;
      minor : nuint;
      revision : nuint
    end;

  Type
    pTNWCCConnInfo = ^TNWCCConnInfo ;
    TNWCCConnInfo = Record
      authenticationState  : nuint;
      broadcastState       : nuint;
      connRef              : nuint32;
      treeName             : array[0..NW_MAX_TREE_NAME_LEN-1] of char ;
      connNum              : nuint;
      userID               : nuint32;
      serverName           : array[0..NW_MAX_SERVER_NAME_LEN-1] of char ;
      NDSState             : nuint;
      maxPacketSize        : nuint;
      licenseState         : nuint;
      distance             : nuint;
      serverVersion        : TNWCCVersion ;
    end;

  { General Connection Information Interface Structure }
  type
    TNWConnectInfo = record
      serverConnID  : TNWConnHandle ;
      loginDateTime : TDateTime     ;
      treeName,
      internet      : string        ;
      sessionID     : word          ;
      ConnectID     : TNWConnNumber ;
      userID        : string        ;
      objType       : TObjType      ;
      serverName    : string        ;
  end;

  {used by getPathInfo/NWParsePath to obtain individual elements}
  type
    TNWParsedPath = record
      nServer      : TNWConnHandle ;
      serverName   : string[40]  ;
      volName      : string[128] ;
      dirPath      : string[128] ;
  end;

  {Used by NWParseNetwarePath to obtain ID Elements}
  type
    TNWPathInfo = record
      nServer      : TNWConnHandle ;
      cServer      : string ;
      volumeID     : TNWVolNum ;
      volumeName   : string ;
      dirHandle    : TNWDirHandle ;
      pathOnly     : string ;
      relativePath : string ;
  end;

  type
    TNWMapInfo = record
      nServer     : TNWConnHandle ;
      serverName  : string[40]  ;
      fullPath    : string[128] ;
      driveStatus : word  ;
  end;

  type
    TNWConnectionStats = record
      bytesRead    : longint ;
      bytesWritten : longint ;
      totalPackets : longint ;
    end;

  { Netware Function Declarations }
  function nwInit : Boolean ;
  function whoAmI(nServer : TNWConnHandle) : string ;
  function fullName(nServer : TNWConnHandle ; cUserID : string ) : string ;
  function getObjName(nServer : TNWConnHandle ; nObjID : TObjID) : string ;
  function getObjID(nServer : TNWConnHandle ;  
                    objectName : string; nObjType : TObjType) : string ;
  function getObjType(nServer : TNWConnHandle ; cUserID : string) : TObjType ;
  function getPrimaryServerID : TNWConnHandle ;
  function getConnectID(nServer : TNWConnHandle) : TNWConnNumber ;
  function maxConns : TNWNumber ;
  function sendLineMessage(nServer : TNWConnHandle ; cUserID, cMessageText : string) : boolean ;
  function getBinderyList(nServer : TNWConnHandle ; SearchType : TObjType) : TStringList ;
  function getMemberList(nServer : TNWConnHandle; cGroupName : string ; lSelf : boolean) : TStringList ;
  function isMember(nServer : TNWConnHandle ; cUserID,cGroup : string) : boolean ;
  function isLoggedIn(nServer : TNWConnHandle ; cUserID : string) : boolean ;
  function isConsoleOperator(nServer : TNWConnHandle) : boolean ;
  function isNWManager(nServer : TNWConnHandle) : boolean ;
  function secureEquiv(nServer : TNWConnHandle ; cUserID : string; cEquiv: string) : boolean ;
  function longSwap(nLong : longint) : longint ;
  function _nwCopy(inFile, outFile : string) : boolean ;
  function nCopy(infile, outfile : string) : boolean ;
  function isInList(nServer : TNWConnHandle ; clist : string ) : boolean ;
  function getFirstNetDrive : char ;
  function getNextNetDrive  : char ;
  function map(cDrive: char; cPath: string ) : boolean ;
  function mapDelete(cDrive : string) : Boolean ;
  function mapShow : TStringList ;
  function getServerName(nServer : TNWConnHandle) : string ;
  function getServerDate(nServer : TNWConnHandle) : TDateTime ;
  function getServerTime(nServer : TNWConnHandle) : TDateTime ;
  function getConnectedServerList : TStringList ;
  function setPreferredServer(servername : string ;
                              var nServer : TNWConnHandle) : boolean ;
  function nwPassCheck(nServer : TNWConnHandle; cUserID, cPassword : string) : boolean ;
  function sList : TStringList ;
  function getMyGroups(nServer : TNWConnHandle ;
                       cUserID : string) : TStringList ;
  function getConnectInfo(nServer : TNWConnHandle ;
                          nConn   : TNWConnNumber ;
                          var connectInfo : TNWConnectInfo) : boolean ;
  function parseNetwarePath(nServer : TNWConnHandle ;
                            cPath : string;
                            var pathInfo : TNWPathInfo ) : boolean ;
  function getMapInfo(drive : char ; var mapInfo : TNWMapInfo) : boolean ;
  function getPathInfo(cPath : string; var parsedPath : TNWParsedPath) : boolean ;
  function getNWLibVersion : string ;
  function convertLocalFileHandle(localFileHandle : TNWFileHandle) : TNWFileHandle ;
  function getOpenFileUsers(nServer : TNWConnHandle ;
                            cFile   : string) : TStringList ;
  function getConnOpenFiles(nServer : TNWConnHandle ;
                            nConn   : TNWConnNumber) : TStringList ;
  function castOn(nServer : TNWConnHandle) : boolean ;
  function castOff(nServer : TNWConnHandle) : boolean ;
  function getBroadcastMessage(nServer : TNWConnHandle) : string ;
  function getBroadcastMode(nServer : TNWConnHandle) : TNWCastMode ;
  function setBroadcastMode(nServer : TNWConnHandle ; nMode : TNWCastMode) : boolean ;
  function getAllConnectInfo(nServer : TNWConnHandle ; var connInfo : TNWCCConnInfo) : boolean ;

             {internal function:}
  function getAllConnectRefInfo(connRef : TNWConnRef ; var connInfo : TNWCCConnInfo) : boolean ;
  function getUserList(nServer : TNWConnHandle; showNotLoggedIn : boolean) : TStringList ;
  function getConnUsageStats(nServer   : TNWConnHandle ; {Netware 2.2 only}
                             nConn     : TNWConnNumber ;
                             var stats : TNWConnectionStats) : boolean ;
  function broadcastToConn(nServer : TNWConnHandle ;
                           nConn   : TNWConnNumber ;
                           msgText: string) : boolean ;
  function broadcastToConsole(nServer : TNWConnHandle ;
                              msgText : string) : boolean ;

  {3.0 additions}                            
  function setShellErrorMode(errorMode : byte) : byte ;

  {3.22 additions}
  procedure setAutoNDS(value : boolean) ;  {if true, some nds functions are auto-called}

  {3.32 - moved from NWServer}
  function getUserConnList(nServer : TNWConnHandle ; 
                           cUserID : string ; 
                           var nConns : integer ;
                           pConnList  : pTConnList) : boolean  ;
  {4.5 changes}
  function getObjNumber(var nServer : TNWConnHandle ; {server by reference, since NDS can change}
                        objectName : string; nObjType : TObjType) : TObjID ;
  {4.6 additions}                      
  function mapRoot(cDrive: char; cPath: string) : boolean ;
          
  {*** component declaration ***}
  type
    TNWLib = class(TObject)
    public
      constructor create ; 
      function    getLastError : string ;
      procedure   resetLastError ;
    end;

var
  isNDS     : boolean ;
  hNWClx,
  hNWCalls  : THandle ;  {shared DLL Instance Handles}
  lasterror : TNWError ;

implementation

uses
  nwTools,
  nwServer,
  nwNDS ;

{$I nwlib.imp} {Novell API Imports}
{$I nwstd.imp}

var
  nPrimaryServerID : TNWConnHandle ;
  nError           : TNWError ;

{**************************** Component Plumbing *******************************}

constructor TNWLib.Create ;
  var
    tmp     : nptr    ;
  begin
    inherited create ;
    nPrimaryServerID := 0 ;
    lastError        := 0 ;
    hNWCalls         := 0 ;
    hNWClx           := 0 ;
    isNDS            := false ;
    hNWCalls := loadLibrary(nw_netwareCalls) ;
    if (hNWCalls = 0) then
      begin
        lastError := NWERROR_DLL_LOAD_FAILURE ;
        exit ;
      end;

    {$IFDEF Win32}
    hNWClx := loadLibrary(nw_netwareClx) ;
    if (hNWClx = 0) then
      begin
        lastError := NWERROR_DLL_LOAD_FAILURE ;
        exit ;
      end;
    {$ELSE}
      hNWClx := 0 ;
    {$ENDIF}
    @NWCallsInit := getProcAddress(hNWCalls,'NWCallsInit') ;
    if (@NWCallsInit = nil) then
      exit ;
    if (NWCallsInit(tmp,tmp)=0) then
      {set internal primaryServerID for faster default servers}
      nPrimaryServerID := getPrimaryServerID ;
  end;

function TNWLib.getLastError : string ;
  begin
    result := intToHex(lastError,sizeOf(TNWError)) ;
  end;

procedure TNWLib.resetLastError ;
  begin
    lastError := 0 ;
  end;


{ ********************* Public Functions ********************* }

procedure setAutoNDS(value : boolean) ;
  begin
    isNDS := value ;
  end;  

function getNWLibVersion : string ;
  begin
    result := '5.01 - 25 February 2000' ;
  end;

function nwInit: Boolean;
  {Ensure Netware Initialization}
  begin
    result := false ;
    {$IFDEF Win32}
      if (hNWCalls <> 0) and (hNWClx <> 0) then
        result := (getPrimaryServerID <> 0) ;
    {$ELSE}
      if (hNWCalls <> 0) then
        result := (getPrimaryServerID <> 0) ;
    {$ENDIF}
  end;

function getConnUsageStats(nServer   : TNWConnHandle ;
                           nConn     : TNWConnNumber ;
                           var stats : TNWConnectionStats) : boolean ;
  var
    nStats : CONN_USE ;
  begin
    result := false ;
    if (@NWGetConnectionUsageStats = nil) then {2.2 only}
      @NWGetConnectionUsageStats := getProcAddress(hNWCalls,'NWGetConnectionUsageStats') ;
    if (@NWGetConnectionUsageStats = nil) then
      exit ;
    nError := NWGetConnectionUsageStats(nServer,nConn,nStats) ;
    if (nError <> 0) then
      lastError := nError
    else
      begin
        result := true ;
        with stats do begin
          bytesRead    := nStats.bytesRead[0] ;
          bytesWritten := nStats.byteswritten[0] ;
          totalPackets := nStats.totalRequestPackets ;
        end;
      end;
  end;

function getUserList(nServer : TNWConnHandle;
                     showNotLoggedIn : boolean) : TStringList ;
  {get active user list}
  var
    nMaxConns,
    nLoop       : word     ;
    cpObjName   : TObjName ;
    objType     : TObjType ;
    objID       : TObjID   ;
    nDateTime   : TNWByte8 ;
    serverStats : TNWServerInfo ;
  begin
    result    := TStringList.create ;
    if (@NWGetConnectionInformation = nil) then
      @NWGetConnectionInformation := getProcAddress(hNWCalls,'NWGetConnectionInformation') ;
    if (@NWGetConnectionInformation = nil) then
      exit ;
    objID     := 0 ;
    objType   := 0 ;
    nMaxConns := 0 ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;

    if getServerStats(nServer,serverStats) then
      nMaxConns := serverStats.maxConns ;
    for nloop := 1 to nMaxConns do begin
      nError := NWGetConnectionInformation(nServer,
                                           nloop,
                                           @cpObjName,
                                           @objType,
                                           @objID,
                                           @nDateTime) ;

      if (nError = 0) and (showNotLoggedIn or (objID <> 0)) then
        begin
          if (strPas(cpObjName) = '') then
            result.addObject('Not Logged In',TObject(nloop))
          else
            result.addObject(strPas(cpObjName),TObject(nloop)) ;
        end;
    end;
  end;

function getPrimaryServerID : TNWConnHandle ;
  {Get Netware Server Primary Connetion ID (logged-in server)}
  var
    reserved : nuint;
    connRef  : TNWConnRef ;
    nServer  : TNWConnHandle ;
  begin
    if (nPrimaryServerID <> 0) then
      begin
        result := nPrimaryServerID ;
        exit ;
      end;
    result   := 0  ;
    reserved := 0  ;
    nError   := -1 ;
    if (@NWCCGetPrimConnRef = nil) then
      @NWCCGetPrimConnRef := getProcAddress(hNWClx,'NWCCGetPrimConnRef') ;
    if (@NWCCGetPrimConnRef <> nil) then
      begin
        if (NWCCGetPrimConnRef(@connRef) = 0) then
          begin
            @NWCCOpenConnByRef := getProcAddress(hNWClx,'NWCCOpenConnByRef') ;
            if (@NWCCOpenConnByRef <> nil) then
              nError := NWCCOpenConnByRef(connRef,
                                          NWCC_OPEN_LICENSED,
                                          reserved,
                                          @nServer);
          end;
      end
    else {pre 1/97}
      begin
        if (@NWGetPrimaryConnectionID = nil) then
          @NWGetPrimaryConnectionID := getProcAddress(hNWCalls,'NWGetPrimaryConnectionID') ;
        if (@NWGetPrimaryConnectionID <> nil) then
          nError := NWGetPrimaryConnectionID(nServer) ;
      end;
    if (nError <> 0) then
      lastError := nError
    else
      result := nServer ;
  end;

function whoAmI(nServer : TNWConnHandle) : string ;
  {Fetch User's Login Name}
  var
    nConn      : TNWConnNumber ;
    userID     : TObjName      ;
    nType      : TObjType      ;
    nObjID     : TObjID        ;
    nDateTime  : TNWByte8      ;
  begin
    result := '' ;
    if (@NWGetConnectionInformation = nil) then
      @NWGetConnectionInformation := getProcAddress(hNWCalls,'NWGetConnectionInformation') ;
    if (@NWGetConnectionInformation = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nConn := getConnectID(nServer) ;
    ntype := 1 ;
    nError := NWGetConnectionInformation(nServer,
                                         nConn,
                                         @userID,
                                         @nType,
                                         @nObjID,
                                         @nDateTime) ;
     if (nError <> 0) then
       lastError := nError
     else
       result := strPas(userID) ;
  end ;


function getObjName(nServer : TNWConnHandle ; nObjID : TObjID) : string ;
  { Return Name from TObjID }
  var
    nType  : TObjType   ;
    buffer : TObjName ;
  begin
    result := '';
    if (@NWGetObjectName = nil) then
      @NWGetObjectName := getProcAddress(hNWCalls,'NWGetObjectName') ;
    if (@NWGetObjectName = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWGetObjectName(nServer,
                              nObjID,
                              @buffer,
                              @nType) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := strPas(buffer);
    if (result = '') and isNDS then
      begin
        result := ndsGetObjName(nServer,nObjID) ;
        exit ;
      end;
  end;


function longSwap(nLong : Longint) : longInt ;
  { swap high-order bits }
  begin
    result := 0 ;
    if (@NWLongSwap = nil) then
      @NWLongSwap := getProcAddress(hNWCalls,'NWLongSwap') ;
    if (@NWLongSwap <> nil) then
      result := NWLongSwap(nLong) ;
  end;

function getConnectID(nServer: TNWConnHandle): TNWConnNumber;
  {Get User's Connection Number (1-1000+)}
  var
    nConn : TNWConnHandle ;
  begin
    result := 0 ;
    nError := -1 ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    if (@NWCCGetConnInfo = nil) then
      @NWCCGetConnInfo := getProcAddress(hNWClx,'NWCCGetConnInfo') ;
    if (@NWCCGetConnInfo <> nil) then
      nError := NWCCGetConnInfo(nServer,
                                NWCC_INFO_CONN_NUMBER,
                                sizeOf(nConn),
                                @nConn)
    else {pre 1/97}
      begin
        @NWGetConnectionNumber := getProcAddress(hNWCalls,'NWGetConnectionNumber') ;
        if (@NWGetConnectionNumber <> nil) then
          nError := NWGetConnectionNumber(nserver,nConn) ;
      end;
    if (nError <> 0) then
      lastError := nError
    else
      result := nConn ;
  end;


function getConnectInfo(nServer : TNWConnHandle ;
                        nConn   : TNWConnNumber ;
                        var connectInfo : TNWConnectInfo) : boolean ;
  { Extended User Login Information }
  var
    nloop          : byte          ;
    internetAddr   : NWINET_ADDR   ;
    nObjType       : TObjType      ;
    nObjID         : TObjID        ;
    nloginDateTime : TNWByte8      ;
    inConnInfo     : TNWCCConnInfo ;
    cpUserID       : TObjName      ;
    ctemp          : string        ;
  begin
    result := false ;
    if (@NWGetInetAddr = nil) then
      @NWGetInetAddr := getProcAddress(hNWCalls,'NWGetInetAddr') ;
    if (@NWGetInetAddr = nil) then
      exit ;
    if (@NWGetConnectionInformation = nil) then
      @NWGetConnectionInformation := getProcAddress(hNWCalls,'NWGetConnectionInformation') ;
    if (@NWGetConnectionInformation = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWGetINetAddr(nServer,nConn,internetAddr) ; {logged in?}
    if (nError <> 0) then
      lastError := nError
    else
      begin
        if getAllConnectInfo(nServer, inConnInfo) then {basic info}
          nError := NWGetConnectionInformation(nServer,  {extended info}
                                               nConn,
                                               @cpuserID,
                                               @nObjType,
                                               @nObjID,
                                               @nLoginDateTime) ;

        if (nError <> 0) then
          lastError := nError
        else
          begin
            result := true ;
            ctemp  := ''   ;
            {hex conversion net/node}
            for nloop := 0 to sizeOf(internetAddr.networkAddr)-1 do
              ctemp := ctemp + intToHex(internetAddr.networkAddr[nloop],2) ;
            for nloop := 0 to sizeOf(internetAddr.netNodeAddr)-1 do
              ctemp := ctemp + intToHex(internetAddr.netNodeAddr[nloop],2) ;
            with connectInfo do begin
              serverConnID  := nServer   ;
              loginDateTime := encodeDate((nLoginDateTime[0]+1900),
                                           nLoginDateTime[1],
                                           nLoginDateTime[2]) ;
              loginDateTime := loginDateTime +
                               encodeTime(nLoginDateTime[3],
                                          nLoginDateTime[4],
                                          nLoginDateTime[5],0) ;
              treeName      := strTran(strPas(inConnInfo.treeName),'_',' ') ;
              internet      := ctemp ;
              connectID     := nConn ;
              userID        := strPas(cpUserID) ;
              objType       := nObjType ;
              serverName    := strPas(inConnInfo.serverName) ;
              if (strLen(inConnInfo.serverName) < 1) then
                serverName  := treeName ;
            end;
          end;
      end;
  end;

function getObjNumber(var nServer  : TNWConnHandle ;
                      objectName  : string;
                      nObjType : TObjType) : TObjID ;
  { Get User's LongInt Object ID (Not Hex String) }
  var
    nObjID     : TObjID   ;
    cpUserID   : TObjName ;
    nObjServer : TNWConnHandle ;
  begin
    result := 0 ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpUserID,upperCase(objectName)) ;
    nObjID := 0 ;
    if isNDS then
      begin
        nObjServer := nServer ;  
        nObjID := ndsGetObjNumber(nObjServer,objectName) ; {nds can change server...don't want it}
        if (nObjID <> 0) then
          result := nObjID ;
      end;
    if (result = 0) then  
      begin
        {bindery style call}
        if (@NWGetObjectID = nil) then
          @NWGetObjectID := getProcAddress(hNWCalls,'NWGetObjectID') ;
        if (@NWGetObjectID <> nil) then
          nError := NWGetObjectID(nServer,
                                  @cpUserID,
                                  nObjType,
                                  @nObjID) ;
        if (nError <> 0) then
          lastError := nError
        else
          result := nObjID;
      end;
  end;


function getObjID(nServer : TNWConnHandle; objectName : string ;
                  nObjType : TObjType) : string ;
  {Get User's Object ID Hex Number}
  var
    nObjID    : TObjID   ;
    cpObjName : TObjName ;
    ctext     : string   ;
  begin
    if isNDS then
      begin
        result := ndsGetObjID(nServer,objectName) ;
        if (result <> '0') then
          exit ;
      end;
    result := '0' ;
    if (@NWGetObjectID = nil) then
      @NWGetObjectID := getProcAddress(hNWCalls,'NWGetObjectID') ;
    if (@NWGetObjectID = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;

    strPCopy(cpObjName,upperCase(objectName)) ;
    if (NWGetObjectID(nServer,
                      @cpObjName,
                      nObjType,
                      @nObjID) = 0) then
      begin
        nObjID := longSwap(nObjID) ;
        ctext := intToHex(nObjID,8) ;
        while (leftString(ctext,1) = '0') do begin
          ctext := copy(ctext,2,length(ctext)-1) ;
          if length(ctext) = 1 then
            break;
        end;
        result := ctext ;
      end;
  end;


function getObjType(nServer : TNWConnHandle ; cUserID: string) : TObjType ;
  { Return Object Type }
  begin
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    result := 0 ;
    if (getObjNumber(nServer,cUserID,nw_user) <> 0) then
      result := nw_user
    else if (getObjNumber(nServer,cUserID,nw_group) <> 0) then
      result := nw_group
    else if (getObjNumber(nServer,cUserID,nw_printQ) <> 0) then
      result := nw_printQ
    else if (getObjNumber(nServer,cUserID,nw_printServer) <> 0) then
      result := nw_printServer
    else if (getObjNumber(nServer,cUserID,nw_server) <> 0) then
      result := nw_server
    else if (getObjNumber(nServer,cUserID,nw_jobServer) <> 0) then
      result := nw_jobServer
    else if (getObjNumber(nServer,cUserID,nw_gateway) <> 0) then
      result := nw_gateway
    else if (getObjNumber(nServer,cUserID,nw_archiveQueue) <> 0) then
      result := nw_archiveQueue
    else if (getObjNumber(nServer,cUserID,nw_archiveServer) <> 0) then
      result := nw_archiveServer
    else if (getObjNumber(nServer,cUserID,nw_jobQueue) <> 0) then
      result := nw_jobQueue
    else if (getObjNumber(nServer,cUserID,nw_administration) <> 0) then
      result := nw_administration
    else if (getObjNumber(nServer,cUserID,nw_nasSNAServer) <> 0) then
      result := nw_nasSNAServer
    else if (getObjNumber(nServer,cUserID,nw_remoteBridge) <> 0) then
      result := nw_remoteBridge
    else if (getObjNumber(nServer,cUserID,nw_TCPIPGateway) <> 0) then
      result := nw_TCPIPGateway ;
  end;

function nwPassCheck(nServer : TNWConnHandle;
                     cUserID,
                     cPassword : string) : boolean ;
  { validate a user's password }
  var
    cpUserId   : TObjName    ;
    cpPassword : TNWPassword ;
    nreturn    : integer     ;
    nObjType   : TObjType    ;
  begin
    result := false ;
    if (@NWVerifyObjectPassword = nil) then
      @NWVerifyObjectPassword := getProcAddress(hNWCalls,'NWVerifyObjectPassword');
    if (@NWVerifyObjectPassword = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nObjType := getObjType(nServer,cUserID) ;
    strPCopy(cpUserID,upperCase(allTrim(cUserID))) ;
    strPCopy(cpPassword,upperCase(allTrim(cPassword))) ;
    nreturn := NWVerifyObjectPassword(nServer,
                                      @cpUserID,
                                      nObjType,
                                      @cpPassword) ;
    if (nreturn = 0) then
      result := true
    else
      lastError := nreturn ;
  end;

function isConsoleOperator(nServer : TNWConnHandle) : boolean ;
  { Check Console Rights }
  begin
    result := false ;
    if (@NWCheckConsolePrivileges = nil) then
      @NWCheckConsolePrivileges := getProcAddress(hNWCalls,'NWCheckConsolePrivileges') ;
    if (@NWCheckConsolePrivileges = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWCheckConsolePrivileges(nServer) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function isNWManager(nServer : TNWConnHandle) : boolean  ;
  { Is a Netware Supervisor/Manager }
  begin
    result := false ;
    if (@NWIsManager = nil) then
      @NWIsManager := getProcAddress(hNWCalls,'NWIsManager') ;
    if (@NWIsManager = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWIsManager(nServer) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function getConnectedServerList : TStringList ;
  { Return a list of connected servers }
  var
    nLoop    : integer   ;
    nConns   : TNWNumber ;
    connlist : TConnList ;
    connRef,
    iterator : nuint32   ;
    connInfo : TNWCCConnInfo ;
  begin
    result := TStringList.create ;
    if (@NWCCScanConnRefs = nil) then
      @NWCCScanConnRefs := getProcAddress(hNWClx,'NWCCScanConnRefs') ;
    if (@NWCCScanConnRefs = nil) then
      begin {pre 1/97}
        @NWGetConnectionList := getProcAddress(hNWCalls,'NWGetConnectionList') ;
        if (@NWGetConnectionList = nil) then
          exit ;
        if (NWGetConnectionList(0,@connList,high(connList),@nConns)=0) then
          for nloop := 0 to (nConns-1) do
            result.addObject(getServerName(connList[nloop]),TObject(connList[nloop]));
      end
    else
      begin
        if (@NWCCGetAllConnRefInfo = nil) then
          @NWCCGetAllConnRefInfo := getProcAddress(hNWClx,'NWCCGetAllConnRefInfo') ;
        if (@NWCCGetAllConnRefInfo = nil) then
          exit ;
        iterator := 0 ;
        while true do begin
          nError := NWCCScanConnRefs(@iterator,@connRef);
          if (nError <> 0) then
            begin
              lastError := nError ;
              break ;
            end;
          nError := NWCCGetAllConnRefInfo(connRef,NWCC_INFO_VERSION,@connInfo) ;
          if (nError <> 0) then
            lastError := nError
          else
            with connInfo do
              result.addObject(strPas(serverName),TObject(connNum)) ;
        end;
      end;
  end;

function sList : TStringList ;
  { Return a StringList of Available Servers}
  begin
    result := getBinderyList(getPrimaryServerID,nw_server) ;
  end;


function map(cDrive: char; cPath: string) : boolean ;
  { Create base Netware Drive Map }
  var
    ntemp      : word    ;
    dirHandle  : TNWDirHandle  ;
    nServer    : TNWConnHandle ;
    cpPath,
    cpRelative : TNWPath ;
    parsedPath : TNWParsedPath ;
  begin
    result := false ;
    if (@NWParseNetwarePath = nil) then
      @NWParseNetwarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetwarePath = nil) then
      exit ;
    if (@NWSetDriveBase = nil) then
      @NWSetDriveBase := getProcAddress(hNWCalls,'NWSetDriveBase') ;
    if (@NWSetDriveBase = nil) then
      exit ;
    ntemp := ord(upCase(cDrive))-64 ;
    strPCopy(cpPath,upperCase(cPath)) ;
    if (NWParseNetWarePath(@cpPath,
                           nServer,
                           dirHandle,
                           @cpRelative)=0) then
      begin
        if getPathInfo(cPath,parsedPath) then
          begin
            mapDelete(cDrive) ;
            strPCopy(cpPath,parsedPath.volName) ;         
            nerror := NWSetDriveBase(ntemp,nServer,
                                     dirHandle,@cpPath,0) ;
            if (nerror <> 0) then
              lastError := nerror
            else
              result := true ;                         
          end;
      end;  
  end;
  

function mapRoot(cDrive: char; cPath: string) : boolean ;
  { Create Root Netware Drive Mapping }
  var
    ntemp      : word    ;
    cpPath,
    cpRelative : TNWPath ;
    dirHandle  : TNWDirHandle  ;
    nServer    : TNWConnHandle ;
    parsedPath : TNWParsedPath ;
  begin
    result := false ;
    if (@NWParseNetwarePath = nil) then
      @NWParseNetwarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetwarePath = nil) then
      exit ;
    if (@NWSetDriveBase = nil) then
      @NWSetDriveBase := getProcAddress(hNWCalls,'NWSetDriveBase') ;
    if (@NWSetDriveBase = nil) then
      exit ;
    ntemp := ord(upCase(cDrive))-64 ;
    strPCopy(cpPath,cPath) ;
    if (NWParseNetWarePath(@cpPath,
                           nServer,
                           dirHandle,
                           @cpRelative)=0) then
      begin
        {delete old mapping, if exists}
        mapDelete(cDrive) ;
        {separate individual elements}
        if getPathInfo(cPath,parsedPath) then
          begin
            strPCopy(cpPath,
                     parsedPath.volName + ':\' +
                     parsedPath.dirPath) ;
            nerror := NWSetDriveBase(ntemp,nServer,
                                     dirHandle,@cpPath,0) ;
            if (nerror <> 0) then
              lastError := nerror
            else
              result := true ;                          
          end;
      end;
  end;
  


function mapDelete(cDrive : string) : Boolean ;
  { Remove Netware Drive Mapping }
  var
    ntemp : byte ;
  begin
    result := false ;
    if (@NWDeleteDriveBase = nil) then
      @NWDeleteDriveBase := getProcAddress(hNWCalls,'NWDeleteDriveBase') ;
    if (@NWDeleteDriveBase = nil) then
      exit ;
    ntemp := ord(UpCase(cDrive[1]))-64 ; {a=65, b=66, etc.}
    nError := NWDeleteDriveBase(ntemp,0) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function getServerName(nServer : TNWConnHandle) : string ;
  var
    buffer : TObjName ;
  begin
    result := '' ;
    nError := -1 ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    if (@NWCCGetConnInfo = nil) then
      @NWCCGetConnInfo := getProcAddress(hNWClx,'NWCCGetConnInfo') ;
    if (@NWCCGetConnInfo <> nil) then
      nError := NWCCGetConnInfo(nServer,
                                NWCC_INFO_SERVER_NAME,
                                sizeOf(buffer),
                                @buffer)
    else
      begin {pre 1/97}
        if (@NWGetFileServerName = nil) then
          @NWGetFileServerName := getProcAddress(hNWCalls,'NWGetFileServerName') ;
        if (@NWGetFileServerName <> nil) then
          nError := NWGetFileServerName(nServer,@buffer) ;
      end;
    if (nError <> 0) then
      begin
        lastError := nError ;
        result := upperCase(word_local) ;
      end
    else
      result := strPas(buffer) ;
  end;

function getServerDate(nServer : TNWConnHandle) : TDateTime ;
  var
    nwdate : TNWDateString ;
  begin
    result := date ;
    if (@NWGetFileServerDateAndTime = nil) then
      @NWGetFileServerDateAndTime := getProcAddress(hNWCalls,'NWGetFileServerDateAndTime') ;
    if (@NWGetFileServerDateAndTime = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWGetFileServerDateAndTime(nServer,@nwDate) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := encodeDate(ord(nwDate[0])+1900,
                           ord(nwDate[1]),
                           ord(nwDate[2])) ;
  end;

function getServerTime(nServer : TNWConnHandle) : TDateTime ;
  var
    nwDate : TNWDateString ;
  begin
    result := time ;
    if (@NWGetFileServerDateAndTime = nil) then
      @NWGetFileServerDateAndTime := getProcAddress(hNWCalls,'NWGetFileServerDateAndTime') ;
    if (@NWGetFileServerDateAndTime = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWGetFileServerDateAndTime(nServer, @nwDate) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := encodeTime(ord(nwDate[3]),
                           ord(nwDate[4]),
                           ord(nwDate[5]),
                           0) ;
  end;


function maxConns : TNWNumber ;
  { max # of ws connections (usually 8)}
  begin
    if (@NWGetMaximumConnections = nil) then
      @NWGetMaximumConnections := getProcAddress(hNWCalls,'NWGetMaximumConnections') ;
    if (@NWGetMaximumConnections = nil) then
      exit ;
    NWGetMaximumConnections(@result) ;
  end;

function setPreferredServer(servername : string ;
                            var nServer : TNWConnHandle) : boolean ;
  var
    nConn    : TNWConnHandle ;
    cpServer : TObjName ;
  begin
    result := false ;
    strPCopy(cpServer,serverName) ;
    if (@NWCCSetPrefServerName = nil) then
      @NWCCSetPrefServerName := getProcAddress(hNWClx,'NWCCSetPrefServerName') ;
    if (@NWCCSetPrefServerName <> nil) then
      nError := NWCCSetPrefServerName(@cpServer)
    else {pre 1/97}
      begin
        if (@NWGetConnectionHandle = nil) then
          @NWGetConnectionHandle := getProcAddress(hNWCalls,'NWGetConnectionHandle') ;
        if (@NWGetConnectionHandle = nil) then
          exit ;
        if (@NWSetPrimaryConnectionID = nil) then
          @NWSetPrimaryConnectionID := getProcAddress(hNWCalls,'NWSetPrimaryConnectionID') ;
        if (@NWSetPrimaryConnectionID = nil) then
          exit ;
        nError := NWGetConnectionHandle(@cpServer,0,nConn,nil) ;
        if (nError = 0) then
          nError := NWSetPrimaryConnectionID(nConn)  ;
      end;
    if (nError <> 0) then
      lastError := nError
    else
      begin
        nServer := getServerHandle(serverName) ;
        if (nServer <> 0) then
          isNDS := ndsIsDSServer(nServer) ;
        nPrimaryServerID := nServer ;  {reset internal default server ID}
        result := true ;
      end;
  end;

function isLoggedIn(nServer : TNWConnHandle ; cUserID : string) : boolean ;
  { is User Logged In? }
  var
    cpName  : TObjName;
    nConns  : word ;
    connList: TConnList ;
  begin
    result := false ;
    if (@NWGetObjectConnectionNumbers = nil) then
      @NWGetObjectConnectionNumbers := getProcAddress(hNWCalls,'NWGetObjectConnectionNumbers') ;
    if (@NWGetObjectConnectionNumbers = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpName,cUserID) ;
    nError := NWGetObjectConnectionNumbers(nServer,
                                     @cpName,
                                     nw_user,
                                     @nConns,
                                     @connList,
                                     50) ;
    if (nError <> 0) then
      lastError := nError
    else if (nConns > 0) then
      result := true ;
  end;

function getMemberList(nServer    : TNWConnHandle;
                       cGroupName : string ;
                       lSelf      : boolean) : TStringList ;
  { Get Members of a Group }
  var
    nloop,
    nmore,
    nindex,
    nflag      : byte       ;
    ntemp      : word       ;
    nObjID     : TObjID     ;
    cpProperty,
    cpGroup    : TObjName   ;
    ctemp,
    cObjId,
    cMe        : string     ;
    cpBuffer   : TRetBuff   ;
  begin
    result := TStringList.create  ;
    if (@NWReadPropertyValue = nil) then
      @NWReadPropertyValue := getProcAddress(hNWCalls,'NWReadPropertyValue') ;
    if (@NWReadPropertyValue = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nindex := 1 ;
    nflag  := type_set ;
    cMe    := whoAmI(nServer) ;
    strPCopy(cpProperty,'GROUP_MEMBERS');
    strPCopy(cpGroup,cGroupName)        ;
    while (NWReadPropertyValue(nServer,
                               @cpGroup,
                               nw_group,
                               @cpProperty,
                               nindex,
                               @cpBuffer,
                               @nmore,
                               @nflag) = 0) do begin
        inc(nindex) ;
        { each elment in returned buffer is object ID }
        for nloop := 0 to 31 do begin  { 128 byte buffer, 4 bytes each }
          ntemp := nloop*4 ;
          ctemp := '$' + intToHex(ord(cpBuffer[ntemp+0]),2) +
                         intToHex(ord(cpBuffer[ntemp+1]),2) +
                         intToHex(ord(cpBuffer[ntemp+2]),2) +
                         intToHex(ord(cpBuffer[ntemp+3]),2) ;
          nobjID := longSwap(longInt(strToInt(ctemp))) ;
          cObjID := getObjName(nServer,nObjID) ;
          if (length(cObjID) > 0) and
             (lSelf or (cObjID <> cMe)) then
            result.add(cObjID) ;
        end;
    end;
  end;

function secureEquiv(nServer : TNWConnHandle ; cUserID,cEquiv : string) : boolean ;
  { User is Equivalent }
  var
    nloop      : byte       ;
    nMore,
    nindex     : TNWSegment ;
    flag       : TNWFlag    ;
    ntemp,
    nObjID     : TObjID     ;
    cpProperty,
    cpUserID   : TObjName   ;
    ctemp,
    cObjId     : string     ;
    cpBuffer   : TRetBuff   ;
  begin
    result  := false ;
    if (@NWReadPropertyValue = nil) then
      @NWReadPropertyValue := getProcAddress(hNWCalls,'NWReadPropertyValue') ;
    if (@NWReadPropertyValue = nil) then
      exit ;
    cuserID := uppercase(cUserID) ;
    cEquiv  := uppercase(cEquiv) ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    if (cUserID = cEquiv) then
      result := true
    else
      begin
        nIndex := 1     ;
        flag  := type_set ;
        Result := False ;
        strPCopy(cpProperty,'SECURITY_EQUALS');
        strPCopy(cpUserID,cUserID)        ;
        while (NWReadPropertyValue(nServer,
                                   @cpUserID,
                                   nw_user,
                                   @cpProperty,
                                   nIndex,
                                   @cpBuffer,
                                   @nMore,
                                   @flag) = 0) do begin
          { each elment in returned buffer is object ID }
          for nloop := 0 to 31 do begin  { 128 byte buffer, 4 bytes each }
            ntemp := nloop*4 ;
            ctemp := '$' + intToHex(ord(cpBuffer[ntemp+0]),2) +
                           intToHex(ord(cpBuffer[ntemp+1]),2) +
                           intToHex(ord(cpBuffer[ntemp+2]),2) +
                           intToHex(ord(cpBuffer[ntemp+3]),2) ;
            nobjID := longSwap(longInt(strToInt(ctemp))) ;
            cObjID := getObjName(nServer,nObjID) ;
            if (length(cObjID) > 0) and
               (cObjID = cEquiv) then
              begin
                result := True ;
                break ;
              end;
          end;
          { next segment }
          inc(nIndex) ;
          if result then
            break;
        end;
      end;
  end;


function getMyGroups(nServer : TNWConnHandle ;
                     cUserID : string) : TStringList ;
  { Returns a user's group names - must have suff. rights }
  var
    nloop      : byte       ;
    flag      : TNWFlag    ;
    ntemp      : word       ;
    more,
    nindex     : TNWSegment ;
    nObjID     : TObjID     ;
    cpProperty,
    cpGroup    : TObjName   ;
    cpBuffer   : TRetBuff   ;
    ctemp,
    cObjId     : string ;
  begin
    if isNDS then
      begin
        result := ndsGetMyGroups('', cUserID) ;
        exit ;
      end;  
    result := TStringList.create  ;
    if (@NWReadPropertyValue = nil) then
      @NWReadPropertyValue := getProcAddress(hNWCalls,'NWReadPropertyValue') ;
    if (@NWReadPropertyValue = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nIndex := 1 ;
    flag  := type_set ;
    strPCopy(cpProperty,'GROUPS_I''M_IN');
    strPCopy(cpGroup,cUserID)        ;
    while (NWReadPropertyValue(nServer,
                               @cpGroup,
                               nw_user,
                               @cpProperty,
                               nIndex,
                               @cpBuffer,
                               @more,
                               @flag) = 0) do begin
      inc(nIndex) ;
      { each elment in returned buffer is object ID }
      for nloop := 0 to 31 do begin  { 128 byte buffer, 4 bytes each }
        ntemp := nloop*4 ;
        ctemp := '$' + intToHex(ord(cpBuffer[ntemp+0]),2) +
                       intToHex(ord(cpBuffer[ntemp+1]),2) +
                       intToHex(ord(cpBuffer[ntemp+2]),2) +
                       intToHex(ord(cpBuffer[ntemp+3]),2) ;
        nobjID := longSwap(longInt(strToInt(ctemp))) ;
        cObjID := getObjName(nServer,nObjID) ;
        if (length(cObjID) > 0) then
          begin
            try
              result.add(cObjID) ;
            except
              on exception do
                break ;
            end;
          end;
      end;
    end;
  end;


function getBroadcastMode(nServer : TNWConnHandle) : TNWCastMode ;
  { Return Broadcast mode }
  var
    nmode : TNWCastMode ;
  begin
    result := 0 ;
    if (@NWCCGetConnInfo = nil) then
      @NWCCGetConnInfo := getProcAddress(hNWCalls,'NWCCGetConnInfo') ;
    if (@NWCCGetConnInfo = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWCCGetConnInfo(nServer,
                              NWCC_INFO_BCAST_STATE,
                              sizeOf(nMode),
                              @nMode) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := nMode ;
  end;

function setBroadcastMode(nServer : TNWConnHandle ; nMode : TNWCastMode) : boolean ;
  begin
    result := false ;
    if (@NWSetBroadcastMode = nil) then
      @NWSetBroadcastMode := getProcAddress(hNWCalls,'NWSetBroadcastMode') ;
    if (@NWSetBroadcastMode = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWSetBroadcastMode(nServer, nMode) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function castOn(nServer : TNWConnHandle) : boolean ;
  { Enable Broadcasts }
  begin
    result := false ;
    if (@NWEnableBroadcasts = nil) then
      @NWEnableBroadcasts := getProcAddress(hNWCalls,'NWEnableBroadcasts') ;
    if (@NWEnableBroadcasts = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWEnableBroadcasts(nServer) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function castOff(nServer : TNWConnHandle) : boolean ;
  { Disable Broadcasts}
  begin
    result := false ;
    if (@NWDisableBroadcasts = nil) then
      @NWDisableBroadcasts := getProcAddress(hNWCalls,'NWDisableBroadcasts') ;
    if (@NWDisableBroadcasts = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWDisableBroadcasts(nServer) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function getBroadcastMessage(nServer : TNWConnHandle) : string ;
  {fetch a stored broadcast message (use cast mode nw_castPoll) }
  var
    cpMessage : array[0..254] of char ;
  begin
    result := '' ;
    if (@NWGetBroadcastMessage = nil) then
      @NWGetBroadcastMessage := getProcAddress(hNWCalls,'NWGetBroadcastMessage') ;
    if (@NWGetBroadcastMessage = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWGetBroadcastMessage(nServer, @cpMessage) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := strPas(cpMessage) ;
  end;

function broadcastToConsole(nServer : TNWConnHandle ;
                            msgText : string) : boolean ;
  var
    cpMessage : TRetbuff ;
  begin
    result := false ;
    if (@NWBroadcastToConsole = nil) then
      @NWBroadcastToConsole := getProcAddress(hNWCalls,'NWBroadcastToConsole') ;
    if (@NWBroadcastToConsole = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpMessage,msgText) ;
    nError := NWBroadcastToConsole(nServer,@cpMessage) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function broadcastToConn(nServer : TNWConnHandle ;
                         nConn   : TNWConnNumber ;
                         msgText : string) : boolean ;
  { Send a line message to single connection }
  var
    sendToList    : TSendToList ;
    resultList    : TResultList ;
    cpMessageText : TRetBuff    ;
  begin
    result := false ;
    if (@NWSendBroadcastMessage = nil) then
      @NWSendBroadcastMessage := getProcAddress(hNWCalls,'NWSendBroadcastMessage') ;
    if (@NWSendBroadcastMessage = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpMessageText,msgText) ;
    sendToList[0] := nConn ;
    nError := NWSendBroadcastMessage(nServer,
                           @cpMessageText,
                           1,
                           @sendToList,
                           @resultList) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := (resultList[0] = 0) ;
  end;


function sendLineMessage(nServer : TNWConnHandle ; cUserID, cMessageText : string) : boolean ;
  { Send a line message to user on all connections}
  var
    ntemp         : integer     ;
    nConns        : word        ;
    cpUserID      : TObjName    ;
    sendToList    : TSendToList ;
    resultList    : TResultList ;
    connList      : TConnList   ;
    nObjServer    : TNWConnHandle ;
    cpMessageText : TRetBuff      ;
  begin
    result := false ;
    if (@NWGetObjectConnectionNumbers = nil) then
      @NWGetObjectConnectionNumbers := getProcAddress(hNWCalls,'NWGetObjectConnectionNumbers') ;
    if (@NWGetObjectConnectionNumbers = nil) then
      exit ;
    if (@NWSendBroadcastMessage = nil) then
      @NWSendBroadcastMessage := getProcAddress(hNWCalls,'NWSendBroadcastMessage') ;
    if (@NWSendBroadcastMessage = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nObjServer := nServer ;  
    strPCopy(cpUserID,upperCase(cUserID)) ;
    strPCopy(cpMessageText,cMessageText) ;
    NWGetObjectConnectionNumbers(nServer,
                                 @cpUserID,
                                 nw_user,
                                 @nConns,
                                 @connList,
                                 high(TConnList));
    {on connectionList retrieve failure, try 4.x NDS style}
    if (nConns < 1) or (nConns > 20000) then {if more than 20,000 conns, likely a failure}
      begin
        if (@NWGetConnListFromObject = nil) then
          @NWGetConnListFromObject := getProcAddress(hNWCalls,'NWGetConnListFromObject') ;
        if (@NWGetConnListFromObject <> nil) then
          NWGetConnListFromObject(nServer,
                                  getObjNumber(nObjServer,cUserID,nw_user),
                                  1,
                                  @nConns,
                                  @connList)
      end;
      
    if (nConns > 0) then
      begin
        for ntemp := 0 to (nConns-1) do
          sendToList[ntemp] := word(connlist[ntemp]) ;
        NWSendBroadcastMessage(nServer,
                               @cpMessageText,
                               nConns,
                               @sendToList,
                               @resultList) ;
        for ntemp := 0 to high(ResultList) do begin
          if resultList[ntemp] = 0 then
            begin
              result := true ;
              break;
            end;
        end;
      end;
  end;


function getBinderyList(nServer : TNWConnHandle;
                        searchType : TObjType) : TStringList ;
  { Query Bindery - Get All Objects of XX Type }
  var
    QID        : TObjID    ;
    QType      : TObjType  ;
    propflag,
    objflag,
    secmask    : TNWFlags  ;
    qName,
    searchText : TObjName  ;
  begin
    result := TStringList.create  ;
    if (@NWScanObject = nil) then
      @NWScanObject := getProcAddress(hNWCalls,'NWScanObject') ;
    if (@NWScanObject = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(searchText,'*') ;
    QID := -1                ;
    while true do begin
      if (NWScanObject(nServer,
                      @searchText,
                      searchType,
                      @QID,
                      @QName,
                      @QType,
                      @propflag,
                      @objflag,
                      @secmask)=0) then
        begin
          try
            result.addObject(QName,TObject(QID)) ;
          except
            on exception do
              break ;
          end;
        end
      else
        break;
    end;
  end;

function fullName(nServer : TNWConnHandle ; cUserID : string ) : string ;
  { Retreive UserID's Full Bindery Name }
  var
    flag      : TNWFlag    ;
    more      : TNWSegment ;
    cpProperty,
    cpUserID   : TObjName   ;
    cpBuffer   : TRetBuff   ;
  begin
    if isNDS and ndsGetStringAttr('',cUserID,dsaFullName,result) then
      exit ;
    result := uppercase(alltrim(cUserID)) ;
    if (@NWReadPropertyValue = nil) then
      @NWReadPropertyValue := getProcAddress(hNWCalls,'NWReadPropertyValue') ;
    if (@NWReadPropertyValue = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpUserID,result) ;
    strPCopy(cpProperty,'IDENTIFICATION');
    if (NWReadPropertyValue(nServer,
                            @cpUserID,
                            getObjType(nServer,cUserID),
                            @cpProperty,
                            1,
                            @cpBuffer,
                            @more,
                            @flag)=0) then
        result := strPas(cpBuffer) ;
  end;

function isMember(nServer : TNWConnHandle ; cUserID,cGroup : string) : boolean ;
  { Is UserID a Group Member? }
  var
    cpGroup,
    cpUserID,
    cpPropName : TObjName ;
  begin
    result := false ;
    if (@NWIsObjectInSet = nil) then
      @NWIsObjectInSet := getProcAddress(hNWCalls,'NWIsObjectInSet') ;
    if (@NWIsObjectInSet = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpGroup,upperCase(cGroup)) ;
    strPCopy(cpUserID,upperCase(cUserID)) ;
    strPCopy(cpPropName,'GROUP_MEMBERS') ;
    result := (NWIsObjectInSet(nServer,
                               @cpGroup,
                               nw_group,
                               @cpPropName,
                               @cpUserID,
                               nw_user)=0) ;
  end;


function isInList(nServer : TNWConnHandle ; clist : string ) : boolean ;
  { User is Found in List (maybe inside a group) }
  var
    cuser  : string  ;
    nloop  : integer ;
    ncount : integer ;
  begin
    result := false ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    ncount := charCount(clist,',')+1 ;
    for nloop := 1 to ncount do begin
      cuser := upperCase(allTrim(strExtract(clist,',',nloop))) ;
      case getObjType(nServer,cuser) of
        nw_user  :
          begin
            if (cuser = whoAmi(nserver)) then
              begin
                result := true ;
                break
              end;
          end;
        nw_group :
          begin
            if isMember(nServer,whoAmI(nServer),cUser) then
              begin
                result := True ;
                break ;
              end;
          end;
      end;
    end;
  end;

function getFirstNetDrive : char ;
  { Return first non-local drive letter }
  var
    ntemp : word ;
  begin
    result := #0;
    if (@NWGetFirstDrive = nil) then
      @NWGetFirstDrive := getProcAddress(hNWCalls,'NWGetFirstDrive') ;
    if (@NWGetFirstDrive = nil) then
      exit ;
    ntemp := 0 ;
    if (NWGetFirstDrive(@ntemp) = 0) then
      result := chr(ntemp+64);
  end;


function getNextNetDrive  : char ;
  { Returns Next Free Net Drive Letter }
  var
    nstart,
    nloop        : byte     ;
    status       : word     ;
    connHandle   : TNWConnHandle ;
    relativePath,
    rootPath,
    fullPath     : TNWPath ;
  begin
    result := #0 ;
    if (@NWGetDriveStatus = nil) then
      @NWGetDriveStatus := getProcAddress(hNWCalls,'NWGetDriveStatus') ;
    if (@NWGetDriveStatus = nil) then
      exit ;
    nstart := (ord(getFirstNetDrive) - 64) ;
    for nloop := nstart to 26 do begin
      if (NWGetDriveStatus(nloop,
                           nw_format_netware,
                           @status,
                           connHandle,
                           @rootPath,
                           @relativePath,
                           @fullPath)=0) and
         (status=0) then
        begin
          result := chr(nloop+64) ;
          break;
        end;
    end;
  end;


function getMapInfo(drive : char ; var mapInfo  : TNWMapInfo) : boolean ;
  var
    nDrive       : byte ;
    status       : word ;
    connHandle   : TNWConnHandle ;
    rootPath,
    relativePath,
    fullNWPath   : TNWPath ;
  begin
    result := false ;
    if (@NWGetDriveStatus = nil) then
      @NWGetDriveStatus := getProcAddress(hNWCalls,'NWGetDriveStatus') ;
    if (@NWGetDriveStatus = nil) then
      exit ;
    nDrive := ord(upCase(drive)) - 64 ;
    nError := NWGetDriveStatus(nDrive,
                               NW_FORMAT_NETWARE,
                               @status,
                               connHandle,
                               @rootPath,
                               @relativePath,
                               @fullNWPath) ;
    if (nError <> 0) then
      lastError := nError
    else
      begin
        if (status < $2000) then {not mapped or local}
          exit ;
        result := true ;
        with mapInfo do begin
          nServer     := connHandle ;
          serverName  := getServerName(connHandle) ;
          fullPath    := strPas(fullNWPath) ;
          driveStatus := status ;
        end;
      end;
  end;


function mapShow : TStringList ;
  { Parse-able TStringList of Map Table }
  var
    firstNet,
    nloop        : byte ;
    status       : word ;
    connHandle   : TNWConnHandle ;
    rootPath,
    relativePath,
    fullPath     : TObjName ;
  begin
    result := TStringList.create ;
    if (@NWGetDriveStatus = nil) then
      @NWGetDriveStatus := getProcAddress(hNWCalls,'NWGetDriveStatus') ;
    if (@NWGetDriveStatus = nil) then
      exit ;
    firstNet := (ord(getFirstNetDrive) - 64) ;
    for nloop := 1 to 26 do begin
      if (NWGetDriveStatus(nloop,
                           NW_FORMAT_NETWARE,
                           @status,
                           connHandle,
                           @rootPath,
                           @relativePath,
                           @fullPath)=0) then
         begin
           if (nloop >= firstNet) and
              (status = NW_FREE_DRIVE) then   { free network }
             begin
               try
                 result.add(chr(nloop+64) + ':  [Free Net Device]' )  ;
               except
                 on exception do
                   break ;
               end;
             end
           else if ((status and NW_CDROM_DRIVE) > 0) then
             begin
               try
                 result.add(chr(nloop+64) + ':  [CD-ROM Drive]')  ;
               except
                 on exception do
                   break ;
               end;
             end
           else if ((status and NW_LOCAL_FREE_DRIVE) >0) or
                   ((status and NW_LOCAL_DRIVE) >0) then
             begin
               try
                 result.add(chr(nloop+64) + ':  [Local Device]')      ;
               except
                 on exception do
                   break ;
               end;
             end
           else if ((status and NW_NETWORK_DRIVE) >0) then
             begin
               try
                 result.add(chr(nloop+64) + ':  ' +
                            getServerName(connHandle) +
                            '//' + strPas(rootPath)) ;
               except
                 on exception do
                   break ;
               end;
             end
           else {case}
             result.add(chr(nloop+64) + ':  [Unknown]')           ;
         end;
    end;
  end;

function convertLocalFileHandle(localFileHandle : TNWFileHandle) : TNWFileHandle ;
  {turn local file handle into netware file handle}
  var
    handleSize : word ;
    nServer    : TNWConnHandle ;
  begin
    result     := 0 ;
    if (@NWConvertFileHandle = nil) then
      @NWConvertFileHandle := getProcAddress(hNWCalls,'NWConvertFileHandle') ;
    if (@NWConvertFileHandle = nil) then
      exit ;
    nServer    := 0 ;
    handleSize := sizeOf(handleSize) ;
    nError := NWConvertFileHandle(localFileHandle,
                                  handleSize,
                                  @result,
                                  nServer) ;
    if (nError <> 0) then
      begin
        lastError := nError ;
        exit ;
      end;
  end;
  

function _nwCopy(inFile, outFile : string) : boolean ;
  { internal server to server copy function }
  var
    ntemp,
    nbytes       : longint ;
    inHandle,
    outHandle    : longint ;
    pathInfo     : TNWParsedPath ;
    parseInfo    : TNWPathInfo ;
  begin
    result       := false ;
    if (@NWFileServerFileCopy = nil) then
      @NWFileServerFileCopy := getProcAddress(hNWCalls,'NWFileServerFileCopy') ;
    if (@NWFileServerFileCopy = nil) then
      exit ;
    if (not getPathInfo(inFile, pathInfo)) then
      exit ;
    if (not parseNetwarePath(pathInfo.nServer,
                             inFile,
                             parseInfo)) then
      exit ;
    inHandle := fileOpen(infile,fmShareDenyNone) ;  
    if (inHandle < 1) then
      exit ;
    outHandle := fileCreate(outFile) ;
    if (outHandle < 1) then
      begin
        fileClose(inHandle) ;
        exit ;  
      end;  
    try  
      nbytes := 0 ;
      ntemp  := fileSeek(word(inHandle),0,2) ;  { eof }
      fileSeek(inHandle,0,0) ;
      nError := NWFileServerFileCopy(inHandle,
                                     outHandle,
                                     0,
                                     0,
                                     ntemp,
                                     @nbytes) ;
   
    finally
      fileClose(inHandle) ;
      fileClose(outHandle) ;
    end;                              
    if (nError <> 0) then
      lastError := nError
    else
      result := (nbytes >= ntemp) ;
  end;

function nCopy(infile, outfile : string) : boolean ;
  { The NCopy Shell Interface }
  var
    resetROFlag : boolean  ;
    numRead,
    numWritten  : integer  ;
    inHandle,
    outHandle   : THandle  ;
    ncursor     : TCursor  ;
    filein,
    fileout     : file     ;
    dateStamp   : longint  ;
    buf         : array[0..2047] of char;
  begin
    result        := false ;
    resetROFlag   := false ;
    ncursor       := screen.cursor ;
    screen.cursor := crHourglass ;
    dateStamp     := 0 ;
    if (not fileExists(infile)) then
      exit ;
    try
      {****** Server-to-Server Copy ******}
      if _nwCopy(inFile,outFile) and
         fileExists(outFile) then
        begin
          result := true ;
          exit ;
        end;
        
      {****** Local Copy *******}  
      if ((fileGetAttr(infile) and faReadOnly) > 0) then
        begin
          resetROFlag := true ;
          fileSetAttr(inFile,(fileGetAttr(infile) - faReadOnly)) ;
        end;
      {$IFDEF Win32}
      deleteFile(PChar(outfile)) ;
      {$ELSE}
      deleteFile(outFile) ;
      {$ENDIF}

      {get input file's DOS Date Stamp}
      inHandle := fileOpen(inFile,fmShareDenyNone) ;
      if (inHandle > 0) then
        try
          dateStamp := fileGetDate(inHandle) ;
        finally
          fileClose(inHandle) ;
        end;

      assignFile(fileIn,infile);
      assignFile(fileOut,outfile) ;
      reset(fileIn,1) ;
      rewrite(fileOut,1) ;
      repeat
        blockRead(fileIn, buf, sizeOf(buf), numRead);
        if (numRead > 0) then
          blockWrite(fileOut, buf, numRead, numWritten);
      until (numRead = 0) or (numWritten <> numRead);
      closeFile(fileIn) ;
      closeFile(fileOut) ;
      result := fileExists(outfile) ;

      { retain time stamp on output filename }
      if result and (dateStamp > 0) then
        begin
          outhandle := fileOpen(outFile,fmOpenReadWrite) ;
          if (outHandle > 0) then
            try
              fileSetDate(outHandle,dateStamp) ;
            finally
              fileClose(outHandle);
            end;
        end;
    finally
      {restore infile's RO flag if we reset it}
      if resetROFlag then
        fileSetAttr(inFile,fileGetAttr(inFile) or faReadOnly) ;
      screen.cursor := ncursor ;
    end;
  end;


function getPathInfo(cPath : string; var parsedPath : TNWParsedPath) : boolean ;
  var
    nServer    : TNWConnHandle ;
    cpServer   : TObjName ;
    cpPath,
    cpVol,
    cpDir      : TNWPath ;
  begin
    {init return struct}
    result := false ;
    if (@NWParsePath = nil) then
      @NWParsePath := getProcAddress(hNWCalls,'NWParsePath') ;
    if (@NWParsePath = nil) then
      exit ;
    parsedPath.nServer    := 0 ;
    parsedPath.serverName := '' ;
    parsedPath.volName    := '' ;
    parsedPath.dirPath    := '' ;
    strPCopy(cpPath,upperCase(cPath)) ;
    nError := NWParsePath(@cpPath,
                          @cpServer,
                          nServer,
                          @cpVol,
                          @cpDir) ;
    if (nError <> 0) then
      lastError := nError
    else
      begin
        result := true ;
        parsedPath.nServer    := nServer ;
        parsedPath.serverName := strPas(cpServer) ;
        parsedPath.volName    := strPas(cpVol) ;
        parsedPath.dirPath    := strPas(cpDir) ;
      end;
  end;

function parseNetwarePath(nServer : TNWConnHandle ;
                          cPath   : string;
                          var pathInfo : TNWPathInfo ) : boolean ;
  var
    nwDirHandle : TNWDirHandle ;
    cpVolName   : TNWVolName   ;
    ctemp       : string       ;
    cpPath,
    cpRelative  : TNWPath      ;
  begin
    result := false ;
    if (@NWParseNetwarePath = nil) then
      @NWParseNetwarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetwarePath = nil) then
      exit ;
    if (@NWGetVolumeNumber = nil) then
      @NWGetVolumeNumber := getProcAddress(hNWCalls,'NWGetVolumeNumber') ;
    if (@NWGetVolumeNumber = nil) then
      exit ;
    if (@NWGetVolumeName = nil) then
      @NWGetVolumeName := getProcAddress(hNWCalls,'NWGetVolumeName') ;
    if (@NWGetVolumeName = nil) then
      exit ;
    nwDirHandle := 0 ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpPath,upperCase(cPath)) ;
    if (NWParseNetWarePath(@cpPath,
                           nServer,
                           nwDirHandle,
                           @cpRelative)=0) then
      begin
        ctemp := strPas(cpRelative) ;
        ctemp := strExtract(ctemp,':',1) ;
        strPCopy(cpVolName,ctemp) ;
        NWGetVolumeNumber(nServer,
                          @cpVolName,
                          @pathInfo.volumeID) ;
        NWGetVolumeName(nServer,
                        pathInfo.volumeID,
                        @cpVolName) ;
        pathInfo.volumeName := strPas(cpVolName) + ':' ;
        result := true ;
        pathInfo.nServer      := nServer ;
        pathInfo.cServer      := getServerName(nServer) ;
        pathInfo.dirHandle    := nwDirHandle ;
        pathInfo.relativePath := strPas(cpRelative) ;
        pathInfo.pathOnly     := setPath(strExtract(ctemp,':',2)) ;
      end;
  end;


function getAllConnectRefInfo(connRef : TNWConnRef;
                              var connInfo : TNWCCConnInfo) : boolean ;
  {get all connection info by conn. reference}
  begin
    result := false ;
    if (@NWCCGetAllConnRefInfo = nil) then
      @NWCCGetAllConnRefInfo := getProcAddress(hNWClx,'NWCCGetAllConnRefInfo') ;
    if (@NWCCGetAllConnRefInfo = nil) then
      exit ;
    nError := NWCCGetAllConnRefInfo(connRef,
                                    NWCC_INFO_VERSION,
                                    @connInfo) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function getAllConnectInfo(nServer : TNWConnHandle ;
                           var connInfo : TNWCCConnInfo) : boolean ;
  {get all connection info by conn handle}
  var
    oldConnInfo : CONNECT_INFO ;
  begin
    result := false ;
    nError := -1 ;
    if (@NWCCGetAllConnInfo = nil) then
      @NWCCGetAllConnInfo := getProcAddress(hNWClx,'NWCCGetAllConnInfo') ;
    if (@NWCCGetAllConnInfo <> nil) then
      nError := NWCCGetAllConnInfo(nServer,
                                   NWCC_INFO_VERSION,
                                   @connInfo)
    else
      begin  {pre 1/97}
        if (@NWGetConnectionStatus = nil) then
          @NWGetConnectionStatus := getProcAddress(hNWCalls,'NWGetConnectionStatus') ;
        if (@NWGetConnectionStatus = nil) then
          exit ;
        nError := NWGetConnectionStatus(nServer,oldConnInfo,nw_user) ;
        if (nError = 0) then
          with oldConnInfo do begin
            connInfo.connNum := connID ;
            strPCopy(connInfo.serverName,strPas(serverName)) ;
          end;
      end;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function getConnOpenFiles(nServer : TNWConnHandle ;
                          nConn   : TNWConnNumber) : TStringList ;
  {get a connection's open files}
  var
    sequence,
    iteration   : TNWSequence   ;
    nDateTime   : TNWDateTime   ;
    augmentFlag : word          ;
    rightsMask  : TNWRightsMask ;
    ownerID     : TObjID        ;
    cpSearchPath,
    cpDirName   : TNWPath       ;
    openCtrl    : OPEN_FILE_CONN_CTRL;
    openFile    : OPEN_FILE_CONN     ;
  begin
    result := TStringList.create ;
    if (@NWScanOpenFilesByConn2 = nil) then
      @NWScanOpenFilesByConn2 := getProcAddress(hNWCalls,'NWScanOpenFilesByConn2') ;
    if (@NWScanOpenFilesByConn2 = nil) then
      exit ;
    if (@NWIntScanDirectoryInformation2 = nil) then
      @NWIntScanDirectoryInformation2 := getProcAddress(hNWCalls,'NWIntScanDirectoryInformation2') ;
    if (@NWIntScanDirectoryInformation2 = nil) then
      exit ;
    iteration := 0 ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    while true do begin
      nError := NWScanOpenFilesByConn2(nServer,
                                       nConn,
                                       @iteration,
                                       openCtrl,
                                       openFile) ;
      if (nError <> 0) then
        begin
          lastError := nError ;
          break ;
        end
      else
        begin
          augmentFlag := 0 ;
          strCopy(cpSearchPath,openFile.fileName);
          if (NWIntScanDirectoryInformation2(nServer,
                                             openFile.dirEntry,
                                             @cpSearchPath,
                                             @sequence,
                                             @cpDirName,
                                             @nDateTime,
                                             @ownerID,
                                             @rightsMask,
                                             augmentFlag) = 0) then
            result.addObject(strPas(cpDirName) + '\' +
                             strPas(openFile.fileName),
                             TObject(openFile.lockType))
          else
            result.addObject(strPas(openFile.fileName),
                             TObject(openFile.lockType)) ;
        end;
    end;
  end;

function getOpenFileUsers(nServer : TNWConnHandle ;
                          cFile   : string) : TStringList ;
  {get users with a file open (console req)}
  var
    nObjID     : TObjID ;
    nObjType   : TObjType ;
    nDateTime  : TNWDateTime ;
    dirHandle  : TNWDirHandle  ;
    iteration  : TNWSequence ;
    cpRelative,
    cpFile     : TNWPath ;
    buffer     : TRetBuff ;
    fileUse    : CONN_USING_FILE ;
    fileUsed   : CONNS_USING_FILE ;
  begin
    result    := TStringList.create ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWScanConnectionsUsingFile = nil) then
      @NWScanConnectionsUsingFile := getProcAddress(hNWCalls,'NWScanConnectionsUsingFile') ;
    if (@NWScanConnectionsUsingFile = nil) then
      exit ;
    if (@NWGetConnectionInformation = nil) then
      @NWGetConnectionInformation := getProcAddress(hNWCalls,'NWGetConnectionInformation') ;
    if (@NWGetConnectionInformation = nil) then
      exit ;
    iteration := 0 ;
    nObjType  := nw_user ;
    strPCopy(cpFile,allTrim(cFile)) ;
    
    {$IFDef win32}
    // convert to short file name 
    if (pos(' ',extractFileName(cFile)) > 0) then
      begin
        getShortPathName(@cpFile,@cpRelative,sizeOf(cpRelative))  ;
        strPCopy(cpFile,strPas(cpRelative)) ;
      end;  
    {$ENDIF}
    
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    if (NWParseNetWarePath(@cpFile,
                           nServer,
                           dirHandle,
                           @cpRelative)=0) then
      begin
        while true do begin
          nError := NWScanConnectionsUsingFile(nServer,
                                               0,
                                               @cpRelative,
                                               @iteration,
                                               fileUse,
                                               fileUsed) ;
          if (nError <> 0) then
            begin
              lastError := nError ;
              break ;
            end
          else if (fileUsed.connCount < 1) then
            begin
              result.addObject('[0]  ' + word_none,TObject(0)) ;
              break ;
            end
          else if (NWGetConnectionInformation(
                           nServer,
                           fileUse.connNumber,
                           @buffer,
                           @nObjType,
                           @nObjID,
                           @nDateTime) = 0) then
            result.addObject('[' + intToStr(fileUse.connNumber) + ']   ' +
                       strPas(buffer),TObject(nObjID))
                       //3/99 change - put objectID instead of connNumber in objects array
                       //  strPas(buffer),TObject(fileUse.connNumber))
          else
            result.addObject('[0]  ' + word_unknown,TObject(0)) ;
        end;
      end;
  end;

function setShellErrorMode(errorMode : byte) : byte ;
  var
    prevMode : byte ;
  begin
    result := 0 ;
    if (@NWSetNetWareErrorMode = nil) then
      @NWSetNetWareErrorMode := getProcAddress(hNWCalls,'NWSetNetWareErrorMode') ;
    if (@NWSetNetWareErrorMode = nil) then
      exit ;
    nError := NWSetNetWareErrorMode(errorMode, @prevMode) ;
    if (nError <> 0) then
      lastError := nError 
    else
      result := prevMode ;  
  end;

function getUserConnList(nServer    : TNWConnHandle ;
                         cUserID    : string        ;
                         var nConns : integer       ;
                         pConnList  : pTConnList) : boolean ;
  { Returns any user's logged-in connection ID list }
  var
    ntemp      : integer  ;
    objType    : TObjType ;
    cpUserID   : TObjName ;
    nObjServer : TNWConnHandle ;
  begin
    ntemp  := 0 ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nObjServer := nServer ;  
    fillChar(pConnList^,sizeOf(pConnList^),0) ;
    strPCopy(cpUserID,cUserID) ;
    objType := getObjType(nServer,cUserID) ;
    if (objType < nw_user) then
      objType := nw_user ;
    {first try 4.x style Connection List (supports NDS names)}                                       
    if (@NWGetConnListFromObject = nil) then
      @NWGetConnListFromObject := getProcAddress(hNWCalls,'NWGetConnListFromObject') ;
    if (@NWGetConnListFromObject <> nil) then
      NWGetConnListFromObject(nServer,
                              getObjNumber(nObjServer,cUserID,objType),
                              1,
                              @ntemp,
                              pConnList) ;

    if (ntemp < 1) then {old-style connectionList} 
      begin
        if (@NWGetObjectConnectionNumbers = nil) then
          @NWGetObjectConnectionNumbers := getProcAddress(hNWCalls,'NWGetObjectConnectionNumbers') ;
        if (@NWGetObjectConnectionNumbers <> nil) then
          NWGetObjectConnectionNumbers(nServer, {always success except on 2.x}
                                       @cpUserID,
                                       objType,
                                       @ntemp,
                                       pConnList,
                                       high(TConnList));
      end;                             
    nConns := ntemp ;
    result := (nConns > 0) ;
  end;
  

initialization
  begin
    exit ;
    {$IFNDEF Registered}
    okBox('NWLib / ' + {$IFDEF Win32}'32'{$ELSE}'16'{$ENDIF} + ' by Devont Software Inc.  Evaluation Version;;'+
          'Register Today to Remove this Annoying Banner;;' +
          'Devont Software Inc. Fax: (281)754-4203;' +
          'Internet: sales@devont.com;' +
          'World Wide Web: www.devont.com') ;
    {$ENDIF}
  end;

end.
