{ *************** Capture & Print Related Functions ****************

  Developed with Novell Inc.'s Client SDK for C SDK

  Copyright 1995-1999 by Devont Software Inc. and Jim Tyson.  All
  rights reserved.

}

unit NWprint;

interface

uses
  SysUtils,
  winTypes,
  winProcs,
  Classes ;

{$IFDEF Ver110}
  {$ObjExportAll On}
{$ENDIF}

  {$I nwlib.inc}

  type
    {$I nwTypes.inc}

    { Capture Flags }
    type
      TNWCaptureFlags = record
        copies      : longint ;
        tabSize     : longint ;
        formType    : longint ;
        timeOut     : longint ;
        description : string     ;
        formName    : string     ;
        banner      : string     ;
        qname       : string     ;
        nServer     : TNWConnHandle ;
        autoEndCap  : boolean       ;
        formFeed    : boolean       ;
        jobNotify   : boolean       ;
        userID      : string        ;
      end;

    type
      TNWQueueJobCtrlFlags = record
        auto_start      : boolean ;
        entry_restart   : boolean ;
        entry_open      : boolean ;
        user_hold       : boolean ;
        operator_hold   : boolean ;
      end;

    type
      TNWQueueJobInfo = record
        nServer         : TNWConnHandle ;
        cQueue          : string        ;
        jobID           : TNWQueueJobID ;
        ownerName       : string        ;
        serverName      : string        ;
        queueServerName : string        ;
        jobFileName     : string        ;
        jobDescription  : string        ;
        formName        : string        ;
        formFeed        : boolean       ;
        workstationID   : longint       ;
        entryDateTime   : TDateTime     ;
        execDateTime    : TDateTime     ;
        jobType,
        jobPosition     : word          ;
        jobFlags        : TNWQueueJobCtrlFlags ;
    end;


    { Interface Function Declarations }
    function endCap(nPort : byte) : boolean ;
    function isCaptured(nPort : byte) : boolean ;
    function setBannerUserName(cBanner : string) : boolean ;
    function getBannerUserName : string ;
    function getMaxPrinters : byte ;
    function capture(nServer: TNWConnHandle ; cQueue: string ;
                             nPort : TNWLpt ;
                             var CaptureFlags : TNWCaptureFlags) : boolean ;
    function getCaptureFlags(nPort : TNWLpt;
                             var captureFlags : TNWCaptureFlags) : boolean  ;
    function setCaptureFlags(nServer : TNWConnHandle;
                             nPort : TNWLpt ;
                             var captureFlags : TNWCaptureFlags) : boolean;
    function getQueueUsers(nServer : TNWConnHandle;
                           cQueue : string) : TStringList ;
    function getQueueOperators(nServer : TNWConnHandle ;
                               cQueue : string) : TStringList ;
    function getQueueServers(nServer : TNWConnHandle ;
                             cQueue : string) : TStringList ;
    {1.5}
    function getQueueJobList(nServer     : TNWConnHandle ;
                             cQueue      : string ;
                             var jobList : TStringList ) : boolean ;
    function getQueueJobNumbers(nServer        : TNWConnHandle ;
                                cQueue         : string ;
                                var jobNumbers : TNWQueueJobList) : boolean ;
    function getQueueJobInfo(var jobInfo : TNWQueueJobInfo) : boolean ;
    function deleteQueueJob(jobInfo : TNWQueueJobInfo) : boolean ;
    function setQueueJobPosition(var jobInfo : TNWQueueJobInfo ;
                                  newPos      : word) : boolean ;
    function setQueueJobInfo(jobInfo : TNWQueueJobInfo) : boolean ;

    {2.0}
    function isQueueOperator(nServer : TNWConnHandle ;
                             cQueue,
                             cUser  : string) : boolean ;
    {2.02}
    function NPrint(nServer    : TNWConnHandle ;
                    queueName,
                    inFileName : string) : boolean ;

    {3.5 additions}                
    function cancelCapture(nPort : TNWLpt) : boolean ;
    
    {$IFNDEF Win32}
    function captureToFile(nServer  : TNWConnHandle ;  
                           nPort    : TNWLpt ; 
                           fileName : string;
                           var captureFlags : TNWCaptureFlags) : boolean ;
    {$ENDIF}       
    function NPrintEx(nServer    : TNWConnHandle ;
                    queueName,
                    inFileName : string;
                    captureFlags : TNWCaptureFlags) : boolean ; {added captureFlags parameter}
                    
  type
    TNWPrint = class(TObject)
    public
      function getLastError : string ;
      procedure resetLastError ;
  end;

var
  lasterror : TNWError ;

implementation

uses
  nwTools,
  nwLib,
  nwServer ;

{$I nwPrint.imp}   {Novell API Imports}
{$I nwstd.imp}

var
  nError : TNWError ;

{ ****************** Component functions ******************* }
function TNWPrint.getLastError : string ;
  begin
    result := intToHex(lastError,sizeOf(TNWError)) ;
  end;

procedure TNWPrint.resetLastError ;
  begin
    lastError := 0 ;
  end;


{ ******************** Public Functions ******************* }

function setQueueJobInfo(jobInfo : TNWQueueJobInfo) : boolean ;
  var
    queueID      : TObjID ;
    jobStruct    : NW_QUEUE_JOB ;
    jobCtrlFlags : TNWQueueJobFlags  ;
    nHour,
    nMin,
    nSec,
    nMSec,
    nYear,
    nMonth,
    nDay         : word ;
    nObjServer   : TNWConnHandle ;
  begin
    result := false ;
    if (@NWChangeQueueJobEntry2 = nil) then
      @NWChangeQueueJobEntry2 := getProcAddress(hNWCalls,'NWChangeQueueJobEntry2') ;
    if (@NWChangeQueueJobEntry2 = nil) then
      exit;
    if (@NWReadQueueJobEntry2 = nil) then
      @NWReadQueueJobEntry2 := getProcAddress(hNWCalls,'NWReadQueueJobEntry2') ;
    if (@NWReadQueueJobEntry2 = nil) then
      exit;
    queueID      := getObjNumber(jobInfo.nServer,jobInfo.cQueue,nw_printQ);
    jobCtrlFlags := 0 ;
    {read existing values}
    if (NWReadQueueJobEntry2(jobInfo.nServer,
                             queueID,
                             jobInfo.jobID,
                             jobStruct) <> 0) then
      exit ;
    {set job execution date/time ( $xffff = immediately) }
    if (jobInfo.execDateTime < 1) then
      fillChar(jobStruct.targetExecutionTime,sizeOf(jobStruct.targetExecutionTime),$FF)
    else
      begin
        decodeDate(jobInfo.execDateTime,nYear,nMonth,nDay) ;
        decodeTime(jobInfo.execDateTime,nHour,nMin,nSec,nMsec) ;
        with jobStruct do begin
          targetExecutionTime[0] := nYear ;
          targetExecutionTime[1] := nMonth ;
          targetExecutionTime[2] := nDay ;
          targetExecutionTime[3] := nHour;
          targetExecutionTime[4] := nMin ;
          targetExecutionTime[5] := nSec ;
        end;
      end;
    {set job control flags}
    with jobInfo.jobFlags do begin
      if auto_start then
        jobCtrlFlags := (jobCtrlFlags or qms_auto_start) ;
      if entry_restart  then
        jobCtrlFlags := (jobCtrlFlags or qms_entry_restart) ;
      if entry_open then
        jobCtrlFlags := (jobCtrlFlags or qms_entry_open) ;
      if user_hold  then
        jobCtrlFlags := (jobCtrlFlags or qms_user_hold) ;
      if operator_hold then
        jobCtrlFlags := (jobCtrlFlags or qms_operator_hold) ;
    end;
    {replace remainder of jobStruct values}
    with jobStruct do begin
      jobPosition         := jobInfo.jobPosition ;
      jobControlFlags     := jobCtrlFlags ;
      jobType             := jobInfo.jobType ;
      strPCopy(jobDescription,jobInfo.jobDescription) ;
      strPCopy(formName,jobInfo.formName) ;
    end;
    nError  := NWChangeQueueJobEntry2(jobInfo.nServer,
                                      queueID,
                                      jobStruct) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function setQueueJobPosition(var jobInfo : TNWQueueJobInfo ;
                              newPos      : word) : boolean ;
  var
    queueID : TObjID ;
  begin
    result := false ;
    if (@NWChangeQueueJobPosition2 = nil) then
      @NWChangeQueueJobPosition2 := getProcAddress(hNWCalls,'NWChangeQueueJobPosition2') ;
    if (@NWChangeQueueJobPosition2 = nil) then
      exit;
    queueID := getObjNumber(jobInfo.nServer,jobInfo.cQueue,nw_printQ) ;
    nError  := NWChangeQueueJobPosition2(jobInfo.nServer,
                                         queueID,
                                         jobInfo.jobID,
                                         newPos) ;
    if (nError <> 0) then
      lastError := nError
    else
      begin
        result := true ;
        jobInfo.jobPosition := newPos ;
      end;
  end;

function deleteQueueJob(jobInfo : TNWQueueJobInfo) : boolean ;
  var
    queueID : TObjID ;
  begin
    result  := false ;
    if (@NWRemoveJobFromQueue2 = nil) then
      @NWRemoveJobFromQueue2 := getProcAddress(hNWCalls,'NWRemoveJobFromQueue2') ;
    if (@NWRemoveJobFromQueue2 = nil) then
      exit ;
    queueID := getObjNumber(jobInfo.nServer,jobInfo.cQueue,nw_printQ) ;
    nError  := NWRemoveJobFromQueue2(jobInfo.nServer,queueID,jobInfo.jobID) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;


function getQueueJobInfo(var jobInfo : TNWQueueJobInfo) : boolean ;
  var
    jobStruct : NW_QUEUE_JOB ;
  begin
    result := false ;
    fillChar(jobStruct,sizeOf(NW_QUEUE_JOB),0) ;
    if (@NWReadQueueJobEntry2 = nil) then
      @NWReadQueueJobEntry2 := getProcAddress(hNWCalls,'NWReadQueueJobEntry2') ;
    if (@NWReadQueueJobEntry2 = nil) then
      exit ;
    nError := NWReadQueueJobEntry2(jobInfo.nServer,
                                   getObjNumber(jobInfo.nServer,jobInfo.cQueue,nw_printQ),
                                   jobInfo.jobID,
                                   jobStruct) ;
    if (nError <> 0) then
      lastError := nError
    else
      begin
        result := true ;
        with jobInfo do begin
          ownerName       := getObjName(jobInfo.nServer,jobStruct.clientID) ;
          serverName      := getServerName(jobInfo.nServer) ;
          queueServerName := getObjName(jobInfo.nServer,jobStruct.servicingServerID) ;
          if (jobStruct.jobFileName[0] = #10) then
            jobFileName     := copy(strPas(jobStruct.jobFileName),2,255) {1st = chr(10)? y? } 
          else
            jobFileName     := strPas(jobStruct.jobFileName) ;
          jobDescription  := strPas(jobStruct.jobDescription) ;
          formName        := strPas(jobStruct.formName) ;
          formFeed        := ((jobStruct.printControlFlags and JOB_NO_FORM_FEED) = 0) ;
          workstationID   := jobStruct.clientID ;
          jobPosition     := jobStruct.jobPosition ;
          jobType         := jobStruct.jobType ;
          {date/time placed in queue}
          try
            entryDateTime   := encodeDate(jobStruct.jobEntryTime[0],
                                          jobStruct.jobEntryTime[1],
                                          jobStruct.jobEntryTime[2]) ;
            entryDateTime   := entryDateTime +
                               encodeTime(jobStruct.jobEntryTime[3],
                                          jobStruct.jobEntryTime[4],
                                          jobStruct.jobEntryTime[5],
                                          0);
          except
            on E:Exception do
              entryDateTime := now ; {date convert error - ignore}
          end;
          {hold-job-till date ($ffffffff = now) }
          if (jobStruct.targetExecutionTime[0] <> $FF) then
            try
              execDateTime := encodeDate(jobStruct.targetExecutionTime[0],
                                         jobStruct.targetExecutionTime[1],
                                         jobStruct.targetExecutionTime[2]) ;
              execDateTime := execDateTime +
                              encodeTime(jobStruct.targetExecutionTime[3],
                                         jobStruct.targetExecutionTime[4],
                                         jobStruct.targetExecutionTime[5],
                                         0)
            except
              on E:Exception do
                execDateTime := now ; {date convert error - ignore}
            end
          else
            execDateTime := now ;
          {convert job control flags}
          with jobFlags do begin
            auto_start      := ((jobStruct.jobControlFlags and qms_auto_start) > 0) ;
            entry_restart   := ((jobStruct.jobControlFlags and qms_entry_restart) > 0) ;
            entry_open      := ((jobStruct.jobControlFlags and qms_entry_open) > 0) ;
            user_hold       := ((jobStruct.jobControlFlags and qms_user_hold) > 0);
            operator_hold   := ((jobStruct.jobControlFlags and qms_operator_hold) > 0) ;
          end;
        end;
      end;
  end;

function getQueueJobList(nServer     : TNWConnHandle ;
                         cQueue      : string ;
                         var jobList : TStringList ) : boolean ;
   var
     nloop      : integer ;
     queueID    : TObjID ;
     jobNumbers : TNWQueueJobList ;
     nObjServer : TNWConnHandle ;
     jobStruct  : NW_QUEUE_JOB ;
     cJobStatus : string ;
   begin
     result := false ;
     if (@NWReadQueueJobEntry2 = nil) then
       @NWReadQueueJobEntry2 := getProcAddress(hNWCalls,'NWReadQueueJobEntry2') ;
     if (@NWReadQueueJobEntry2 = nil) then
       exit ;
     if (nServer = 0) then
       nServer := getPrimaryServerID ;
     nObjServer := nServer ;  
     jobList.clear ;
     if getQueueJobNumbers(nServer,cQueue,jobNumbers) then
       begin
         result  := true ;
         queueID := getObjNumber(nObjServer,cQueue,nw_printQ) ;
         for nloop := 0 to (high(jobNumbers)-1) do begin
           if (NWReadQueueJobEntry2(nServer,
                                    queueID,
                                    jobNumbers[nloop],
                                    jobStruct) = 0) then
             begin
               cJobStatus := 'Waiting' ;
               if ((jobStruct.jobControlFlags and qms_user_hold) > 0) then
                 cJobStatus := 'Paused'
               else if ((jobStruct.jobControlFlags and qms_operator_hold) > 0) then
                 cJobStatus := 'Operator' 
               else if ((jobStruct.jobControlFlags and qms_entry_adding) > 0) or 
                       ((jobStruct.jobControlFlags and qms_entry_open) > 0) then
                 cJobStatus := 'Adding' 
               else if (jobStruct.servicingServerID <> 0) then
                 cJobStatus := 'Active' ;
                 
               {insert joblist strings - 32 bit has LBS_USETABSTOPS in std listbox ctrl}  
               {$IFDEF Win32}  
                 jobList.addObject(
                   getObjName(nServer,jobStruct.clientID) + '[' +
                         intToStr(jobStruct.clientStation) + ']' + char(VK_TAB) +
                   cJobStatus + char(VK_TAB) + 
                   strPas(jobStruct.jobDescription),
                   TObject(jobStruct.jobNumber) ) ;
               {$ELSE}
                 jobList.addObject(
                   padR( getObjName(nServer,jobStruct.clientID) + '[' +
                         intToStr(jobStruct.clientStation) + ']'    ,20,' ') +
                   padR(cJobStatus,11,' ') +
                   padR(strPas(jobStruct.jobDescription),55,' '),
                   TObject(jobStruct.jobNumber) ) ;
               {$ENDIF}  
             end;
         end;
       end;
   end;

function getQueueJobNumbers(nServer        : TNWConnHandle ;
                            cQueue         : string ;
                            var jobNumbers : TNWQueueJobList) : boolean ;
  var
    nloop        : integer ;
    queueID      : TObjID ;
    jobListReply : TNWQueueJobListReply ;
    nObjServer   : TNWConnHandle ;
  begin
    result  := false ;
    if (@NWGetQueueJobList2 = nil) then
      @NWGetQueueJobList2 := getProcAddress(hNWCalls,'NWGetQueueJobList2') ;
    if (@NWGetQueueJobList2 = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nObjServer := nServer ;  
    queueID := getObjNumber(nObjServer,cQueue,nw_printQ) ;
    if (queueID = 0) then
      exit ;
    {$IFDEF Win32}
      fillChar(jobNumbers,sizeOf(jobNumbers),-1) ;
    {$ELSE}
      fillChar(jobNumbers,sizeOf(jobNumbers),0) ;
    {$ENDIF}
    {read jobs 1-125 (api call gets max 125 at time)}
    nerror := NWGetQueueJobList2(nServer,
                                 queueID,
                                 1, {starting job number}
                                 jobListReply) ;
    if (nError <> 0) then
      lastError := nError
    else
      begin
        result := true ;
        
        {put job numbers into return record}
        for nloop := 0 to (jobListReply.replyQueueJobNumbers-1) do 
          jobNumbers[nloop] := jobListReply.jobNumberList[nloop] ;
      
        {read jobs 126-250}
        if (jobListReply.totalQueueJobs > 125) and
           (NWGetQueueJobList2(nServer,
                               queueID,
                               126,
                               jobListReply)=0) then
          begin
            for nloop := 0 to (jobListReply.replyQueueJobNumbers-1) do 
              jobNumbers[125+nloop] := jobListReply.jobNumberList[nloop] ;
          end;
      end;
  end;

function getPropList(nServer : TNWConnHandle;   {internal function}
                     cObject : string ;
                     cProperty : string ) : TStringList ;
  { Get Queue Users }
  var
    nloop      : byte       ;
    flag       : TNWFlag    ;
    ntemp      : word       ;
    nindex,
    more       : TNWSegment ;
    nObjID     : TObjID     ;
    buffer     : TRetBuff   ;
    cpProperty,
    cpGroup    : TObjName ;
    ctemp,
    cObjId     : string ;
  begin
    result := TStringList.create  ;
    if (@NWReadPropertyValue = nil) then
      @NWReadPropertyValue := getProcAddress(hNWCalls,'NWReadPropertyValue') ;
    if (@NWReadPropertyValue = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nIndex := 1 ;
    flag   := 2 ;  { set type }
    strPCopy(cpProperty,uppercase(allTrim(cProperty)));
    strPCopy(cpGroup    ,upperCase(allTrim(cObject))) ;
    while (NWReadPropertyValue(nServer,
                               @cpGroup,
                               nw_printq,
                               @cpProperty,
                               nIndex,
                               @buffer,
                               @more,
                               @flag) = 0) do begin
        inc(nIndex) ;
        { each elment in returned buffer is object ID }
        for nloop := 0 to 31 do begin  { 128 byte buffer, 4 bytes each }
          ntemp := nloop*4 ;
          ctemp := '$' + intToHex(ord(buffer[ntemp+0]),2) +
                         intToHex(ord(buffer[ntemp+1]),2) +
                         intToHex(ord(buffer[ntemp+2]),2) +
                         intToHex(ord(buffer[ntemp+3]),2) ;
          nobjID := longSwap(longInt(strToInt(ctemp))) ;
          cObjID := getObjName(nServer,nObjID) ;
          if (length(cObjID) > 0) then
             result.add(cObjID) ;
        end;
    end;
  end;

function getQueueUsers(nServer : TNWConnHandle ;
                       cQueue : string) : TStringList ;
  begin
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    result := getPropList(nServer,cQueue,'Q_USERS') ;
  end;

function getQueueOperators(nServer : TNWConnHandle ;
                           cQueue : string) : TStringList ;
  begin
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    result := getPropList(nServer,cQueue,'Q_OPERATORS') ;
  end;

function getQueueServers(nServer : TNWConnHandle ;
                           cQueue : string) : TStringList ;
  begin
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    result := getPropList(nServer,cQueue,'Q_SERVERS') ;
  end;

function getCaptureFlags(nPort : TNWLpt;
                         var captureFlags : TNWCaptureFlags) : boolean;
  var
    NWCapFlags1 : TNWCapFlags1 ;
    NWCapFlags2 : TNWCapFlags2 ;
  begin
    result := false ;
    {provide defaults}
    with captureFlags do begin
      copies     := 1 ;
      tabSize    := 0 ;
      formType   := 0 ;
      timeOut    := 10 ;
      banner     := '' ;
      formName   := '' ;
      qName      := '' ;
      formFeed   := false ;
      jobNotify  := false ;
      autoEndCap := true ;
      nServer    := getPrimaryServerID ;
    end;
    
    if (@NWGetCaptureFlags = nil) then
      @NWGetCaptureFlags := getProcAddress(hNWCalls,'NWGetCaptureFlags') ;
    if (@NWGetCaptureFlags = nil) then
      exit ;
    fillChar(NWCapFlags1,sizeOf(TNWCapFlags1),#0) ;
    nError := NWGetCaptureFlags(nPort,
                                NWCapFlags1,
                                NWCapFlags2) ;
    if (nError <> 0) then
      lastError := nError
    else
      begin
        result := true ;
        { Fill Our CaptureFlags Interface Structure }
        with captureFlags do begin
          copies     := NWCapFlags1.numCopies ;
          tabSize    := NWCapFlags1.tabSize ;
          formType   := word(NWCapFlags1.formType) ;
          formName   := strPas(NWCapFlags1.formName) ;
          banner     := strPas(NWCapFlags1.bannerText) ;
          nServer    := NWCapFlags2.ConnID ;
          qname      := getObjName(NWCapFlags2.connID,NWCapFlags2.QueueID) ;
          timeOut    := word(NWCapFlags1.flushCaptureTimeout) ;
          {autoEndCap := (flushCaptureOnClose = 0) ;}
          autoEndCap := ((NWCapFlags1.printFlags and qms_printFlags_release) > 0) ;
          jobNotify  := ((NWCapFlags1.printFlags and qms_printFlags_notify) > 0) ;
          formFeed   := ((NWCapFlags1.printFlags and qms_printFlags_suppress) = 0) ;
        end;
      end;
  end;


function setCaptureFlags(nServer : TNWConnHandle;
                         nPort : TNWLpt ;
                         var captureFlags : TNWCaptureFlags) : boolean;
  var
    nerror    : integer ;
    capFlags1 : TNWCapFlags1 ;
  begin
    result := false ;
    if (@NWSetCaptureFlags = nil) then
      @NWSetCaptureFlags := getProcAddress(hNWCalls,'NWSetCaptureFlags') ;
    if (@NWSetCaptureFlags = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    { Convert Our Structure back to Novell's}
    with capFlags1 do begin
      tabSize             := captureFlags.tabSize  ;
      numCopies           := captureFlags.copies   ;
      formType            := captureFlags.formType ;
      flushCaptureTimeout := captureFlags.timeOut  ;
      flushCaptureOnClose := 0 ;
      reserved2           := 0 ;
      maxLines            := 0 ;
      jobControlFlags     := 0 ;
      printFlags          := 0 ;
      { Set PrintFlag Bits }
      if captureFlags.autoEndCap then
        printFlags := (printFlags or $04) ;
      if (not captureFlags.formfeed) then
        printFlags := (printFlags or $08) ;
      if captureFlags.jobNotify then 
        printFlags := (printFlags or $10) ;
      if (captureFlags.tabSize > 0) then
        printFlags := (printFlags or $40) ;
      if (length(captureFlags.banner) > 0) then
        printFlags := (printFlags or $80) ;
      strPLCopy(jobDescription,captureFlags.description,sizeOf(jobDescription)-1) ;
      strPLCopy(formName,upperCase(captureFlags.formName),sizeOf(formName)-1) ;
      strPLCopy(bannerText,captureFlags.banner,sizeOf(bannerText)-1) ;
      if (not captureFlags.autoEndCap) then
        flushCaptureOnClose := 1 ;
    end;
    nerror := NWSetCaptureFlags(nServer,nPort,capFlags1) ;
    if (nerror <> 0) then
      lastError := nerror
    else
      result := true;
  end;


function capture(nServer : TNWConnHandle ;
                 cQueue  : string ;
                 nPort   : TNWLpt ;
                 var captureFlags : TNWCaptureFlags) : boolean ;
  { Redirect Port to Network Queue }
  var
    queueID    : TObjID ;
    nObjServer : TNWConnHandle ;
  begin
    result := false ;
    if (@NWStartQueueCapture = nil) then
      @NWStartQueueCapture := getProcAddress(hNWCalls,'NWStartQueueCapture') ;
    if (@NWStartQueueCapture = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nObjServer := nServer ;  
    queueID := getObjNumber(nObjServer,cQueue,nw_printq) ;
    endCap(nPort) ;
    cQueue := upperCase(cQueue) ;
    result := (NWStartQueueCapture(nServer,
                                   nPort,
                                   queueID,
                                   @cQueue)=0) and
               setCaptureFlags(nServer,nPort,captureFlags);
  end;

function setBannerUserName(cBanner : string) : boolean ;
  { Set Print Banner UserName }
  var
    cpBanner : TRetBuff ;
  begin
    result := false ;
    if (@NWSetBannerUserName = nil) then
      @NWSetBannerUserName := getProcAddress(hNWCalls,'NWSetBannerUserName') ;
    if (@NWSetBannerUserName = nil) then
      exit ;
    strPCopy(cpBanner,cBanner) ;
    nError := NWSetBannerUserName(@cpBanner) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function getBannerUserName : string ;
  { Fetch Print Banner Text }
  var
    cpBanner : TRetBuff ;
  begin
    result := '';
    if (@NWGetBannerUserName = nil) then
      @NWGetBannerUserName := getProcAddress(hNWCalls,'NWGetBannerUserName') ;
    if (@NWGetBannerUserName = nil) then
      exit ;
    nError := NWGetBannerUserName(@cpBanner);
    if (nError <> 0) then
      lastError := nError
    else
      result := strPas(cpBanner);
  end;

function isCaptured(nPort : byte) : boolean ;
  { is port captured? }
  begin
    result := false ;
    if (@NWGetCaptureStatus = nil) then
      @NWGetCaptureStatus := getProcAddress(hNWCalls,'NWGetCaptureStatus') ;
    if (@NWGetCaptureStatus = nil) then
      exit ;
    result := (NWGetCaptureStatus(nPort) = $FF);
  end;

function endCap(nPort : byte) : boolean ;
  { End capture on Port }
  var
    capFlags1 : TNWCapFlags1 ;
    capFlags2 : TNWCapFlags2 ;
  begin
    result := false ;
    if (@NWEndCapture = nil) then
      @NWEndCapture := getProcAddress(hNWCalls,'NWEndCapture') ;
    if (@NWEndCapture = nil) then
      exit ;
    if (@NWGetCaptureFlags = nil) then
      @NWGetCaptureFlags := getProcAddress(hNWCalls,'NWGetCaptureFlags') ;
    if (@NWGetCaptureFlags = nil) then
      exit ;
    if (NWGetCaptureFlags(nPort,capFlags1,capFlags2) = 0) and
       (capFlags2.fileCapFlag = $FF) then {file capture in progress - must use NWCancelCapture}
      result := cancelCapture(nPort)
    else    
      begin
        nError := NWEndCapture(nPort) ;
        if (nError <> 0) then
          lastError := nError
        else
          result := true ;
      end;     
  end;

function getMaxPrinters : byte ;
  { Max Number of LPT Ports Support by Shell }
  var
    ntemp : TNWNumber ;
  begin
    result := 0 ;
    if (@NWGetMaxPrinters = nil) then
      @NWGetMaxPrinters := getProcAddress(hNWCalls,'NWGetMaxPrinters') ;
    if (@NWGetMaxPrinters = nil) then
      exit ;
    ntemp  := 0 ;
    nError := NWGetMaxPrinters(@ntemp) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := ntemp ;
  end;

function isQueueOperator(nServer : TNWConnHandle ;
                         cQueue,
                         cUser  : string) : boolean ;
  var
    nloop      : word ;
    tempList   : TStringList ;
  begin
    result := false ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    tempList := getQueueOperators(nServer,cQueue) ;
    cuser    := upperCase(allTrim(cUser)) ;
    for nloop := 0 to (tempList.count -1) do begin
      if isInList(nServer,tempList[nloop]) then
        begin
          result := true  ;
          break ;
        end;
    end;
    tempList.free ;
  end;

function cancelCapture(nPort : TNWLpt) : boolean ;
  {abort jobs in progress, or ends file capture}
  begin
    result := false ;
    if (@NWCancelCapture = nil) then
      @NWCancelCapture := getProcAddress(hNWCalls,'NWCancelCapture') ;
    if (@NWCancelCapture = nil) then
      exit ;
    nError := NWCancelCapture(nPort) ;  
    if (nError <> 0) then
      lastError := nError 
    else
      result := true ;  
  end;  

{$IFNDEF Win32}  
function captureToFile(nServer  : TNWConnHandle ;  
                       nPort    : TNWLpt ; 
                       fileName : string ;
                       var captureFlags : TNWCaptureFlags) : boolean ;
  {capture to file - (16 bit only)}
  var
    dirHandle    : TNWDirHandle ;
    cpFilename,
    cpRelative   : TRetBuff ;
  begin
    result := false ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWStartFileCapture = nil) then
      @NWStartFileCapture := getProcAddress(hNWCalls,'NWStartFileCapture') ;
    if (@NWStartFileCapture = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpFilename,fileName) ;
    if (NWParseNetWarePath(@cpfileName,
                           nserver,
                           dirHandle,
                           @cpRelative) <> 0) then
      strCopy(cpRelative,cpFilename) ;
    nError := NWStartFileCapture(nServer,nPort,dirHandle,@cpRelative) ;
    if (nError <> 0) then
      lastError := nError 
    else
      begin
        setCaptureFlags(nServer,nPort,captureFlags);
        result := true ;  
      end;  
  end;
{$ENDIF}  

function NPrint(nServer    : TNWConnHandle ;
                queueName,
                inFileName : string) : boolean ;
  var
    nyear,nmon,nday,
    nhour,nmin,nsec,nmsec : word ;
    numRead,
    numWritten    : longint ;
    queueID       : TObjID ;
    inFileHandle,
    outFileHandle : NWFILE_HANDLE ;
    dirHandle     : TNWDirHandle  ;
    fileOnServerHandle,
    nObjServer    : TNWConnHandle ;
    capFlags      : TNWCaptureFlags ;
    jobStruct     : NW_QUEUE_JOB ;
    cpRelative,
    cpFileName    : TRetBuff ;
    buf           : array[0..2047] of char;
  begin
    result       := false ;
    numwritten   := 0 ;
    infileHandle := 0 ;
    if (@NWCreateQueueFile2 = nil) then
      @NWCreateQueueFile2 := getProcAddress(hNWCalls,'NWCreateQueueFile2') ;
    if (@NWCreateQueueFile2 = nil) then
      exit ;
    if (@NWCloseFileAndStartQueueJob2 = nil) then
      @NWCloseFileAndStartQueueJob2 := getProcAddress(hNWCalls,'NWCloseFileAndStartQueueJob2') ;
    if (@NWCloseFileAndStartQueueJob2 = nil) then
      exit ;
    if (@NWGetCaptureFlags = nil) then
      @NWGetCaptureFlags := getProcAddress(hNWCalls,'NWGetCaptureFlags') ;
    if (@NWGetCaptureFlags = nil) then
      exit ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    strPCopy(cpFileName,inFileName) ;
    if (NWParseNetWarePath(@cpfileName,
                           fileOnServerHandle, {output - not necessarily queue's serverhandle}
                           dirHandle,
                           @cpRelative) <> 0) then
      strCopy(cpRelative,cpFilename) ;

    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nObjServer := nServer ;  
    queueID := getObjNumber(nObjServer,queueName,nw_printQ) ; {nds can change the nServer parameter}
    if (queueID = 0) or (nServer = 0) then
      exit ;
    fillChar(jobStruct,sizeOf(jobStruct),#0) ;
    decodeDate(now,nYear,nMon,nDay) ;
    decodeTime(now,nHour,nMin,nSec,nMsec) ;
    outFileHandle := 0 ;
    {read/use current cap flags}
    if getCaptureFlags(1,capFlags) then
      strPCopy(jobStruct.formName,capflags.formName) ;
    {provide job information}
    with jobStruct do begin
      strPCopy(jobDescription,strPas(cpRelative)) ;
      targetServerID := qms_any_server ;
      fillChar(targetExecutionTime,sizeOf(targetExecutionTime),$FF) ; {immediately}
      jobEntryTime[0] := byte(nyear - 1900) ;
      jobEntryTime[1] := byte(nmon) ;
      jobEntryTime[2] := byte(nday) ;
      jobEntryTime[3] := byte(nhour);
      jobEntryTime[4] := byte(nmin) ;
      jobEntryTime[5] := byte(nsec) ;
      jobControlFlags   := qms_auto_start ;
      numberOfCopies1   := 0 ;
      numberOfCopies    := capFlags.copies ;
      tabSize           := 0 ;
      versionNumber     := 0 ;
      printControlFlags := 0 ;
    end;
    nError := NWCreateQueueFile2(nServer,
                                 queueID,
                                 jobStruct,
                                 outFileHandle) ;
    if (nError <> 0) then
      begin
        lastError := nError;
        exit ;
      end;

    {copy file to output handle}
    try
      inFileHandle := _lOpen(cpFileName,OF_READ) ;
      repeat
        fillChar(buf,sizeOf(buf),#0) ;
        numRead := _lRead(inFileHandle, @buf, sizeOf(buf));
        if (numRead > 0) then
          numWritten := _lWrite(outFileHandle,buf,numRead) ;
      until (numRead = 0) or (numWritten <> numRead);
    finally
      _lClose(inFileHandle) ;
    end;

    {crank it up}
    nError := NWCloseFileAndStartQueueJob2(nServer,
                                           queueID,
                                           jobStruct.jobNumber,
                                           outFileHandle) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;


function NPrintEx(nServer    : TNWConnHandle ;
                  queueName,
                  inFileName : string;
                  captureFlags : TNWCaptureFlags) : boolean ; {added captureFlags parameter}
  var
    nyear,nmon,nday,
    nhour,nmin,nsec,nmsec : word ;
    numRead,
    numWritten    : longint ;
    queueID       : TObjID ;
    nObjServer,
    fileOnServer  : TNWConnHandle ;
    inFileHandle,
    outFileHandle : NWFILE_HANDLE ;
    dirHandle     : TNWDirHandle ;
    jobStruct     : NW_QUEUE_JOB ;
    cpRelative,
    cpFileName    : TRetBuff ;
    buf           : array[0..2047] of char;
  begin
    result       := false ;
    numwritten   := 0 ;
    infileHandle := 0 ;
    if (@NWCreateQueueFile2 = nil) then
      @NWCreateQueueFile2 := getProcAddress(hNWCalls,'NWCreateQueueFile2') ;
    if (@NWCreateQueueFile2 = nil) then
      exit ;
    if (@NWCloseFileAndStartQueueJob2 = nil) then
      @NWCloseFileAndStartQueueJob2 := getProcAddress(hNWCalls,'NWCloseFileAndStartQueueJob2') ;
    if (@NWCloseFileAndStartQueueJob2 = nil) then
      exit ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    strPCopy(cpFileName,inFileName) ;
    if (NWParseNetWarePath(@cpfileName,
                           fileOnServer, {output - not necessarily the queue's server handle}
                           dirHandle,
                           @cpRelative) <> 0) then
      strCopy(cpRelative,cpFilename) ;

    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nObjServer := nServer ;  
    queueID := getObjNumber(nObjServer,queueName,nw_printQ) ; {nds can change nServer value}
    if (queueID = 0) or (nServer = 0) then
      exit ;
    fillChar(jobStruct,sizeOf(jobStruct),#0) ;
    decodeDate(now,nYear,nMon,nDay) ;
    decodeTime(now,nHour,nMin,nSec,nMsec) ;
    outFileHandle := 0 ;
    {provide job information}
    with jobStruct do begin
      if (length(captureFlags.banner) > 0) then
        strPCopy(bannerFileField,captureFlags.banner) ;
      if (length(captureFlags.description) > 0) then
        strPCopy(jobDescription,captureFlags.description)
      else
        strPCopy(jobDescription,strPas(cpRelative)) ;
      if (length(captureFlags.formName) > 0) then  
        strPCopy(formName,captureflags.formName) ;
      if (length(captureFlags.userID) > 0) then  
        clientID := getObjNumber(nObjServer, captureFlags.userID, nw_user) ;
      targetServerID := qms_any_server ;
      fillChar(targetExecutionTime,sizeOf(targetExecutionTime),$FF) ; {immediately}
      jobEntryTime[0] := byte(nyear - 1900) ;
      jobEntryTime[1] := byte(nmon) ;
      jobEntryTime[2] := byte(nday) ;
      jobEntryTime[3] := byte(nhour);
      jobEntryTime[4] := byte(nmin) ;
      jobEntryTime[5] := byte(nsec) ;
      jobControlFlags   := qms_auto_start ;
      numberOfCopies1   := 0 ;
      numberOfCopies    := captureFlags.copies ;
      tabSize           := captureFlags.tabSize ;
      versionNumber     := 0 ;
      printControlFlags := 0 ;
      if (not captureFlags.formFeed) then
        printControlFlags := (printControlFlags or JOB_NO_FORM_FEED) ;
      if captureFlags.jobNotify then
        printControlFlags := (printControlFlags or JOB_NOTIFICATION) ;
      if (length(captureFlags.banner) > 0) then
        printControlFlags := (printControlFlags or JOB_PRINT_BANNER) ;
    end;
    nError := NWCreateQueueFile2(nServer,
                                 queueID,
                                 jobStruct,
                                 outFileHandle) ;
    if (nError <> 0) then
      begin
        lastError := nError;
        exit ;
      end;

    {copy file to output handle}
    try
      inFileHandle := _lOpen(cpFileName,OF_READ) ;
      repeat
        fillChar(buf,sizeOf(buf),#0) ;
        numRead := _lRead(inFileHandle, @buf, sizeOf(buf));
        if (numRead > 0) then
          numWritten := _lWrite(outFileHandle,buf,numRead) ;
      until (numRead = 0) or (numWritten <> numRead);
    finally
      _lClose(inFileHandle) ;
    end;

    {crank it up}
    nError := NWCloseFileAndStartQueueJob2(nServer,
                                           queueID,
                                           jobStruct.jobNumber,
                                           outFileHandle) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;



end.
