{ *************** Bindery Related Functions ****************

  Developed with Novell Inc.'s Client SDK for C SDK

  Copyright 1995-1999 by Devont Software Inc. and Jim Tyson.  All
  rights reserved.

}

unit NwProp;

interface

uses
  SysUtils,
  winTypes,
  winProcs,
  Classes ;

{$IFDEF Ver110}  
  {$ObjExportAll On}
{$ENDIF}
  
  {$I nwlib.inc}

  type
    {$I nwTypes.inc }

  type
    TNWLoginControl = record
      accountExpiresDate       : TDateTime ;
      accountExpired           : boolean   ;
      passwordExpiresDate      : TDateTime ;
      graceLogins              : byte ;
      expirationInterval       : word ;
      graceReset               : byte ;
      minPasswordLength        : byte ;
      maxConcurrentConnections : word ;
      timeBitMap               : array[0..41] of byte ;
      lastLoginDate            : TDateTime ;
      restrictionFlags         : byte      ;
      maxDiskBlocks            : longint   ;
      badLoginCount            : word      ;
      nextResetTime            : longint   ;
      badStationAddress        : string    ;
  end;


  {General Rights Structure}
  type
    TNWRights = record
      supervisor    : boolean ;
      read          : boolean ;
      open          : boolean ; { Netware 2.x only }
      write         : boolean ;
      create        : boolean ;
      erase         : boolean ;
      modify        : boolean ;
      filescan      : boolean ; { Netware 2.x = Search }
      accessControl : boolean ;
  end;

  type
    TNWAccountStatus = record
      balance : longint ;
      limit          : longint ;
    end;

  { ************** Public Functions *************** }
  function deleteTrusteeRight(nServer    : TNWConnHandle ;
                              cUserName  : string        ;
                              cVolume    : string        ;
                              cPath      : string) : boolean ;

  function modifyTrusteeRights(nServer    : TNWConnHandle ;
                               cUserName  : string        ;
                               cVolName   : string        ;
                               cPath      : string        ;
                               rightsList : TNWRights) : boolean ;

  function changeNWPassword(nServer : TNWConnHandle ;
                            cUserID,
                            cPassword,
                            cOldPassword : string) : boolean ;

  function createObject(nServer       : TNWConnHandle ;
                       objName       : string        ;
                       objType       : TObjType      ;
                       fullName      : string        ;
                       password      : string        ;
                       permanent     : boolean       ;
                       readSecurity  : byte          ;
                       writeSecurity : byte) : boolean ;

  function createProperty(
                       nServer       : TNWConnHandle ;
                       objName       : string        ;
                       objType       : TObjType      ;
                       propName      : string        ;
                       objTypeFlag   : TNWFlags      ;
                       permanent     : boolean       ;
                       readSecurity  : TNWFlags      ;
                       writeSecurity : TNWFlags) : boolean ;

  function renameObject(nServer        : TNWConnHandle ;
                       oldName,newName : string ;
                       objType         : TObjType) : boolean ;

  function deleteObject(
                       nServer       : TNWConnHandle ;
                       objName       : string) : boolean ;

  function addObjectToSet(
                       nServer       : TNWConnHandle ;
                       objName       : string ;
                       propName      : string ;
                       memName       : string) : boolean ;

  function changeObjectSecurity(
                       nServer       : TNWConnHandle ;
                       objName       : string ;
                       readSecurity  : TNWFlags ;
                       writeSecurity : TNWFlags) : boolean ;

  function deleteObjectFromSet(
                       nServer       : TNWConnHandle ;
                       objName       : string ;
                       propName      : string ;
                       memName       : string) : boolean ;

  function deleteProperty(
                        nServer    : TNWConnHandle ;
                        objName    : string ;
                        propName   : string) : boolean ;

  function changePropertySecurity(
                       nServer       : TNWConnHandle ;
                       objName       : string ;
                       propName      : string ;
                       readSecurity  : TNWFlags ;
                       writeSecurity : TNWFlags) : boolean ;

  function getObjectInfo(
                       nServer           : TNWConnHandle ;
                       inObject          : string ;
                       var hasProperties : boolean ;
                       var readSecurity  : TNWFlags ;
                       var writeSecurity : TNWFlags) : boolean ;


  procedure compareObjectAccessRights(
                       nServer           : TNWConnHandle ;
                       targetObject      : string   ;
                       var canRead       : boolean  ;
                       var canWrite      : boolean) ;

  function getPropertyList(
                       nServer           : TNWConnHandle ;
                       objectName        : string ;
                       searchValue       : string) : TStringList ;

  function getTrusteeList(nServer : TNWConnHandle ;
                          cUserID : string) : TStringList ;

  function addUserToGroup(nServer : TNWConnHandle ;
                          cGroup  : string ;
                          cUser   : string ) : boolean ;

  function deleteUserFromGroup(nServer : TNWConnHandle ;
                               cGroup  : string ;
                               cUser   : string ) : boolean ;

  function getEffectiveRights(nServer : TNWConnHandle ;
                              cPath   : string ;
                              var dirRights : TNWRights) : boolean ;

  function getObjectDirRights(nServer : TNWConnHandle ;
                              cUserID : string ;
                              cPath   : string ;
                              var rightsList : TNWRights) : boolean ;

  function setLoginControl(nServer   : TNWConnHandle ;
                           objName   : string ;
                           objType   : TObjType ;
                           loginCtrl : TNWloginControl) : boolean ;

  function writeItemProperty(nServer  : TNWConnHandle ;
                             objName  : string ;
                             objType  : TObjType ;
                             propName : string ;
                             value    : string) : boolean ;
  {2.21 additions}
  function readItemProperty(nServer  : TNWConnHandle ;
                            objName  : string ;
                            objType  : TObjType ;
                            propName : string) : string ;

  function readSetProperty(nServer  : TNWConnHandle ;
                           objName  : string ;
                           objType  : TObjType ;
                           propName : string) : TStringList ;

  function readObjNamesFromSet(nServer  : TNWConnHandle ;
                               objName  : string ;
                               objType  : TObjType ;
                               propName : string) : TStringList ;

  {2.22 additions}
  function getLoginControl(nServer       : TNWConnHandle ;
                           objName       : string ;
                           objType       : TObjType ;
                           var loginCtrl : TNWLoginControl) : boolean ;

  {2.3 additions}
  function addTrusteeToFile(nServer    : TNWConnHandle ;
                            fileSpec,
                            objName    : string ;
                            rightsList : TNWRights) : boolean ;

  {3.21 addition}                          
  function getAccountStatus(nServer : TNWConnHandle ;  {requires accounting installed}
                            objName : string ;
                            objType : TObjType ;
                            var accountStatus : TNWAccountStatus) : boolean ;
  
  type
    TNWProp = class(TObject)
    public
      function getLastError : string ;
      procedure resetLastError ;
  end;

var
  lasterror : TNWError ;

implementation

uses
  nwTools,
  nwLib ;

{$I nwprop.imp}  {Netware API Imports}
{$I nwstd.imp}

var
  nError : TNWError ;

{ ****************** Component functions ******************* }
function TNWProp.getLastError : string ;
  begin
    result := intToHex(lastError,sizeOf(TNWError)) ;
  end;

procedure TNWProp.resetLastError ;
  begin
    lastError := 0 ;
  end;

{ ******************** Public Functions ******************* }
function setLoginControl(nServer   : TNWConnHandle ;
                         objName   : string ;
                         objType   : TObjType ;
                         loginCtrl : TNWLoginControl) : boolean ;
  var
    nwLoginControl : loginControl ;
    nMon           : word ;
    nDay           : word ;
    nYear          : word ;
    nHour          : word ;
    nMin           : word ;
    nSec           : word ;
    nMSec          : word ;
    propertyName   : TObjName ;
    objectName     : TObjName ;
  begin
    result := false ;
    if (@NWWritePropertyValue = nil) then
      @NWWritePropertyValue := getProcAddress(hNWCalls,'NWWritePropertyValue') ;
    if (@NWWritePropertyValue = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    fillChar(nwloginControl,sizeOf(loginControl),0) ;
    {$IFDef Win32}
    fillChar(nwLoginControl.timeBitmap,42,-1) ;
    {$ENDIF}
    decodeDate(loginCtrl.accountExpiresDate,nYear,nMon,nDay) ;
    nwLoginControl.accountExpiresYear  := nYear ;
    nwLoginControl.accountExpiresMonth := nMon ;
    nwLoginControl.accountExpiresDay   := nDay ;
    if loginCtrl.accountExpired then
      nwLoginControl.accountExpired := 1
    else
      nwLoginControl.accountExpired := 0 ;
    if (loginCtrl.passwordExpiresDate > 0) then
      begin
        decodeDate(loginCtrl.passwordExpiresDate,nYear,nMon,nDay) ;
        nwLoginControl.passwordExpiresYear   := nYear ;
        nwLoginControl.passwordExpiresMonth  := nMon ;
        nwLoginControl.passwordExpiresDay    := nDay ;
      end;
    nwLoginControl.passwordGraceLogins   := loginCtrl.graceLogins;
    nwLoginControl.expirationInterval    := swap(loginCtrl.expirationInterval) ;
    nwLoginControl.graceReset            := loginCtrl.graceReset ;
    nwLoginControl.minimumPasswordLength := loginCtrl.minPasswordLength ;
    nwLoginControl.maxConcurrentConnections := swap(loginCtrl.maxConcurrentConnections) ;
    if (loginCtrl.lastLoginDate > 0) then
      begin
        decodeDate(loginCtrl.lastLoginDate,nYear,nMon,nDay) ;
        decodeTime(loginCtrl.lastLoginDate,nHour,nMin,nSec,nMSec) ;
        nwLoginControl.lastLoginDate[0] := nYear ;
        nwLoginControl.lastLoginDate[1] := nMon ;
        nwLoginControl.lastLoginDate[2] := nDay  ;
        nwLoginControl.lastLoginDate[3] := nHour ;
        nwLoginControl.lastLoginDate[4] := nMin ;
        nwLoginControl.lastLoginDate[5] := nSec ;
      end;
    nwLoginControl.restrictionFlags := loginCtrl.restrictionFlags ;
    nwLoginControl.maxDiskBlocks    := loginCtrl.maxDiskBlocks ;
    nwLoginControl.badLoginCount    := swap(loginCtrl.badLoginCount) ;
    nwLoginControl.nextResetTime    := loginCtrl.nextResetTime ;
    strPCopy(propertyName,'LOGIN_CONTROL') ;
    strPCopy(objectName,upperCase(allTrim(objName))) ;
    nError := NWWritePropertyValue(nServer,
                                   @objectName,
                                   objType,
                                   @propertyName,
                                   1,
                                   @nwLoginControl,
                                   0) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;


function getLoginControl(nServer       : TNWConnHandle ;
                         objName       : string ;
                         objType       : TObjType ;
                         var loginCtrl : TNWLoginControl) : boolean ;
  var
    nwLoginControl : loginControl ;
    cpProperty,
    cpObject       : TObjName ;
    more,
    flags          : byte ;
  begin
    result := false ;
    if (@NWReadPropertyValue = nil) then
      @NWReadPropertyValue := getProcAddress(hNWCalls,'NWReadPropertyValue') ;
    if (@NWReadPropertyValue = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpProperty,'LOGIN_CONTROL') ;
    strPCopy(cpObject,upperCase(objName)) ;
    nError := NWReadPropertyValue(nServer,
                                  @cpObject,
                                  objType,
                                  @cpProperty,
                                  1,
                                  @nwLoginControl,
                                  @more,
                                  @flags) ;
    if (nError <> 0) then
      lastError := nError
    else
      with loginCtrl do try
        if (nwLoginControl.accountExpiresYear > 0) then
          accountExpiresDate := encodeDate(nwLoginControl.accountExpiresYear,
                                           nwLoginControl.accountExpiresMonth,
                                           nwLoginControl.accountExpiresDay) ;
        if (nwLoginControl.passwordExpiresYear > 0) then
          passwordExpiresDate := encodeDate(nwLoginControl.passwordExpiresYear,
                                            nwLoginControl.passwordExpiresMonth,
                                            nwLoginControl.passwordExpiresDay) ;
        if (nwLoginControl.lastloginDate[0] > 0) then
          begin
            lastLoginDate := encodeDate(nwLoginControl.lastLoginDate[0],
                                        nwLoginControl.lastLoginDate[1],
                                        nwLoginControl.lastLoginDate[2]) +
                             encodeTime(nwLoginControl.lastLoginDate[3],
                                        nwLoginControl.lastLoginDate[4], +
                                        nwLoginControl.lastLoginDate[5],0) ;
          end;
        accountExpired           := (nwLoginControl.accountExpired = 1) ;
        graceLogins              := loginCtrl.graceLogins ;
        expirationInterval       := loginCtrl.expirationInterval ;
        graceReset               := loginCtrl.graceReset ;
        minPasswordLength        := nwLoginControl.minimumPasswordLength ;
        maxConcurrentConnections := swap(nwLoginControl.maxConcurrentConnections) ;
        restrictionFlags         := nwLoginControl.restrictionFlags ;
        maxDiskBlocks            := nwLoginControl.maxDiskBlocks    ;
        badLoginCount            := nwLoginControl.badLoginCount    ;
        nextResetTime            := nwLoginControl.nextResetTime    ;
        result := true ;
      except
        on e:exception do
         {nothing - date encode error}
      end;
  end;



function writeItemProperty(nServer  : TNWConnHandle ;
                           objName  : string ;
                           objType  : TObjType ;
                           propName : string ;
                           value    : string) : boolean ;
  var
    objectName   : TObjName ;
    propertyName : TObjName ;
    segmentData  : TNWSegmentData;
  begin
    result := false ;
    if (@NWWritePropertyValue = nil) then
      @NWWritePropertyValue := getProcAddress(hNWCalls,'NWWritePropertyValue') ;
    if (@NWWritePropertyValue = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(objectName,upperCase(allTrim(objName))) ;
    strPCopy(propertyName,upperCase(allTrim(propName))) ;
    strPCopy(segmentData,allTrim(value)) ;
    nError := NWWritePropertyValue(nServer,
                                    @objectName,
                                    objType,
                                    @propertyName,
                                    1,
                                    @segmentData,0) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function createProperty(nServer       : TNWConnHandle ;
                        objName       : string        ;
                        objType       : TObjType      ;
                        propName      : string        ;
                        objTypeFlag   : TNWFlags      ;
                        permanent     : boolean       ;
                        readSecurity  : TNWFlags      ;
                        writeSecurity : TNWFlags) : boolean ;
  var
    securityFlags : TNWFlags ;
    objectName    : TObjName ;
    propertyName  : TObjName ;
    propertyFlags : TNWFlags ;
    objDuration   : TNWFlags ;
  begin
    result := false ;
    if (@NWCreateProperty = nil) then
      @NWCreateProperty := getProcAddress(hNWCalls,'NWCreateProperty') ;
    if (@NWCreateProperty = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(objectName,upperCase(allTrim(objName))) ;
    strPCopy(propertyName,upperCase(allTrim(propName))) ;
    objDuration := BF_DYNAMIC ;
    if permanent then
      objDuration := BF_STATIC;
    propertyFlags := (objTypeFlag or objDuration) ;
    securityFlags := (writeSecurity or readSecurity) ;
    nError := NWCreateProperty( nServer,
                                @objectName,
                                objType,
                                @propertyName,
                                propertyFlags,
                                securityFlags) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function createObject(nServer       : TNWConnHandle ;
                      objName       : string        ;
                      objType       : TObjType      ;
                      fullName      : string        ;
                      password      : string        ;
                      permanent     : boolean       ;
                      readSecurity  : byte          ;
                      writeSecurity : byte) : boolean ;
  var
    objectName    : TObjName ;
    objectFlags   : TNWFlags ;
    securityFlags : TNWFlags ;
    propertyFlags : TNWFlags ;
    propName      : TObjName ;
  begin
    result := false ;
    if (@NWCreateObject = nil) then
      @NWCreateObject := getProcAddress(hNWCalls,'NWCreateObject') ;
    if (@NWCreateObject = nil) then
      exit ;
    if (@NWCreateProperty = nil) then
      @NWCreateProperty := getProcAddress(hNWCalls,'NWCreateProperty') ;
    if (@NWCreateProperty = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    securityFlags := (writeSecurity or readSecurity ) ;
    strPCopy(objectName,objName) ;
    objectFlags := BF_DYNAMIC ;
    if permanent then
      objectFlags := BF_STATIC;
    if (NWCreateObject(nServer,
                       @objectName,
                       objType,
                       objectFlags,
                       securityFlags)=0) then
      begin
        { Object Type specific Properties }
        case objType of
          nw_user :
            begin
              { Create Default User Properties }
              strPCopy(propName,'IDENTIFICATION'+#0) ;
              propertyFlags := BF_ITEM ;
              securityFlags := (BS_ANY_READ or BS_OBJECT_WRITE ) ;
              if (NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags)=0) then
                begin
                  result := True ;
                  { Set ID property }
                  writeItemProperty(nServer,objName,objType,
                                'IDENTIFICATION',
                                 fullName) ;

                  { Create PASSWORD Property and Fill It }
                  changeNWPassword(nServer,objName,password,'') ;

                  securityFlags := (BS_LOGGED_READ or BS_SUPER_WRITE) ;
                  strPCopy(propName,'GROUPS_I''M_IN') ;
                  propertyFlags := BF_SET ;
                  NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags);

                  securityFlags := (BS_OBJECT_READ or BS_SUPER_WRITE) ;
                  strPCopy(propName,'SECURITY_EQUALS') ;
                  propertyFlags := BF_SET ;
                  NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags);

                  securityFlags := (BS_OBJECT_READ or BS_SUPER_WRITE) ;
                  strPCopy(propName,'LOGIN_CONTROL') ;
                  propertyFlags := BF_ITEM ;
                  NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags);

                  securityFlags := (BS_OBJECT_READ or BS_SUPER_WRITE) ;
                  propertyFlags := BF_ITEM ;
                  strPCopy(propName,'MISC_LOGIN_INFO') ;
                  NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags);
                  addUserToGroup(nServer,'EVERYONE',objName) ;
                end;
            end;
          nw_group :
            begin
              strPCopy(propName,'IDENTIFICATION') ;
              propertyFlags := BF_ITEM ;
              securityFlags := (BS_ANY_READ or BS_OBJECT_WRITE) ;
              { Create Default Group Properties }
              if (NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags)=0) then
                begin
                  result := True ;
                  { Set ID property }
                  writeItemProperty(nServer,objName,objType,
                                'IDENTIFICATION',
                                 fullName) ;

                  securityFlags := (BS_LOGGED_READ or BS_SUPER_WRITE) ;
                  strPCopy(propName,'GROUP_MEMBERS') ;
                  propertyFlags := BF_SET ;
                  NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags);
              end;
            end;
          nw_printQ :
            begin
              securityFlags := (BS_LOGGED_READ or BS_SUPER_WRITE ) ;
              strPCopy(propName,'Q_USERS') ;
              propertyFlags := BF_SET ;
              { Create Default Queue Properties }
              if (NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags)=0) then
                begin
                  result := True ;
                  securityFlags := (BS_LOGGED_READ or BS_SUPER_WRITE ) ;
                  strPCopy(propName,'Q_SERVERS') ;
                  propertyFlags := BF_SET ;
                  NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags);

                  securityFlags := (BS_LOGGED_READ or BS_SUPER_WRITE ) ;
                  strPCopy(propName,'Q_OPERATORS') ;
                  propertyFlags := BF_SET ;
                  NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags);

                  securityFlags := (BS_LOGGED_READ or BS_SUPER_WRITE ) ;
                  strPCopy(propName,'Q_DIRECTORY') ;
                  propertyFlags := BF_ITEM ;
                  NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags);
                end;
            end;
          nw_printServer :
            begin
              { Create Remainder of PS Properties }
              strPCopy(propName,'IDENTIFICATION') ;
              propertyFlags := BF_ITEM ;
              securityFlags := (BS_ANY_READ or BS_OBJECT_WRITE ) ;
              if (NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags)=0) then

                begin
                  result := True ;
                  { Set ID property }
                  writeItemProperty(nServer,objName,objType,
                                   'IDENTIFICATION',
                                    fullName) ;
                  securityFlags := (BS_ANY_READ or BS_SUPER_WRITE ) ;
                  strPCopy(propName,'PS_OPERATORS') ;
                  propertyFlags := BF_SET ;
                  NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags);

                  securityFlags := (BS_ANY_READ or BS_SUPER_WRITE ) ;
                  strPCopy(propName,'PS_USERS') ;
                  propertyFlags := BF_SET ;
                  NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags);

                  securityFlags := (BS_OBJECT_READ or BS_SUPER_WRITE ) ;
                  strPCopy(propName,'LOGIN_CONTROL') ;
                  propertyFlags := BF_ITEM ;
                  NWCreateProperty(nServer,
                                   @objectName,
                                   objType,
                                   @propName,
                                   propertyFlags,
                                   securityFlags);
                end;
            end;
        end;
      end;
  end;


function changeNWPassword(nServer : TNWConnHandle ;
                          cUserID,
                          cPassword,
                          cOldPassword : string) : boolean ;
  var
    cpUserID   : TObjName ;
    cpPassword : TNWPassword ;
    cpOldPass  : TNWPassword ;
    objType    : TObjType ;
  begin
    result := false ;
    if (@NWChangeObjectPassword = nil) then
      @NWChangeObjectPassword := getProcAddress(hNWCalls,'NWChangeObjectPassword') ;
    if (@NWChangeObjectPassword = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpUserID,upperCase(allTrim(cUserID))) ;
    strPCopy(cpPassword,upperCase(allTrim(cPassword))) ;
    strPCopy(cpOldPass,upperCase(allTrim(cOldPassword))) ;
    objType := getObjType(nServer,cUserID) ;
    nError := NWChangeObjectPassword(nServer,
                                   @cpUserID,
                                   objType,
                                   @cpOldPass,
                                   @cpPassword) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function renameObject(nServer : TNWConnHandle ;
                      oldName, newName : string ;
                      objType : TObjType) : boolean ;
  var
    oldObjectName,
    newObjectName : TObjName ;
  begin
    result := false ;
    if (@NWRenameObject = nil) then
      @NWRenameObject := getProcAddress(hNWCalls,'NWRenameObject') ;
    if (@NWRenameObject = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(oldObjectName,upperCase(allTrim(oldName))) ;
    strPCopy(newObjectName,upperCase(allTrim(newName))) ;
    nError := NWRenameObject(nServer,
                            @oldObjectName,
                            @newObjectName,
                            objType) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

 function deleteObject(
                      nServer       : TNWConnHandle ;
                      objName       : string) : boolean ;
  var
    objectName : TObjName ;
    objectType : TObjType ;
  begin
    result := false ;
    if (@NWDeleteObject = nil) then
      @NWDeleteObject := getProcAddress(hNWCalls,'NWDeleteObject') ;
    if (@NWDeleteObject = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    objectType := getObjType(nServer,objName) ;
    strPCopy(objectName,upperCase(allTrim(objName))) ;
    nError := NWDeleteObject(nServer,
                             @objectName,
                             objectType) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function deleteObjectFromSet(
                     nServer       : TNWConnHandle ;
                     objName       : string ;
                     propName      : string ;
                     memName       : string) : boolean ;
  var
    objectName,
    propertyName,
    memberName : TObjName ;
    objectType,
    memberType : TObjType ;
  begin
    result := false ;
    if (@NWDeleteObjectFromSet = nil) then
      @NWDeleteObjectFromSet := getProcAddress(hNWCalls,'NWDeleteObjectFromSet') ;
    if (@NWDeleteObjectFromSet = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(objectName,upperCase(allTrim(objName))) ;
    strPCopy(propertyName,upperCase(allTrim(propName))) ;
    strPCopy(memberName,upperCase(allTrim(memName))) ;
    objectType := getObjType(nServer,objName) ;
    memberType := getObjType(nServer,memName) ;
    if (objectType < nw_user) or (memberType < nw_user) then
      exit ;
    nError  := NWDeleteObjectFromSet( nServer,
                                      @objectName,
                                      objectType,
                                      @propertyName,
                                      @memberName,
                                      memberType);
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function addObjectToSet(
                     nServer       : TNWConnHandle ;
                     objName       : string ;
                     propName      : string ;
                     memName       : string) : boolean ;
  var
    objectName,
    propertyName,
    memberName : TObjName ;
    objectType,
    memberType : TObjType ;
  begin
    result := false ;
    if (@NWAddObjectToSet = nil) then
      @NWAddObjectToSet := getProcAddress(hNWCalls,'NWAddObjectToSet') ;
    if (@NWAddObjectToSet = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    objectType := getObjType(nServer,objName) ;
    memberType := getObjType(nServer,memName) ;
    if (objectType < nw_user) or (memberType < nw_user) then
      exit ;
    strPCopy(objectName,upperCase(allTrim(objName))) ;
    strPcopy(memberName,upperCase(allTrim(memName))) ;
    strPCopy(propertyName,upperCase(allTrim(propName))) ;
    nError := NWAddObjectToSet(
                      nServer,
                      @objectName,
                      objectType,
                      @propertyName,
                      @memberName,
                      memberType);
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function changePropertySecurity(
                     nServer       : TNWConnHandle ;
                     objName       : string ;
                     propName      : string ;
                     readSecurity  : TNWFlags ;
                     writeSecurity : TNWFlags) : boolean ;
  var
    objectName,
    propertyName : TObjName ;
    objectType   : TObjType ;
  begin
    result := false ;
    if (@NWChangePropertySecurity = nil) then
      @NWChangePropertySecurity := getProcAddress(hNWCalls,'NWChangePropertySecurity') ;
    if (@NWChangePropertySecurity = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(objectName,upperCase(allTrim(objName))) ;
    strPCopy(propertyName,upperCase(allTrim(propName))) ;
    objectType := getObjType(nServer,objName) ;
    if (objectType < nw_user) then
      exit ;
    nError := NWChangePropertySecurity(
                  nServer,
                  @objectName,
                  objectType,
                  @propertyName,
                  (readSecurity or writeSecurity)) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function changeObjectSecurity(
                     nServer       : TNWConnHandle ;
                     objName       : string ;
                     readSecurity  : TNWFlags ;
                     writeSecurity : TNWFlags) : boolean ;
  var
    objectName : TObjName ;
    objectType : TObjType ;
  begin
    result := false ;
    if (@NWChangeObjectSecurity = nil) then
      @NWChangeObjectSecurity := getProcAddress(hNWCalls,'NWChangeObjectSecurity') ;
    if (@NWChangeObjectSecurity = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(objectName,upperCase(allTrim(objName))) ;
    objectType := getObjType(nServer,objName) ;
    if (objectType < nw_user) then
      exit ;
    nError := NWChangeObjectSecurity( nServer,
                                      @objectName,
                                      objectType,
                                      (readSecurity or writeSecurity)) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function getObjectInfo(nServer           : TNWConnHandle ;
                       inObject          : string ;
                       var hasProperties : boolean ;
                       var readSecurity  : TNWFlags ;
                       var writeSecurity : TNWFlags) : boolean ;
  var
    cpTarget    : TObjName ;
    objID       : TObjID   ;
    objType     : TObjType ;
    hasProps    : TNWFlags ;
    objDuration : TNWFlags ;
    objSecurity : TNWFlags ;
  begin
    result := false ;
    if (@NWScanObject = nil) then
      @NWScanObject := getProcAddress(hNWCalls,'NWScanObject') ;
    if (@NWScanObject = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    objID := -1 ;
    strPCopy(cpTarget,upperCase(allTrim(inObject))) ;
    nError := NWScanObject(nServer,
                     @cpTarget,
                     getObjType(nServer,inObject),
                     @objID,
                     @cpTarget,
                     @objType,
                     @hasProps,
                     @objDuration,
                     @objSecurity) ;
    if (nError <> 0) then
      lastError := nError
    else
      begin
        { Denotes Object Contains Properties }
        hasProperties := (hasProps > 0) ;
        { get obj write rights }
        if ((BS_ANY_WRITE and objSecurity) = BS_ANY_WRITE) then
          writeSecurity := BS_ANY_WRITE ;
        if ((BS_LOGGED_WRITE and objSecurity) = BS_LOGGED_WRITE) then
          writeSecurity := BS_LOGGED_WRITE ;
        if ((BS_OBJECT_WRITE and objSecurity) = BS_OBJECT_WRITE) then
          writeSecurity := BS_OBJECT_WRITE ;
        if ((BS_SUPER_WRITE and objSecurity) = BS_SUPER_WRITE) then
          writeSecurity := BS_SUPER_WRITE ;
        if ((BS_BINDERY_WRITE and objSecurity) = BS_BINDERY_WRITE) then
          writeSecurity := BS_BINDERY_WRITE ;
        { get obj read rights }
        if ((BS_ANY_READ and objSecurity) = BS_ANY_READ) then
          readSecurity := BS_ANY_READ ;
        if ((BS_LOGGED_READ and objSecurity) = BS_LOGGED_READ) then
          readSecurity := BS_LOGGED_READ ;
        if ((BS_OBJECT_READ and objSecurity) = BS_OBJECT_READ) then
          readSecurity := BS_OBJECT_READ ;
        if ((BS_SUPER_READ and objSecurity) = BS_SUPER_READ) then
          readSecurity := BS_SUPER_READ ;
        if ((BS_BINDERY_READ and objSecurity) = BS_BINDERY_READ) then
          readSecurity := BS_BINDERY_READ ;
        result := true ;
      end;
  end;

function getPropertyList(
                     nServer           : TNWConnHandle ;
                     objectName        : string ;
                     searchValue       : string) : TStringList ;
  { get object's property list }
  var
    cpObjectName     : TObjName ;
    cpSearchValue    : TObjName ;
    cpProperty       : TObjName ;
    propertyFlags    : TNWFlags ;
    propertySecurity : TNWFlags ;
    hasValueFlag     : TNWFlags ;
    moreFlag         : TNWFlags ;
    sequence         : longint  ;
  begin
    result := TStringList.create ;
    if (@NWScanProperty = nil) then
      @NWScanProperty := getProcAddress(hNWCalls,'NWScanProperty') ;
    if (@NWScanProperty = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    sequence := -1 ;
    strPCopy(cpSearchValue,upperCase(allTrim(searchValue))) ;
    strPCopy(cpObjectName,upperCase(allTrim(objectName))) ;
    while true do begin
      if (NWScanProperty(nServer,
                         @cpObjectName,
                         getObjType(nServer,objectName),
                         @cpSearchValue,
                         @sequence,
                         @cpProperty,
                         @propertyFlags,
                         @propertySecurity,
                         @hasValueFlag,
                         @moreFlag) = 0) then
        result.add(strPas(cpProperty))
      else
        break;
      if (moreFlag < 1) then
        break ;
    end;
    result.sort ;
  end;

procedure compareObjectAccessRights(
                     nServer           : TNWConnHandle ;
                     targetObject      : string   ;
                     var canRead       : boolean  ;
                     var canWrite      : boolean) ;
  var
    myReadRights   : TNWFlags ;
    myWriteRights  : TNWFlags ;
    myRightsMask   : TNWFlags ;
    objReadRights  : TNWFlags ;
    objWriteRights : TNWFlags ;
    hasProps       : TNWFlags ;
    objDuration    : TNWFlags ;
    objSecurity    : TNWFlags ;
    objID          : TObjID   ;
    cpTarget       : TObjName ;
    objType        : TObjType ;
  begin
    if (@NWGetBinderyAccessLevel = nil) then
      @NWGetBinderyAccessLevel := getProcAddress(hNWCalls,'NWGetBinderyAccessLevel') ;
    if (@NWGetBinderyAccessLevel = nil) then
      exit ;
    if (@NWScanObject = nil) then
      @NWScanObject := getProcAddress(hNWCalls,'NWScanObject') ;
    if (@NWScanObject = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpTarget,upperCase(allTrim(targetObject))) ;
    myReadRights   := 0 ;
    myWriteRights  := 0 ;
    myRightsMask   := 0 ;
    objReadRights  := 0 ;
    objWriteRights := 0 ;
    if (NWGetBinderyAccessLevel(nServer,
                                @myRightsMask,
                                @objID)=0) then
      begin
        { get my read rights }
        if ((BS_ANY_WRITE and myRightsMask) = BS_ANY_WRITE) then
          myWriteRights := BS_ANY_WRITE ;
        if ((BS_LOGGED_WRITE and myRightsMask) = BS_LOGGED_WRITE) then
          myWriteRights := BS_LOGGED_WRITE ;
        if ((BS_OBJECT_WRITE and myRightsMask) = BS_OBJECT_WRITE) then
          myWriteRights := BS_OBJECT_WRITE ;
        if ((BS_SUPER_WRITE and myRightsMask) = BS_SUPER_WRITE) then
          myWriteRights := BS_SUPER_WRITE ;
        if ((BS_BINDERY_WRITE and myRightsMask) = BS_BINDERY_WRITE ) then
          myWriteRights := BS_BINDERY_WRITE ;
        { get my write rights }
        if ((BS_ANY_READ and myRightsMask) = BS_ANY_READ) then
          myReadRights := BS_ANY_READ ;
        if ((BS_LOGGED_READ and myRightsMask) = BS_LOGGED_READ) then
          myReadRights := BS_LOGGED_READ ;
        if ((BS_OBJECT_READ and myRightsMask) = BS_OBJECT_READ) then
          myReadRights := BS_OBJECT_READ ;
        if ((BS_SUPER_READ and myRightsMask) = BS_SUPER_READ) then
          myReadRights := BS_SUPER_READ ;
        if ((BS_BINDERY_READ and myRightsMask) = BS_BINDERY_READ) then
          myReadRights := BS_BINDERY_READ ;
        { Retrieve Object Access Level Mask }
        objID := -1 ;
        if (NWScanObject(nServer,
                         @cpTarget,
                         getObjType(nServer,targetObject),
                         @objID,
                         @cpTarget,
                         @objType,
                         @hasProps,
                         @objDuration,
                         @objSecurity) = 0) then
          begin
            { get obj read writes }
            if ((BS_ANY_WRITE and objSecurity) > 0) then
              objWriteRights := BS_ANY_WRITE ;
            if ((BS_LOGGED_WRITE and objSecurity) > 0) then
              objWriteRights := BS_LOGGED_WRITE ;
            if ((BS_OBJECT_WRITE and objSecurity) > 0) then
              objWriteRights := BS_OBJECT_WRITE ;
            if ((BS_SUPER_WRITE and objSecurity) > 0) then
              objWriteRights := BS_SUPER_WRITE ;
            if ((BS_BINDERY_WRITE and objSecurity) > 0) then
              objWriteRights := BS_BINDERY_WRITE ;
            { get my write rights }
            if ((BS_ANY_READ and objSecurity) > 0) then
              objReadRights := BS_ANY_READ ;
            if ((BS_LOGGED_READ and objSecurity) > 0) then
              objReadRights := BS_LOGGED_READ ;
            if ((BS_OBJECT_READ and objSecurity) > 0) then
              objReadRights := BS_OBJECT_READ ;
            if ((BS_SUPER_READ and objSecurity) > 0) then
              objReadRights := BS_SUPER_READ ;
            if ((BS_BINDERY_READ and objSecurity) > 0) then
              objReadRights := BS_BINDERY_READ ;
          end;
        { Fill Return Values }
        canRead  := (myReadRights >= objReadRights) ;
        canWrite := (myWriteRights >= objWriteRights) ;
      end;
  end;

function getObjectDirRights(nServer : TNWConnHandle ;
                            cUserID : string ;
                            cPath   : string ;
                            var rightsList : TNWRights) : boolean ;
  var
    cpPath     : TNWPath          ;
    pathInfo   : TNWPathInfo      ;
    rightsMask : TNWRightsMask    ;
    sequence   : TNWTrusteeSeqNum ;
    objectID   : TObjID           ;
  begin
    result := false ;
    if (@NWScanObjectTrusteePaths = nil) then
      @NWScanObjectTrusteePaths := getProcAddress(hNWCalls,'NWScanObjectTrusteePaths') ;
    if (@NWScanObjectTrusteePaths = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    sequence := -1 ;
    objectID := getObjNumber(nServer,cUserID,
                             getObjType(nServer,cUserID)) ;
    if parseNetwarePath(nServer,
                        cPath,
                        pathInfo) then
      begin
        strPCopy(cpPath,cPath) ;
        while (NWScanObjectTrusteePaths(nServer,
                                        objectID,
                                        pathInfo.volumeID,
                                        @sequence,
                                        @rightsMask,
                                        @cpPath)=0) do begin
          if (strLen(cpPath) < 2) then
            break ;
          if (pos(upperCase(strPas(cpPath)),
                  upperCase(pathInfo.cServer + '/' + cPath)) > 0)  then
            begin
              result := true ;
              with rightsList do begin
                read          := ((rightsMask and TA_READ)>0)   ;
                write         := ((rightsMask and TA_WRITE)>0)  ;
                open          := ((rightsMask and TA_OPEN)>0)   ;
                create        := ((rightsMask and TA_CREATE)>0) ;
                erase         := ((rightsMask and TA_DELETE)>0) ;
                modify        := ((rightsMask and TA_MODIFY)>0) ;
                fileScan      := ((rightsMask and TA_SEARCH)>0) ;
                accessControl := ((rightsMask and TA_OWNERSHIP)>0) ;
                supervisor    := accessControl ; {not completely accurate..see api notes}
              end;
              break ;
            end;
        end;
      end;
  end;

function getEffectiveRights(nServer : TNWConnHandle ;
                            cPath   : string ;
                            var dirRights : TNWRights) : boolean ;
  { Get current user's effective rights in specifed directory }
  var
    connHandle : TNWConnHandle ;
    dirHandle  : TNWDirHandle ;
    cprelative : TNWPath ;
    cpPath     : TNWPath ;
    rightsMask : TNWRightsMask ;
  begin
    result    := false ;
    if (@NWParseNetwarePath = nil) then
      @NWParseNetwarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetwarePath = nil) then
      exit ;
    if (@NWGetEffectiveRights = nil) then
      @NWGetEffectiveRights := getProcAddress(hNWCalls,'NWGetEffectiveRights') ;
    if (@NWGetEffectiveRights = nil) then
      exit ;
    dirHandle := 0 ;
    strPCopy(cpPath,upperCase(cPath)) ;
    NWParseNetwarePath(@cpPath,
                       connhandle,
                       dirHandle,
                       @cpRelative) ;
    if (NWGetEffectiveRights(connHandle,
                             dirHandle,
                             @cpRelative,
                             @rightsMask)=0) then
      begin
        result := true ;
        with dirRights do begin
          read          := ((rightsMask and TA_READ)>0)   ;
          write         := ((rightsMask and TA_WRITE)>0)  ;
          open          := ((rightsMask and TA_OPEN)>0)   ;
          create        := ((rightsMask and TA_CREATE)>0) ;
          erase         := ((rightsMask and TA_DELETE)>0) ;
          modify        := ((rightsMask and TA_MODIFY)>0) ;
          fileScan      := ((rightsMask and TA_SEARCH)>0) ;
          accessControl := ((rightsMask and TA_OWNERSHIP)>0) ;
        end;
      end;
  end;

function getTrusteeList(nServer : TNWConnHandle ;
                        cUserID : string) : TStringList ;
  { get complete list of user's directory rights  - formerly getObjectTrusteeRights}
  var
    crights    : string[8]   ;
    cserver    : string[48]  ;
    cpObject   : TObjName         ;
    objType    : TObjType         ;
    cpPath     : TNWPath          ;
    volNo      : TNWVol           ;
    sequence   : TNWTrusteeSeqNum ;
    rightsMask : TNWRightsMask    ;
    objID      : TObjID           ;
  begin
    result   := TStringList.create ;
    if (@NWScanObjectTrusteePaths = nil) then
      @NWScanObjectTrusteePaths := getProcAddress(hNWCalls,'NWScanObjectTrusteePaths') ;
    if (@NWScanObjectTrusteePaths = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    sequence := -1 ;
    objType  := getObjType(nServer,cUserID) ;
    objID    := getObjNumber(nServer,cUserID,objType) ;
    cserver  := getServerName(nServer) ;
    strPCopy(cpObject,cUserID) ;

    if (objType = nw_user) or (objType = nw_group) then
      begin
        for volno := 0 to 254 do begin
          { grab all trustee rights on volume }
          while true do begin
            cRights    := '--------' ;
            if (NWScanObjectTrusteePaths(nServer,
                                         objID,
                                         volno,
                                         @sequence,
                                         @rightsMask,
                                         @cpPath) = 0) then
              begin
                { no more trustees on volume }
                if (strLen(cpPath) < 2) then
                  break;
                if ((TA_NONE and rightsMask) < 1) then
                  begin
                    if ((TA_READ and rightsMask) > 0) then
                      cRights[2] := 'R' ;
                    if ((TA_WRITE and rightsMask) > 0) then
                      cRights[3] := 'W' ;
                    if ((TA_CREATE and rightsMask) > 0) then
                      cRights[4] := 'C' ;
                    if ((TA_DELETE and rightsMask) > 0) then
                      cRights[5] := 'E' ;
                    if ((TA_MODIFY and rightsMask) > 0) then
                      cRights[6] := 'M' ;
                    if ((TA_SEARCH and rightsMask) > 0) then
                      cRights[7] := 'F' ;
                    if ((TA_OWNERSHIP and rightsMask) > 0) then
                      begin
                       { to get real 'S' right, must call NWScanforTrustees
                         and/or NWDirectoryScanforTrustees2 }
                        cRights[1] := 'S' ;
                        cRights[8] := 'A' ;
                      end;
                    (* {  2.x defines }
                      begin
                        if cRights[7] = 'F' then
                          cRights[7] := 'S' ;  { 2.x Search  }
                        if cRights[2] = 'S' then
                          cRights[2] := 'P' ;  { 2.x Parental }
                        cRights[8] := 'O' ;    { 2.x Open }
                      end; *)
                  end;
                result.add(cServer + '/' +
                           strPas(cpPath) + '    [' +
                           cRights + ']') ;
              end
            else
              break ;
          end;
        end;
      end;
  end;


function addUserToGroup(nServer : TNWConnHandle ;
                        cGroup  : string ;
                        cUser   : string ) : boolean ;
  { add user to an existing group }
  begin
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    result := (addObjectToSet(nServer,
                              cUser,
                              'GROUPS_I''M_IN',
                              cGroup) and
               addObjectToSet(nServer,
                              cUser,
                              'SECURITY_EQUALS',
                              cGroup)) and
               addObjectToSet(nServer,
                              cGroup,
                              'GROUP_MEMBERS',
                              cUser) ;
  end;

function deleteUserFromGroup(nServer : TNWConnHandle ;
                             cGroup  : string ;
                             cUser   : string ) : boolean ;
  { delete user from existing group }
  begin
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    result := (deleteObjectFromSet(nServer,cUser,'GROUPS_I''M_IN',cGroup) and
               deleteObjectFromSet(nServer,cUser,'SECURITY_EQUALS',cGroup) and
               deleteObjectFromSet(nServer,cGroup,'GROUP_MEMBERS',cUser)) ;
  end;

function deleteProperty(nServer    : TNWConnHandle ;
                        objName    : string ;
                        propName   : string) : boolean ;
  {delete an object's property}
  var
    objectName   : TObjName ;
    propertyName : TObjName ;
  begin
    result := false ;
    if (@NWDeleteProperty = nil) then
      @NWDeleteProperty := getProcAddress(hNWCalls,'NWDeleteProperty') ;
    if (@NWDeleteProperty = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(objectName,upperCase(allTrim(objName))) ;
    strPCopy(propertyName,upperCase(allTrim(propName))) ;
    nError := NWDeleteProperty(nServer,
                               @objectName,
                               getObjType(nServer,objName),
                               @propertyName) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function modifyTrusteeRights(nServer    : TNWConnHandle ;
                             cUserName  : string        ;
                             cVolName   : string        ;
                             cPath      : string        ;
                             rightsList : TNWRights) : boolean ;
  {edit/create directory entry trustee list}
  var
    cpPath     : TNWPath       ;
    objectID   : TObjID        ;
    rightsMask : TNWRightsMask ;
    dirHandle  : TNWDirHandle  ;
  begin
    result := false ;
    if (@NWAddTrusteeToDirectory = nil) then
      @NWAddTrusteeToDirectory := getProcAddress(hNWCalls,'NWAddTrusteeToDirectory') ;
    if (@NWAddTrusteeToDirectory = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    rightsMask := 0 ;
    dirHandle  := 0 ;
    strPCopy(cpPath,upperCase(allTrim(cVolName))+setPath(upperCase(cPath))) ;
    objectID := getObjNumber(nServer,cUserName,getObjType(nServer,cUserName)) ;
    with rightsList do begin
      if supervisor then
        rightsMask := (rightsMask or TA_ALL) ;
      if read then
        rightsMask := (rightsMask or TA_READ) ;
      if open then
        rightsMask := (rightsMask or TA_OPEN) ;
      if write then
        rightsMask := (rightsMask or TA_WRITE) ;
      if create then
        rightsMask := (rightsMask or TA_CREATE) ;
      if erase then
        rightsMask := (rightsMask or TA_DELETE) ;
      if modify then
        rightsMask := (rightsMask or TA_MODIFY) ;
      if fileScan then
        rightsMask := (rightsMask or TA_SEARCH) ;
      if accessControl then
        rightsMask := (rightsMask or TA_OWNERSHIP) ;
    end;
    {create new trustee}
    nError := NWAddTrusteeToDirectory(nServer,
                                      dirHandle,
                                      @cpPath,
                                      objectID,
                                      rightsMask) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function deleteTrusteeRight(nServer    : TNWConnHandle ;
                            cUserName  : string        ;
                            cVolume    : string        ;
                            cPath      : string) : boolean ;
  var
    cpPath     : TNWPath   ;
    cpRelative : TNWPath   ;
    dirHandle  : TNWDirHandle  ;
    objectID   : TObjID        ;
  begin
    result := false ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    @NWDeleteTrusteeFromDirectory := getProcAddress(hNWCalls,'NWDeleteTrusteeFromDirectory') ;
    if (@NWDeleteTrusteeFromDirectory = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    dirHandle := 0 ;
    objectID  := getObjNumber(nServer,cUserName,getObjType(nServer,cUserName)) ;
    strPCopy(cpPath,upperCase(alltrim(cVolume)) +
                    upperCase(strExtract(cPath,' ',1))) ;
    nError := NWParseNetWarePath(@cpPath,
                                   nServer,
                                   dirHandle,
                                   @cpRelative) ;
    if (nError <> 0) then
      begin
        lastError := nError ;
        exit ;
      end;
    nError := NWDeleteTrusteeFromDirectory(nServer,
                                           dirHandle,
                                           @cpRelative,
                                           objectID) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;


function readItemProperty(nServer  : TNWConnHandle ;
                          objName  : string ;
                          objType  : TObjType ;
                          propName : string) : string ;
  var
    nindex,
    nmore,
    propType    : byte     ;
    cpBuffer    : TRetBuff ;
    cpProperty,
    cpObject    : TObjName ;
  begin
    result := '' ;
    if (@NWReadPropertyValue = nil) then
      @NWReadPropertyValue := getProcAddress(hNWCalls,'NWReadPropertyValue') ;
    if (@NWReadPropertyValue = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nindex := 1 ;
    propType := type_item ;
    strPCopy(cpProperty,upperCase(propName));
    strPCopy(cpObject,objName)   ;
    nError := NWReadPropertyValue(nServer,
                                  @cpObject,
                                  objType,
                                  @cpProperty,
                                  nindex,
                                  @cpBuffer,
                                  @nmore,
                                  @propType) ;
    if (nError <> 0) then
      begin
        lastError := nError ;
        exit ;
      end
    else
      result := strPas(cpBuffer) ;
  end;

function readSetProperty(nServer  : TNWConnHandle ;
                         objName  : string ;
                         objType  : TObjType ;
                         propName : string) : TStringList ;
  var
    nindex,
    nmore,
    propType    : byte     ;
    cpBuffer    : TRetBuff ;
    cpProperty,
    cpObject    : TObjName ;
  begin
    result := TStringList.create ;
    if (@NWReadPropertyValue = nil) then
      @NWReadPropertyValue := getProcAddress(hNWCalls,'NWReadPropertyValue') ;
    if (@NWReadPropertyValue = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nindex := 1 ;
    propType := type_set ;
    strPCopy(cpProperty,upperCase(propName));
    strPCopy(cpObject,objName)   ;
    while true do begin
      nError := NWReadPropertyValue(nServer,
                                    @cpObject,
                                    objType,
                                    @cpProperty,
                                    nindex,
                                    @cpBuffer,
                                    @nmore,
                                    @propType) ;
      if (nError <> 0) and (result.count < 1) then
        begin
          lastError := nError ;
          break ;
        end
      else if (nError <> 0) and (result.count > 0) then
        break
      else if (nError = 0) and (strLen(cpBuffer) > 0) then
        result.add(cpBuffer) ;
      {more data segments?}
      if (nmore = 0) then
        break
      else
        inc(nindex) ;
    end;
  end;

function readObjNamesFromSet(nServer  : TNWConnHandle ;
                             objName  : string ;
                             objType  : TObjType ;
                             propName : string) : TStringList ;
  {convert objNumbers stored in set properties to proper objNames}
  var
    nindex,
    nmore,
    propType    : byte     ;
    ntemp,
    nloop       : integer  ;
    nObjID      : TObjID   ;
    ctemp,
    cObjID      : string   ;
    cpBuffer    : TRetBuff ;
    cpProperty,
    cpObject    : TObjName ;
  begin
    result := TStringList.create ;
    if (@NWReadPropertyValue = nil) then
      @NWReadPropertyValue := getProcAddress(hNWCalls,'NWReadPropertyValue') ;
    if (@NWReadPropertyValue = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nindex := 1 ;
    propType := type_set ;
    strPCopy(cpProperty,upperCase(propName));
    strPCopy(cpObject,objName)   ;
    while true do begin
      nError := NWReadPropertyValue(nServer,
                                    @cpObject,
                                    objType,
                                    @cpProperty,
                                    nindex,
                                    @cpBuffer,
                                    @nmore,
                                    @propType) ;
      if (nError <> 0) and (result.count < 1) then
        begin
          lastError := nError ;
          break ;
        end
      else if (nError <> 0) and (result.count > 0) then
        break
      else if (nError = 0) and (strLen(cpBuffer) > 0) then
        for nloop := 0 to 31 do begin  { 128 byte buffer, 4 bytes each }
          ntemp := nloop*4 ;
          ctemp := '$' + intToHex(ord(cpBuffer[ntemp+0]),2) +
                         intToHex(ord(cpBuffer[ntemp+1]),2) +
                         intToHex(ord(cpBuffer[ntemp+2]),2) +
                         intToHex(ord(cpBuffer[ntemp+3]),2) ;
          nobjID := longSwap(longInt(strToInt(ctemp))) ;
          cObjID := getObjName(nServer,nObjID) ;
          if (length(cObjID) > 0) then
            result.add(cObjID) ;
        end;
      {more data segments?}
      if (nmore = 0) then
        break
      else
        inc(nindex) ;
    end;
  end;

function addTrusteeToFile(nServer   : TNWConnHandle ;
                         fileSpec,
                         objName    : string ;
                         rightsList : TNWRights) : boolean ;
  var
    cpObjName  : TObjName ;
    rightsMask : word ;
    objID      : TObjID ;
    dirHandle  : TNWDirHandle ;
    cpPath,
    cpRelative : TRetBuff ;
  begin
    result := false ;
    if (@NWParseNetwarePath = nil) then
      @NWParseNetwarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetwarePath = nil) then
      exit ;
    if (@NWAddTrustee = nil) then
      @NWAddTrustee := getProcAddress(hNWCalls,'NWAddTrustee') ;
    if (@NWAddTrustee = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpObjName,objName) ;
    strPCopy(cpPath,upperCase(fileSpec)) ;
    nError := NWParseNetWarePath(@cpPath,
                                 nServer,
                                 dirHandle,
                                 @cpRelative) ;
    if (nError <> 0) then
      begin
        lastError := nError ;
        exit ;
      end;
    objID := getObjNumber(nServer,objName,nw_user) ;
    rightsMask := 0 ;
    with rightsList do begin
      if supervisor then
        rightsMask := (rightsMask or TA_ALL) ;
      if read then
        rightsMask := (rightsMask or TA_READ) ;
      if open then
        rightsMask := (rightsMask or TA_OPEN) ;
      if write then
        rightsMask := (rightsMask or TA_WRITE) ;
      if create then
        rightsMask := (rightsMask or TA_CREATE) ;
      if erase then
        rightsMask := (rightsMask or TA_DELETE) ;
      if modify then
        rightsMask := (rightsMask or TA_MODIFY) ;
      if fileScan then
        rightsMask := (rightsMask or TA_SEARCH) ;
      if accessControl then
        rightsMask := (rightsMask or TA_OWNERSHIP) ;
    end;
    nError := NWAddTrustee(nServer,dirHandle,@cpRelative,objID,rightsMask) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function getAccountStatus(nServer : TNWConnHandle ;
                           objName : string ;
                           objType : TObjType ;
                           var accountStatus : TNWAccountStatus) : boolean ;
  var
    balance,
    limit      : longint ;
    cpObjName  : TObjName ;
    holdStatus : HOLDS_STATUS ;                         
  begin
    result := false ;
    if (@NWGetAccountStatus = nil) then
      @NWGetAccountStatus := getProcAddress(hNWCalls,'NWGetAccountStatus') ;
    if (@NWGetAccountStatus = nil) then
      exit ;
    strPCopy(cpObjName,objName) ;
    nError := NWGetAccountStatus(nServer,objType,@cpObjName,@balance,@limit,@holdStatus) ;
    if (nError <> 0) then
      lastError := nError 
    else
      begin
        result := true ;
        accountStatus.balance := balance ;
        accountStatus.limit   := limit ;
      end;  
  end;                           
  

{ end of unit }
end.
