{ *************** Server - Related Functions ****************

  Developed with Novell Inc.'s Client SDK for C SDK

  Copyright 1995-1999 by Devont Software Inc. and Jim Tyson.  All
  rights reserved.

}

unit NWServer;

interface

uses
  SysUtils,
  winTypes,
  winProcs,
  Forms,
  Controls,
  Classes ;
  
{$IFDEF Ver110}  
  {$ObjExportAll On}
{$ENDIF}

  {$I nwlib.inc}
  
  type
    {$I nwTypes.inc}

  {object disk usage record}
  type
     TNWDiskUsageStats = record
       usedDirectories : longint ;
       usedFiles       : longint ;
       usedBlocks      : longint ;
    end;

  {File/Directory Inherited Rights List}
  type
    TNWInheritedRightsList = record
      read          : boolean ;
      write         : boolean ;
      open          : boolean ;
      create        : boolean ;
      canDelete     : boolean ;
      owner         : boolean ;
      accessControl : boolean ;
      fileScan      : boolean ;
      search        : boolean ;
      modify        : boolean ;
      all           : boolean ;
      supervisor    : boolean ;
      normal        : boolean ;
   end;

   type
     TNWNameSpaceSet = (nameSpaceDOS, nameSpaceMac, nameSpaceNFS, nameSpaceFTAM, nameSpaceOS2) ;

   type
     TNWAllFileFlags = record
       normal,
       readOnly,
       hidden,
       system,
       executeOnly,
       directory,
       needsArchive,
       shareable,
       transactional,
       indexed,
       readAudit,
       writeAudit,
       immediatePurge,
       renameInhibit,
       deleteInhibit,
       copyInhibit,
       fileMigrated,
       dontMigrate,
       immediateCompress,
       fileCompressed,
       dontCompress,
       cantCompress : boolean 
     end;  

  { General File Information Interface Structure }
  type
    TNWFileInfo = record
      name              : string        ;
      updatedBy         : string        ;
      updatedDateTime   : TDateTime     ;
      ownerID           : string        ;
      lastArchivedBy    : string        ;
      creationdate      : TDateTime     ;
      lastArchiveDate   : TDateTime     ;
      lastAccessDate    : TDateTime     ;
      lastModifyDate    : TDateTime     ;
      fileSize          : longint       ;
      inheritedRights   : TNWInheritedRightsList ;
      maximumSpace      : TNWDirSpace   ;  { space left in directory }
      attributes        : TNWAttributes ;  {numeric version}
      attrsRec          : TNWAllFileFlags  ;  {expanded: record}
      flags             : TNWFlags      ;
      nameSpace         : TNWNameSpace  ;
      nameLength        : TNWNameLen    ;
  end;

  { Purged File Information }
  type
    TNWDeletedFileInfo = record
      attributes       : TNWAttributes ;
      flags            : TNWFlags      ;
      nameSpace        : TNWNameSpace  ;
      nameLength       : TNWNameLen    ;
      name             : string        ;
      creationDate     : TDateTime     ;
      ownerID          : string        ;
      lastArchiveDate  : TDateTime     ;
      lastArchivedBy   : string        ;
      updateDateTime   : TDateTime     ;
      updatedBy        : string        ;
      fileSize         : longint       ;
      inheritedRights  : TNWInheritedRightsList ;
      lastAccessDate   : TDateTime     ;
      deletedDateTime  : TDateTime     ;
      deletedBy        : string        ;
  end;

  { Internal DeletedFile Info}
  type
    pTNWDelFileInfo = ^TNWDelFileInfo ;
    TNWDelFileInfo = record
      nServer     : TNWConnHandle ;
      dirHandle   : TNWDirHandle ;
      iterHandle  : TNWSequence ;
      volNum      : TNWVol ;
      dirBase     : longint ;
      pFileName   : TRetBuff ;
    end;

  { General Connection Statistics Structure }
  type
    TNWConnStats = record
      loginTime      : TDateTime     ;
      bytesRead      : TNWNumBytes   ;
      bytesWritten   : TNWNumBytes   ;
      totalRequests  : TNWNumPackets ;
      recordLocks    : TNWNumPackets ;
      fileLocks      : TNWNumPackets ;
      expirationTime : TDateTime   ;
  end;

  { General Server Stats Interface Structure some 4.0 only }
  type
    TNWServerInfo = record
      serverName        : string ;
      serverUpTime      : TNWSysTime  ;    {4.0}
      processor         : byte ;
      numprocs          : byte ;
      utilization       : byte ;           {4.0}
      totalmemory       : TNWNum ;
      freememory        : TNWNum ;
      PacketsIn         : TNWNumPackets ;  {4.0}
      packetsOut        : TNWNumPackets ;  {4.0}
      version           : string ;
      maxConns          : word ;
      ConnsInUse        : word ;
      maxConnsUsed      : word ;
      numVolumes        : word ;
      sftLevel          : TNWSupportLevel ;
      ttsLevel          : TNWSupportLevel ;
  end;

  { Memory Cache Interface Structure - Netware 4.0 Only }
  type TNWMemCacheInfo = record
     serverUpTime,
     writeBlockCount,
     diskWriteCount,
     writeErrorCount,
     numCacheHits,
     numDirtyCacheHits,
     cacheDirtyWaitTime,
     cacheMaxConcurrentWrites,
     maxDirtyTime,
     DirCacheBuffers,
     maxByteCount,
     minCacheBuffers,
     LRUSittingTime,
     originalCacheBuffers,
     currentCacheBuffers        : longint ;
  end;

  type
    FILESYS_STATS = record
      systemElapsedTime      : nuint32;
      maxOpenFiles           : nuint16;
      maxFilesOpened         : nuint16;
      currOpenFiles          : nuint16;
      totalFilesOpened       : nuint32;
      totalReadRequests      : nuint32;
      totalWriteRequests     : nuint32;
      currChangedFATSectors  : nuint16;
      totalChangedFATSectors : nuint32;
      FATWriteErrors         : nuint16;
      fatalFATWriteErrors    : nuint16;
      FATScanErrors          : nuint16;
      maxIndexFilesOpened    : nuint16;
      currOpenIndexedFiles   : nuint16;
      attachedIndexFiles     : nuint16;
      availableIndexFiles    : nuint16
    end;

  type
    DSK_CACHE_STATS = record
      systemElapsedTime      : nuint32;
      cacheBufferCount       : nuint16;
      cacheBufferSize        : nuint16;
      dirtyCacheBuffers      : nuint16;
      cacheReadRequests      : nuint32;
      cacheWriteRequests     : nuint32;
      cacheHits              : nuint32;
      cacheMisses            : nuint32;
      physicalReadRequests   : nuint32;
      physicalWriteRequests  : nuint32;
      physicalReadErrors     : nuint16;
      physicalWriteErrors    : nuint16;
      cacheGetRequests       : nuint32;
      cacheFullWriteRequests : nuint32;
      cachePartialWriteRequests : nuint32;
      backgroundDirtyWrites     : nuint32;
      backgroundAgedWrites      : nuint32;
      totalCacheWrites          : nuint32;
      cacheAllocations          : nuint32;
      thrashingCount            : nuint16;
      LRUBlockWasDirtyCount     : nuint16;
      readBeyondWriteCount      : nuint16;
      fragmentedWriteCount      : nuint16;
      cacheHitOnUnavailCount    : nuint16;
      cacheBlockScrappedCount   : nuint16
    end;

  type
    VOL_STATS = record
      systemElapsedTime       : nint32;
      volumeNumber            : nuint8;
      logicalDriveNumber      : nuint8;
      sectorsPerBlock         : nuint16;
      startingBlock           : nuint16;
      totalBlocks             : nuint16;
      availableBlocks         : nuint16;
      totalDirectorySlots     : nuint16;
      availableDirectorySlots : nuint16;
      maxDirectorySlotsUsed   : nuint16;
      isHashing               : nuint8;
      isCaching               : nuint8;
      isRemovable             : nuint8;
      isMounted               : nuint8;
      volumeName              : array[0..15] of char ;
    end;

  type
    DIR_SPACE_INFO = record
      totalBlocks           : nuint32;
      availableBlocks       : nuint32;
      purgeableBlocks       : nuint32;
      notYetPurgeableBlocks : nuint32;
      totalDirEntries       : nuint32;
      availableDirEntries   : nuint32;
      reserved              : nuint32;
      sectorsPerBlock       : nuint8;
      volLen                : nuint8;
      volName               : array[0..NW_MAX_VOLUME_NAME_LEN-1] Of char ;
    end;

  type
    SERVER_AND_VCONSOLE_INFO = record
      currentServerTime : nuint32;
      vconsoleVersion   : nuint8;
      vconsoleRevision  : nuint8
    end;

  type
    FILE_SERVER_COUNTERS = record
      tooManyHops          : nuint16;
      unknownNetwork       : nuint16;
      noSpaceForService    : nuint16;
      noReceiveBuffers     : nuint16;
      notMyNetwork         : nuint16;
      netBIOSProgatedCount : nuint32;
      totalPacketsServiced : nuint32;
      totalPacketsRouted   : nuint32
    end;

  type
    FSE_SERVER_INFO = record
      replyCanceledCount          : nuint32;
      writeHeldOffCount           : nuint32;
      writeHeldOffWithDupRequest  : nuint32;
      invalidRequestTypeCount     : nuint32;
      beingAbortedCount           : nuint32;
      alreadyDoingReallocCount    : nuint32;
      deAllocInvalidSlotCount     : nuint32;
      deAllocBeingProcessedCount  : nuint32;
      deAllocForgedPacketCount    : nuint32;
      deAllocStillTransmittingCount  : nuint32;
      startStationErrorCount         : nuint32;
      invalidSlotCount               : nuint32;
      beingProcessedCount            : nuint32;
      forgedPacketCount              : nuint32;
      stillTransmittingCount         : nuint32;
      reExecuteRequestCount          : nuint32;
      invalidSequenceNumCount        : nuint32;
      duplicateIsBeingSentAlreadyCnt : nuint32;
      sentPositiveAcknowledgeCount   : nuint32;
      sentDuplicateReplyCount        : nuint32;
      noMemForStationCtrlCount       : nuint32;
      noAvailableConnsCount          : nuint32;
      reallocSlotCount               : nuint32;
      reallocSlotCameTooSoonCount    : nuint32
    end;

  type         
    LAN_CONFIG_INFO = Record    
       DriverCFG_MajorVersion : nuint8;      
       DriverCFG_MinorVersion : nuint8;      
       DriverNodeAddress : array[0..5] of nuint8;   
       DriverModeFlags : nuint16;      
       DriverBoardNum : nuint16;      
       DriverBoardInstance : nuint16;
       DriverMaxSize : nuint32;      
       DriverMaxRecvSize : nuint32;      
       DriverRecvSize : nuint32;      
       Reserved1 : array[0..2] of nuint32;      
       DriverCardID : nuint16;      
       DriverMediaID : nuint16;      
       DriverTransportTime : nuint16;      
       DriverReserved : array[0..15] of nuint8;     
       DriverMajorVersion : nuint8;      
       DriverMinorVersion : nuint8;      
       DriverFlags : nuint16;      
       DriverSendRetries : nuint16;      
       DriverLink : nuint32;      
       DriverSharingFlags : nuint16;      
       DriverSlot : nuint16;      
       DriverIOPortsAndLengths : array[0..3] of nuint16;
       DriverMemDecode0 : nuint32;      
       DriverLength0 : nuint16;      
       DriverMemDecode1 : nuint32;      
       DriverLength1 : nuint16;      
       DriverInterrupt : array[0..1] of nuint8;      
       DriverDMAUsage : array[0..1] of nuint8;      
       Reserved2 : array[0..2] of nuint32;      
       DriverLogicalName : array[0..17] of nuint8;      
       DriverLinearMem : array[0..1] of nuint32;      
       DriverChannelNum : nuint16;      
       DriverIOReserved : array[0..5] of nuint8   
  end; 


  type
    NWFSE_LAN_CONFIG_INFO = Record     
      serverTimeAndVConsoleInfo : SERVER_AND_VCONSOLE_INFO;      
      reserved : nuint16;      
      LANConfigInfo : LAN_CONFIG_INFO   
    End; 

  type
    NWFSE_FILE_SERVER_INFO = record
      serverTimeAndVConsoleInfo : SERVER_AND_VCONSOLE_INFO;
      reserved                  : nuint16;
      NCPStationsInUseCount     : nuint32;
      NCPPeakStationsInUseCount : nuint32;
      numOfNCPRequests          : nuint32;
      serverUtilization         : nuint32;
      ServerInfo                : FSE_SERVER_INFO;
      fileServerCounters        : FILE_SERVER_COUNTERS
    end;

  {file attribute flags}
  type
    TNWFileFlags = record
      normal       : boolean ;
      readOnly     : boolean ;
      hidden       : boolean ;
      system       : boolean ;
      executeOnly  : boolean ;
      directory    : boolean ;
      needsArchive : boolean ;
      shareable    : boolean ;
  end;
  
  {extended file attribute flags}
  type
    TNWExtFileFlags = record
      normal,
      readOnly,
      hidden,
      system,
      executeOnly,
      directory,
      needsArchive,
      shareable,
      transaction,
      indexed,
      readAudit,
      writeAudit    : boolean ;
    end;

  {extended volume information}
  type
    TNWVolExtendedInfo = record
      volType,
      statusFlag,
      sectorSize,
      sectorsPerCluster,
      volSizeInClusters,
      freeClusters,
      subAllocFreeableClusters,
      freeableLimboSectors,
      nonfreeableLimboSectors,
      availSubAllocSectors,
      nonuseableSubAllocSectors,
      subAllocClusters,
      numDataStreams,
      numLimboDataStreams,
      oldestDelFileAgeInTicks,
      numCompressedDataStreams,
      numCompressedLimboDataStreams,
      numNoncompressibleDataStreams,
      precompressedSectors,
      compressedSectors,
      numMigratedDataStreams,
      migratedSectors,
      clustersUsedByFAT,
      clustersUsedByDirs,
      clustersUsedByExtDirs,
      totalDirEntries,
      unusedDirEntries,
      totalExtDirExtants,
      unusedExtDirExtants,
      extAttrsDefined,
      extAttrExtantsUsed,
      DirectoryServicesObjectID,
      volLastModifiedDateAndTime   : nuint32 ;
  end;

  type
    TNWNLMInfo = record
      NLMFileName,
      NLMName,
      version,
      copyRight       : string ;
      nlmDate         : TDateTime ;
      NLMID,
      NLMFlags,
      NLMType,
      parentID,
      allocFreeBytes,
      allocFreeCount,
      messageLanguage,
      numReferPublics : longint ;
    end;

    type
      TNWDirectoryInfo = record
        ownerName : string ;
        dateTime  : TDateTime ;
        rights    : TNWInheritedRightsList ;
      end;

    type 
      TNWTransportType = (ttIPX, ttDDP, ttASP, ttUDP, ttTCP, ttUDP6, ttTCP6, ttWild) ;
      
    type
      TNWTransportAddr = record     
        transportType : TNWTransportType ;
        address       : string ;
    end;
      

    { public Functions }
    function disableLogins(nServer : TNWConnHandle) : boolean ;
    function enableLogins(nServer : TNWConnHandle) : boolean ;
    function downServer(nServer : TNWConnHandle) : boolean ;
    function NWLogin(server,userID,password : string ;
                     objType : TObjType) : boolean ;
    function NWLogout(nServer : TNWConnHandle) : boolean ;
    function getServerHandle(cServer : string) : TNWConnHandle ;
    function getServerSerial(cServer : string) : TNWSerial ;
    function getVolFileList(srchText : string) : TStringList ;
    function getDeletedFileInfo(inFile   : string;
                                var deletedFileInfo : TNWDeletedFileInfo) : boolean ;
{4} function getCacheInfo(nServer : TNWConnHandle;
                      var cacheInfo : TNWMemCacheInfo) : boolean ;
{2} function getFileSysStats(nServer : TNWConnHandle ; var FileSysInfo : FILESYS_STATS) : boolean ;
{2} function getDiskCacheStats(nServer : TNWConnHandle;
                     var cacheInfo : DSK_CACHE_STATS) : boolean ;
    function getVolumes(nServer : TNWConnHandle) : TStringList ;
{4} function getServerStats(nServer : TNWConnHandle ;
                            var serverStats : TNWServerInfo) : boolean ;
    function getDeletedFileList(nServerHandle : TNWConnHandle;
                     cpath : string) : TStringList ;
    function salvage(nServer : TNWConnHandle;
                     inFile  : string ;
                     outFile : string) : boolean ;
{4} function getUserStats(nServer : TNWConnHandle ;
                     nConn : TNWConnNumber ;
                     var connStats : TNWConnStats) : boolean ;
    function getServerHandleFromPath(cPath : string) : TNWConnHandle ;
    function nwDateTimeToTDateTime(datein : TNWDateTime) : TDateTime ;
    function setServerDateTime(nServer : TNWConnHandle ; dt : TDateTime) : boolean ;

    {1.5 additions}
    function createSemaphore(nServer       : TNWConnHandle ;
                             semaphoreName : string   ;
                             maxInstances : word ) : TNWSemaHandle  ;
    function decSemaValue(nServer    : TNWConnhandle ;
                          semaHandle : TNWSemaHandle ;
                          amount     : word) : boolean ;
    function incSemaValue(nServer    : TNWConnHandle ;
                          semaHandle : TNWSemaHandle ;
                          amount     : word) : boolean ;

    function freeSemaphore(nServer    : TNWConnHandle ;
                           semaHandle : TNWSemaHandle) : boolean ;
    function querySemaphore(nServer      : TNWConnHandle ;
                          semaHandle   : TNWSemaHandle ;
                          var semaValue : word) : word;   {returns open station count}
    function serverLoginOK(nServer : TNWConnHandle) : boolean ;

    function attach(serverName : string) : TNWConnHandle ;

    {2.0 Additions}
    function getVolBlockInfo(nServer : TNWConnHandle ;
                             volNum  : TNWVolNum ;
                             var volBlockInfo : DIR_SPACE_INFO) : boolean ;
    function getServerInformation(nServer : TNWConnHandle; { Netware 4.x Only }
                                  var serverStats : NWFSE_FILE_SERVER_INFO) : boolean ;
    function getDiskUtilization(nServer : TNWConnHandle ;
                                userID  : string ;
                                volNum  : byte ;
                                var diskStats : TNWDiskUsageStats) : boolean ;
    function getDirTrustees(nServer : TNWConnHandle ;
                            cPath : string) : TStringList ;

    {2.03 Changes}
    function purgeAllFiles(nServerHandle : TNWConnHandle ;
                           cpath : string) : boolean ; {new parm}

    {2.03 Additions}
    function ticksToTimeString(ticks : longInt) : string ;

    {2.11 Additions}
    function getDirectoryInfo(pathName : string;
                              var fileInfo : TNWFileInfo) : boolean ;

    {2.2 Changes}
    function volInfo(nServer : TNWConnHandle ;
                     volName : string ;
                     var volumeInfo : VOL_STATS) : boolean ; {now supports all platforms}

    {2.23 additions}
    function getVolDirList(srchText : string) : TStringList ;
    function setFileFlags(nServer   : TNWConnHandle ; {renamed from flagFile}
                      fileName  : string        ;
                      fileFlags : TNWFileFlags) : boolean ;
    function getFileFlags(nServer       : TNWConnHandle ;
                          fileName      : string ;
                          var fileFlags : TNWFileFlags) : boolean ;
    function getVolNum(nServer : TNWConnHandle ;
                       volName : string;
                       var volNum : integer) : boolean ;
    function getObjDiskRestrictions(nServer        : TNWConnHandle ;
                                    volName,
                                    objName        : string;
                                    var byteLimit  : longint;
                                    var bytesInUse : longint) : boolean ;
    function getVolExtendedInfo(nServer        : TNWConnHandle ;
                                volName        : string ;
                                var volExtInfo : TNWVolExtendedInfo) : boolean ;
    function purgeFile(nServerHandle : TNWConnHandle; fileName : string) : boolean ;
    
    {2.3 additions}
    function removeObjDiskRestrictions(nServer : TNWConnHandle ;
                                       volName, userID: string): boolean ;
    function setObjDiskRestrictions(nServer : TNWConnHandle ;
                                    volName,
                                    userID  : string;
                                    bytesRestrict: longInt): boolean;
    function clearConnection(nServer : TNWConnHandle; connID : word) : boolean ;
    function openConnByName(serverName : string ; ndsFormat, licensed : boolean) : TNWConnHandle ;
 
    {3.0 additions - all require Netware 4.x or higher}
    function getNLMID(nServer : TNWConnHandle ; NLMName : string) : longint ;
    function getNLMInfo(nServer     : TNWConnHandle ;
                        NLM_ID      : longint ;
                        var nlmInfo : TNWNLMInfo) : boolean ;
    function getNLMList(nServer : TNWConnHandle) : TStringList ;
    function nlmLoad(nServer : TNWConnHandle ; nlmFileName, parameters : string) : boolean ;  
    function nlmUnload(nServer : TNWConnHandle ; nlmFileName : string) : boolean ;
    function serverExecNCFFile(nServer : TNWConnHandle ; ncfFileName : string) : boolean ;
{not working}   function mountVolume(nServer : TNWConnHandle ; volName : string) : integer ;
{not working}   function dismountVolume(nServer : TNWConnHandle ; volName : string) : boolean ;

    {3.01 addition}
    function closeConnection(nServer : TNWConnHandle) : boolean ;
    function serverSetStringValue(nServer : TNWConnHandle ;
                                  setParmName,
                                  setParmString : string) : boolean ;
                           
    function serverSetIntegerValue(nServer : TNWConnHandle ;
                              setParmName : string ;
                              setParmInt  : longint) : boolean ;

    {3.5 additions}
    function setExtendedFileFlags(inFile : string ;
                                  const extFileFlags : TNWExtFileFlags ;
                                  delFileFlags : boolean) : boolean ; {thanks to Gunnar Prien!}
    function TTSEnable(nServer : TNWConnHandle) : boolean ;
    function TTSDisable(nServer : TNWConnHandle) : boolean ;
    function TTSBeginTransaction(nServer : TNWConnHandle) : boolean ;
    function TTSEndTransaction(nServer : TNWConnHandle) : longint ;
    function TTSAbortTransaction(nServer : TNWConnHandle) : boolean ;
    function getFileInfoOld(infile : string; {old version - DOS name space only}
                         var fileInfo : TNWFileInfo) : boolean ;
    
    {4.x additions}
    function getFileInfo(fileName : string ; 
                         var fileInfo : TNWFileInfo) : boolean ; {supports other name spaces}
    function purgeSingleFile(deletedFileInfo : TNWDelFileInfo) : boolean ;

    {4.6x additions}
    function setDirSpaceLimit(cPath : string ;
                              KByteLimit : longint) : boolean ; {0=No Limit, -1=No Bytes Allowed}
    function setDirFlags(pathName : string ; 
                         fileFlags : TNWAllFileFlags ;
                         nameSpace : TNWNameSpaceSet) : boolean ;
    function setFileOwner(fileName,
                          ownerName : string) : boolean ;
    function setDirectoryInfo(dirName : string; dirInfo : TNWDirectoryInfo) : boolean ;

    {new to 5.1}
    function getLANConfigInfo(nServer : TNWConnHandle ;
                              boardNum : integer ;
                              var lanConfigInfo : LAN_CONFIG_INFO) : boolean ;
    function getServerTransportAddr(nServer : TNWConnHandle; var ta : TNWTransportAddr) : boolean ;                         

    
  type
    TNWServer = class(TObject) 
    public
      function getLastError : string ;
      procedure resetLastError ;
  end;

var
  lasterror : TNWError ;

implementation

uses
  NWLib,
  NWTools ;

{$I nwserver.imp}
{$I nwstd.imp}

var
  nError : TNWError ;

{ ****************** Component functions ******************* }
function TNWServer.getLastError : string ;
  begin
    result := intToHex(lastError,sizeOf(TNWError)) ;
  end;

procedure TNWServer.resetLastError ;
  begin
    lastError := 0 ;
  end;


{ ********************* public functions ******************* }
function createSemaphore(nServer       : TNWConnHandle ;
                         semaphoreName : string  ;
                         maxInstances  : word ) : TNWSemaHandle  ;
  var
    openCount : word ;
    cpSema    : TObjName ;
  begin
    result := 0 ;
    if (@NWOpenSemaphore = nil) then
      @NWOpenSemaphore := getProcAddress(hNWCalls,'NWOpenSemaphore') ;
    if (@NWOpenSemaphore = nil) then
      exit ;
    if (@NWSignalSemaphore = nil) then
      @NWSignalSemaphore := getProcAddress(hNWCalls,'NWSignalSemaphore') ;
    if (@NWSignalSemaphore = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpSema,upperCase(semaphoreName)) ;
    nError := NWOpenSemaphore(nServer ,
                              @cpSema,
                              maxInstances,
                              @result,
                              @openCount) ;
    if (nError <> 0) then
      begin
        lastError := nError ;
        result := 0 ;
      end
    else if (maxInstances < 1) and
            (openCount = 1) then  {new semaphores must be signaled}
       NWSignalSemaphore(nServer,result) ;
  end;

function decSemaValue(nServer    : TNWConnhandle ;
                      semaHandle : TNWSemaHandle ;
                      amount     : word ) : boolean ;
  var
    nloop : word ;
  begin
    result := false ;
    if (@NWWaitOnSemaphore = nil) then
      @NWWaitOnSemaphore := getProcAddress(hNWCalls,'NWWaitOnSemaphore') ;
    if (@NWWaitOnSemaphore = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    for nloop := 1 to amount do begin
      nError := NWWaitOnSemaphore(nServer,semaHandle,0) ;  {0=timeout in ticks (1/18) }
      if (nError <> 0) then
        break ;
    end;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function incSemaValue(nServer    : TNWConnHandle ;
                      semaHandle : TNWSemaHandle ;
                      amount     : word ) : boolean ;
  var
    nloop : word ;
  begin
    result := false ;
    if (@NWSignalSemaphore = nil) then
      @NWSignalSemaphore := getProcAddress(hNWCalls,'NWSignalSemaphore') ;
    if (@NWSignalSemaphore = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    for nloop := 1 to amount do begin
      nError := NWSignalSemaphore(nServer,semaHandle) ;
      if (nError <> 0) then
        break ;
    end;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function freeSemaphore(nServer    : TNWConnHandle ;
                       semaHandle : TNWSemaHandle) : boolean ;
  begin
    result := false ;
    @NWCloseSemaphore := getProcAddress(hNWCalls,'NWCloseSemaphore') ;
    if (@NWCloseSemaphore = nil) then
      exit ;
    nError := 0 ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    if (semaHandle <> 0) then
      nError := NWCloseSemaphore(nServer,semaHandle) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function querySemaphore(nServer       : TNWConnHandle ;
                        semaHandle    : TNWSemaHandle ;
                        var semaValue : word) : word;
  var
    openCount : word ;
  begin
    result    := 0 ;
    if (@NWExamineSemaphore = nil) then
      @NWExamineSemaphore := getProcAddress(hNWCalls,'NWExamineSemaphore') ;
    if (@NWExamineSemaphore = nil) then
      exit ;
    openCount := 0 ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWExamineSemaphore(nServer,semaHandle,@semaValue,@openCount) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := openCount ;
  end;


function getServerHandleFromPath(cPath : string) : TNWConnHandle ;
  { Get Server Connection Handle from Path String }
  var
    nServer    : TNWConnHandle ;
    dirHandle  : TNWDirHandle  ;
    cpPath,
    cpRelative : TNWPath       ;
  begin
    result := 0 ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    strPCopy(cpPath,upperCase(cPath)) ;
    if (NWParseNetWarePath(@cpPath,
                           nServer, {out}
                           dirHandle, {out}
                           @cpRelative)=0) then
     result := nServer ;
  end;

function getUserStats(nServer : TNWConnHandle ;  {4.0 Only }
                      nConn   : TNWConnNumber ;
                      var connStats : TNWConnStats) : boolean ;
  { Return User Connection Stats }
  var
    cpUser           : TObjName ;
    internaluserInfo : NWFSE_USER_INFO ;
  begin
    result := false ;
    if (@NWGetUserInfo = nil) then
      @NWGetUserInfo := getProcAddress(hNWCalls,'NWGetUserInfo') ;
    if (@NWGetUserInfo = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    if (NWGetUserInfo(nServer,
                      nConn,
                      @cpUser,
                      internalUserInfo) = 0) then
      begin
        result := True ;
        with connStats do begin
          loginTime := encodeDate(word(internalUserInfo.userInfo.loginTime[0]),
                                  word(internalUserInfo.userInfo.loginTime[1]),
                                  word(internalUserInfo.userInfo.loginTime[2])) +
                       encodeTime(word(internalUserInfo.userInfo.loginTime[3]),
                                  word(internalUserInfo.userInfo.loginTime[4]),
                                  word(internalUserInfo.userInfo.loginTime[5]),
                                  word(internalUserInfo.userInfo.loginTime[6])) ;
          bytesRead      := NWHexToLong(internalUserInfo.userInfo.totalBytesRead) ;
          bytesWritten   := NWHexToLong(internalUserInfo.userInfo.totalBytesWritten) ;
          totalRequests  := internalUserInfo.userinfo.totalRequests     ;
          recordLocks    := internalUserInfo.userinfo.recordLockCount   ;
          fileLocks      := internalUserInfo.userinfo.fileLockCount     ;
          if (internalUserInfo.userInfo.expirationTime > 0) then
            expirationTime := now + (internalUserInfo.userInfo.expirationTime*18.6);
        end;
      end;
  end;

function getVolFileList(srchText : string) : TStringList ;
  { Retrieve Volume File Listing }
  var
    sequence    : TNWSequence     ;
    nServer     : TNWConnHandle   ;
    attrs       : TNWAttributes   ;
    dirHandle,
    outHandle   : TNWDirHandle    ;
    cpPath,
    cpRelative  : TNWPath         ;
    effRights   : TNWAccessRights ;
    srchPattern : TRetbuff        ;
    entryInfo   : NWENTRY_INFO    ;
  begin
    result := TStringList.create  ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    @NWAllocTemporaryDirectoryHandle := getProcAddress(hNWCalls,'NWAllocTemporaryDirectoryHandle') ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      exit ;
    @NWIntScanDirEntryInfo := getProcAddress(hNWCalls,'NWIntScanDirEntryInfo') ;
    if (@NWIntScanDirEntryInfo = nil) then
      exit ;
    attrs    := fa_normal or (fa_system or fa_hidden)  ;
    sequence := -1 ;
    srchText := upperCase(srchText) ;
    strPCopy(srchPattern,extractFileName(srchText))   ;
    strPCopy(cpPath,strTran(extractFilePath(srchText),'\','/')) ;
    fillChar(entryInfo,sizeOf(entryInfo),0) ;
    if (NWParseNetWarePath(@cpPath,
                           nServer,
                           dirHandle,
                           @cpRelative)=0) and
       (NWAllocTemporaryDirectoryHandle(nServer,
                                        0,
                                        @cpRelative,
                                        outHandle,
                                        @effRights)=0) then
      begin
        while ( NWIntScanDirEntryInfo(nServer,
                                      outHandle,
                                      attrs,
                                      @sequence,
                                      @srchPattern,
                                      entryInfo,
                                      0) = 0) do begin
          try
            result.add(copy(strPas(entryInfo.name),1,entryInfo.nameLength)) ;
          except
            on exception do
              break ;
          end;
        end;
      end;
  end;

function getVolDirList(srchText : string) : TStringList ;
  { Retrieve Volume Directory Listing }
  var
    sequence    : TNWSequence     ;
    nServer     : TNWConnHandle   ;
    attrs       : TNWAttributes   ;
    dirHandle,
    outHandle   : TNWDirHandle    ;
    cpPath,
    cpRelative  : TNWPath         ;
    effRights   : TNWAccessRights ;
    srchPattern : TRetbuff        ;
    entryInfo   : NWENTRY_INFO    ;
  begin
    result := TStringList.create  ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      @NWAllocTemporaryDirectoryHandle := getProcAddress(hNWCalls,'NWAllocTemporaryDirectoryHandle') ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      exit ;
    if (@NWIntScanDirEntryInfo = nil) then
      @NWIntScanDirEntryInfo := getProcAddress(hNWCalls,'NWIntScanDirEntryInfo') ;
    if (@NWIntScanDirEntryInfo = nil) then
      exit ;
    attrs    := ((fa_normal or fa_directory) or (fa_system or fa_hidden)) ;
    sequence := -1 ;
    srchText := upperCase(srchText) ;
    strPCopy(srchPattern,extractFileName(srchText))   ;
    strPCopy(cpPath,strTran(extractFilePath(srchText),'\','/')) ;
    fillChar(entryInfo,sizeOf(entryInfo),0) ;
    if (NWParseNetWarePath(@cpPath,
                           nServer,
                           dirHandle,
                           @cpRelative)=0) and
       (NWAllocTemporaryDirectoryHandle(nServer,
                                        0,
                                        @cpRelative,
                                        outHandle,
                                        @effRights)=0) then
      begin
        while ( NWIntScanDirEntryInfo(nServer,
                                      outHandle,
                                      attrs,
                                      @sequence,
                                      @srchPattern,
                                      entryInfo,
                                      0) = 0) do begin
          try
            result.add(copy(strPas(entryInfo.name),1,entryInfo.nameLength)) ;
          except
            on exception do
              break ;
          end;
        end;
      end;
  end;

function getFileInfo(fileName : string ; var fileInfo : TNWFileInfo) : boolean ;
  {same as getFileInfoOld, except works with DOS and other name spaces}
  var
    searchAttrs : TNWAttributes ;                            
    nServer     : TNWConnHandle ;
    effRights   : TNWAccessRights ;
    iteration   : longint ;
    dirHandle,
    outHandle   : TNWDirHandle ;
    cpRelative,
    cpPath      : TNWPath ;
    entryInfo   : NW_EXT_FILE_INFO ;
  begin
    result    := false ;
    iteration := -1 ;
    outHandle := 0 ;
    fillChar(entryInfo,sizeOf(entryInfo),0) ;
    if (@NWIntScanExtendedInfo = nil) then
      @NWIntScanExtendedInfo := getProcAddress(hNWCalls,'NWIntScanExtendedInfo') ;
    if (@NWIntScanExtendedInfo = nil) then
      exit ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      @NWAllocTemporaryDirectoryHandle := getProcAddress(hNWCalls,'NWAllocTemporaryDirectoryHandle') ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      exit ;
    if (@NWDeallocateDirectoryHandle = nil) then
      @NWDeAllocateDirectoryHandle := getProcAddress(hNWCalls,'NWDeallocateDirectoryHandle') ;
    if (@NWDeallocateDirectoryHandle = nil) then
      exit ;
      
    {fileName := upperCase(fileName) ;}
    strPCopy(cpPath,extractFilePath(fileName)) ;
    
    searchAttrs := ((FA_NORMAL or FA_HIDDEN) or FA_SYSTEM) ;
    if (NWParseNetWarePath(@cpPath,
                           nServer,
                           dirHandle,
                           @cpRelative)=0) and
       (NWAllocTemporaryDirectoryHandle(nServer,
                                        0,
                                        @cpRelative,
                                        outHandle,
                                        @effRights)=0) then
      try                 
        strPCopy(cpPath,extractFileName(fileName)) ;
        while (NWIntScanExtendedInfo(nServer,outHandle,searchAttrs,@iteration,
                                     @cpPath,entryInfo,0) = 0) do begin
          result := true ;                          
          with fileInfo.inheritedRights do begin
            read          := ((entryInfo.inheritedRightsMask and TR_READ) > 0) ;
            write         := ((entryInfo.inheritedRightsMask and TR_WRITE) > 0) ;
            open          := ((entryInfo.inheritedRightsMask and TR_OPEN) > 0) ;
            create        := ((entryInfo.inheritedRightsMask and TR_CREATE) > 0) ;
            canDelete     := ((entryInfo.inheritedRightsMask and TR_DELETE) > 0) ;
            owner         := ((entryInfo.inheritedRightsMask and TR_OWNER) > 0)  ;
            accessControl := ((entryInfo.inheritedRightsMask and TR_ACCESSCTRL) > 0) ;
            fileScan      := ((entryInfo.inheritedRightsMask and TR_FILESCAN) > 0) ;
            search        := ((entryInfo.inheritedRightsMask and TR_FILESCAN) > 0) ;
            modify        := ((entryInfo.inheritedRightsMask and TR_MODIFY) > 0) ;
            all           := ((entryInfo.inheritedRightsMask and TR_ALL) > 0) ;
            supervisor    := ((entryInfo.inheritedRightsMask and TR_SUPERVISOR) > 0) ;
            normal        := ((entryInfo.inheritedRightsMask and TR_NORMAL) > 0) ;
          end;
          fileInfo.lastModifyDate  := nwDateTimeToTDateTime(entryInfo.updateDateAndTime) ;
          fileInfo.attributes      := entryInfo.attributes   ;
          fileInfo.flags           := entryInfo.flags        ;
          fileInfo.nameSpace       := entryInfo.namespace    ;
          fileInfo.nameLength      := entryInfo.nameLength   ;
          fileInfo.name            := strPas(entryInfo.name) ;
          fileInfo.creationdate    := nwDateTimeToTDateTime(entryInfo.creationDateAndTime) ;
          fileInfo.ownerID         := getObjName(nServer,entryInfo.ownerID)                ;
          fileInfo.lastArchiveDate := nwDateTimeToTDateTime(entryInfo.lastArchiveDateAndTime)     ;
          fileInfo.lastArchivedBy  := getObjName(nServer,entryInfo.lastArchiverID)         ;
          fileInfo.lastAccessDate  := encodeDate( (entryInfo.lastAccessDate shr 9) + 80,
                                                  ((entryInfo.lastAccessDate AND $01E0) shr 5),
                                                  (entryInfo.lastAccessDate AND $001F)) ;
          fileInfo.lastModifyDate  := nwDateTimeToTDateTime(entryInfo.updateDateAndTime) ;
          fileInfo.updatedDateTime := nwDateTimeToTDateTime(entryInfo.updateDateAndTime) ;
          fileInfo.updatedBy       := getObjName(nServer,entryInfo.lastUpdatorID) ;
          fileInfo.filesize        := entryInfo.dataForkSize ;
        end;
      finally
        if (outHandle <> 0) then
          NWDeallocateDirectoryHandle(nServer,outHandle) ;
      end;
  end;

function getFileInfoOld(infile : string;
                     var fileInfo : TNWFileInfo) : boolean ;
  { Retrieve General File Information - DOS name space only }
  var
    nServer    : TNWConnHandle   ;
    augmentFlag: word            ;
    attrs      : TNWAttributes   ;
    dirHandle,
    outHandle  : TNWDirHandle    ;
    sequence   : TNWSequence     ;
    effRights  : TNWAccessRights ;
    cpInfile   : TRetBuff        ;
    cpPath,
    cpRelative : TNWPath         ;
    srchPattern: TRetBuff        ;
    entryInfo  : NWENTRY_INFO    ;
  begin
    result      := false ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      @NWAllocTemporaryDirectoryHandle := getProcAddress(hNWCalls,'NWAllocTemporaryDirectoryHandle') ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      exit ;
    if (@NWIntScanDirEntryInfo = nil) then
      @NWIntScanDirEntryInfo := getProcAddress(hNWCalls,'NWIntScanDirEntryInfo') ;
    if (@NWIntScanDirEntryInfo = nil) then
      exit ;
    if (@NWDeallocateDirectoryHandle = nil) then
      @NWDeAllocateDirectoryHandle := getProcAddress(hNWCalls,'NWDeallocateDirectoryHandle') ;
    if (@NWDeallocateDirectoryHandle = nil) then
      exit ;
    attrs       := 0     ;
    augmentFlag := 0 ;
    sequence    := -1    ;
    inFile      := upperCase(inFile) ;
    fillChar(entryInfo,sizeOf(entryInfo),0) ;
    strPCopy(srchPattern,extractFileName(infile)) ;
    strPCopy(cpInfile,extractFileName(infile))   ;
    strPCopy(cpPath,strTran(extractFilePath(infile),'\','/')) ;
    if (NWParseNetWarePath(@cpPath,
                           nServer,
                           dirHandle,
                           @cpRelative)=0) and
       (NWAllocTemporaryDirectoryHandle(nServer,
                                        0,
                                        @cpRelative,
                                        outHandle,
                                        @effRights)=0) then
      try
        while (NWIntScanDirEntryInfo(nServer,
                                     outHandle,
                                     attrs,
                                     @sequence,
                                     @srchPattern,
                                     entryInfo,
                                     augmentFlag) = 0) do begin
          if (strPos(strUpper(entryinfo.name),cpInFile) <> nil) then
            begin
              with fileInfo.inheritedRights do
                begin
                  read          := ((entryInfo.dir.inheritedRightsMask and TR_READ) > 0) ;
                  write         := ((entryInfo.dir.inheritedRightsMask and TR_WRITE) > 0) ;
                  open          := ((entryInfo.dir.inheritedRightsMask and TR_OPEN) > 0) ;
                  create        := ((entryInfo.dir.inheritedRightsMask and TR_CREATE) > 0) ;
                  canDelete     := ((entryInfo.dir.inheritedRightsMask and TR_DELETE) > 0) ;
                  owner         := ((entryInfo.dir.inheritedRightsMask and TR_OWNER) > 0)  ;
                  accessControl := ((entryInfo.dir.inheritedRightsMask and TR_ACCESSCTRL) > 0) ;
                  fileScan      := ((entryInfo.dir.inheritedRightsMask and TR_FILESCAN) > 0) ;
                  search        := ((entryInfo.dir.inheritedRightsMask and TR_FILESCAN) > 0) ;
                  modify        := ((entryInfo.dir.inheritedRightsMask and TR_MODIFY) > 0) ;
                  all           := ((entryInfo.dir.inheritedRightsMask and TR_ALL) > 0) ;
                  supervisor    := ((entryInfo.dir.inheritedRightsMask and TR_SUPERVISOR) > 0) ;
                  normal        := ((entryInfo.dir.inheritedRightsMask and TR_NORMAL) > 0) ;
                end;
            fileInfo.lastAccessDate  := encodeDate( (NWFILE_INFO(entryInfo.dir).lastAccessDate shr 9) + 80,
                                                    ((NWFILE_INFO(entryInfo.dir).lastAccessDate AND $01E0) shr 5),
                                                    (NWFILE_INFO(entryInfo.dir).lastAccessDate AND $001F)) ;
              
              fileInfo.lastModifyDate  := nwDateTimeToTDateTime(entryInfo.dir.lastModifyDateAndTime) ;
              fileInfo.updatedDateTime := nwDateTimeToTDateTime(NWFILE_INFO(entryInfo.dir).updateDateAndTime) ;
              fileInfo.updatedBy       := getObjName(nServer,NWFILE_INFO(entryInfo.dir).updatorID) ;
              fileInfo.filesize        := NWFILE_INFO(entryInfo.dir).fileSize ;
              fileInfo.maximumSpace    := entryInfo.dir.maximumSpace;
              fileInfo.attributes      := entryInfo.attributes   ;
              fileInfo.flags           := entryInfo.flags        ;
              fileInfo.nameSpace       := entryInfo.namespace    ;
              fileInfo.nameLength      := entryInfo.nameLength   ;
              fileInfo.name            := strPas(entryInfo.name) ;
              fileInfo.creationdate    := nwDateTimeToTDateTime(entryInfo.creationDateAndTime) ;
              fileInfo.ownerID         := getObjName(nServer,entryInfo.ownerID)                ;
              fileInfo.lastArchiveDate := nwDateTimeToTDateTime(entryInfo.lastArchiveDateAndTime)     ;
              fileInfo.lastArchivedBy  := getObjName(nServer,entryInfo.lastArchiverID)         ;
              result := true ;
              break;
            end;
        end;
      finally
        if (outHandle > 0) then
          NWDeallocateDirectoryHandle(nServer,outHandle) ;
      end;
  end;


function getDeletedFileInfo(inFile   : string;
                            var deletedFileInfo : TNWDeletedFileInfo) : boolean ;
  { Return Information About Deleted File }
  var
    nServer    : TNWConnHandle   ;
    dirHandle,
    outHandle  : TNWDirHandle    ;
    dirEntry   : TNWDirEntry     ;
    cpRelative : TNWPath         ;
    sequence   : TNWSequence     ;
    volume     : TNWVol          ;
    effRights  : TNWAccessRights ;
    deletedInfo: NWDELETED_INFO  ;
    cpInfile   : TRetBuff        ;
    cpPath     : TNWPath         ;
    ncursor    : TCursor         ;
  begin
    result    := false ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      @NWAllocTemporaryDirectoryHandle := getProcAddress(hNWCalls,'NWAllocTemporaryDirectoryHandle') ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      exit ;
    if (@NWIntScanDirEntryInfo = nil) then
      @NWIntScanDirEntryInfo := getProcAddress(hNWCalls,'NWIntScanDirEntryInfo') ;
    if (@NWIntScanDirEntryInfo = nil) then
      exit ;
    if (@NWScanForDeletedFiles = nil) then
      @NWScanForDeletedFiles := getProcAddress(hNWCalls,'NWScanForDeletedFiles') ;
    if (@NWScanForDeletedFiles = nil) then
      exit ;
    if (@NWDeallocateDirectoryHandle = nil) then
      @NWDeallocateDirectoryHandle := getProcAddress(hNWCalls,'NWDeallocateDirectoryHandle') ;
    if (@NWDeallocateDirectoryHandle = nil) then
      exit ;
    sequence  := -1    ;
    ncursor   := screen.cursor     ;
    screen.cursor := crHourglass   ;
    strPCopy(cpPath,strTran(extractFilePath(infile),'\','/')) ;
    strPCopy(cpInfile,extractFileName(infile))   ;
    if (NWParseNetWarePath(@cpPath,
                           nServer,
                           dirHandle,
                           @cpRelative)=0) and
       (NWAllocTemporaryDirectoryHandle(nServer,
                                        0,
                                        @cpRelative,
                                        outHandle,
                                        @effRights)=0) then
      try
        while (NWScanForDeletedFiles(nServer,
                                     outHandle,
                                     @sequence,
                                     @volume,
                                     @dirEntry,
                                     deletedInfo) = 0) do begin
          if (strPos(deletedinfo.name,cpInFile) <> nil) then
            begin
              { Fill Our Interface Structure }
              with deletedFileInfo do begin
                attributes       := deletedInfo.attributes ;
                flags            := deletedInfo.flags      ;
                nameSpace        := deletedInfo.nameSpace  ;
                nameLength       := deletedInfo.nameLength ;
                name             := strPas(deletedInfo.name) ;
                creationdate     := nwDateTimeToTDateTime(deletedinfo.creationDateAndTime) ;
                ownerID          := getObjName(nServer,deletedInfo.ownerID) ;
                updateDateTime   := nwDateTimeToTDateTime(deletedinfo.updateDateAndTime) ;
                updatedBy        := getObjName(nServer,deletedInfo.updatorID) ;
                fileSize         := deletedInfo.fileSize         ;
                with inheritedRights do
                  begin
                    read          := ((deletedInfo.inheritedRightsMask and TR_READ) > 0)   ;
                    write         := ((deletedInfo.inheritedRightsMask and TR_WRITE) > 0)  ;
                    open          := ((deletedInfo.inheritedRightsMask and TR_OPEN) > 0)   ;
                    create        := ((deletedInfo.inheritedRightsMask and TR_CREATE) > 0) ;
                    canDelete     := ((deletedInfo.inheritedRightsMask and TR_DELETE) > 0) ;
                    owner         := ((deletedInfo.inheritedRightsMask and TR_OWNER)  > 0) ;
                    accessControl := ((deletedInfo.inheritedRightsMask and TR_ACCESSCTRL) > 0) ;
                    fileScan      := ((deletedInfo.inheritedRightsMask and TR_FILESCAN) > 0)   ;
                    search        := ((deletedInfo.inheritedRightsMask and TR_FILESCAN) > 0)   ;
                    modify        := ((deletedInfo.inheritedRightsMask and TR_MODIFY) > 0) ;
                    all           := ((deletedInfo.inheritedRightsMask and TR_ALL) > 0)    ;
                    supervisor    := ((deletedInfo.inheritedRightsMask and TR_SUPERVISOR) > 0);
                    normal        := ((deletedInfo.inheritedRightsMask and TR_NORMAL) > 0);
                end;
                lastAccessDate   := nwDateTimeToTDateTime(longSwap(
                                            longInt(deletedinfo.lastAccessDate))) ;
                deletedDateTime  := nwDateTimeToTDateTime(deletedinfo.deletedDateAndTime) ;
                deletedBy        := getObjName(nServer,deletedInfo.deletorID) ;
                lastArchiveDate  := nwDateTimeToTDateTime(deletedInfo.lastArchiveDateAndTime);
                lastArchivedBy   := getObjName(nServer,deletedInfo.lastArchiverID) ;
              end;
              result := true ;
              break ;
            end;
        end;
      finally  
        NWDeallocateDirectoryHandle(nServer,outHandle) ;
      end;
    screen.cursor := ncursor ;
  end;

function purgeFile(nServerHandle : TNWConnHandle; fileName : string) : boolean ;
  var
    nloop   : integer ;
    delList : TStringList ;
    ncursor : TCursor ;
    cPath   : string  ;
  begin
    result   := false ;
    ncursor  := screen.cursor ;
    fileName := upperCase(fileName) ;
    cPath    := extractFilePath(fileName) ;
    fileName := extractFileName(fileName) ;
    try
      if (@NWPurgeDeletedFile = nil) then
        @NWPurgeDeletedFile := getProcAddress(hNWCalls,'NWPurgeDeletedFile') ;
      if (@NWPurgeDeletedFile = nil) then
        exit ;
      if (nServerHandle = 0) then
        nServerHandle := getPrimaryServerID ;
      screen.cursor := crHourglass   ;
      delList       := getDeletedFileList(nServerHandle, cPath) ;
      for nloop := 0 to (delList.count -1) do
        with pTNWDelFileInfo(delList.objects[nloop])^ do 
          if (strPas(strUpper(pFileName)) = fileName) then
            begin
              nError := NWPurgeDeletedFile(nServerHandle,
                                           dirHandle,
                                           iterHandle,
                                           volNum,
                                           dirBase,
                                           @pFileName) ;
              if (nError <> 0) then
                lastError := nError 
              else 
                result := true ;
              break ;  
            end;
        delList.free ;
    finally
      screen.cursor := ncursor ;
    end;
  end;

function purgeSingleFile(deletedFileInfo : TNWDelFileInfo) : boolean ;
  begin
    result := false ;
    if (@NWPurgeDeletedFile = nil) then
      @NWPurgeDeletedFile := getProcAddress(hNWCalls,'NWPurgeDeletedFile') ;
    if (@NWPurgeDeletedFile = nil) then
      exit ;
    with deletedFileInfo do 
      nError := NWPurgeDeletedFile(nServer,
                                   dirHandle,
                                   iterHandle,
                                   volNum,
                                   dirBase,
                                   @pFileName) ;
    if (nError <> 0) then
      lastError := nError 
    else 
      result := true ;
  end;

function purgeAllFiles(nServerHandle : TNWConnHandle; cpath : string) : boolean ;
  { immediately kill all deleted files in cpath}
  var
    nloop   : integer ;
    delList : TStringList ;
    nServer : TNWConnHandle ;
    ncursor : TCursor ;
  begin
    result  := false ;
    ncursor := screen.cursor ;
    try
      if (@NWPurgeDeletedFile = nil) then
        @NWPurgeDeletedFile := getProcAddress(hNWCalls,'NWPurgeDeletedFile') ;
      if (@NWPurgeDeletedFile = nil) then
        exit ;
      if (nServerHandle = 0) then
        nServerHandle := getPrimaryServerID ;
      {$IFDEF Use this if you need Netware 2.x-only support}
        if (@NWPurgeErasedFiles = nil) then
          @NWPurgeErasedFiles := getProcAddress(hNWCalls,'NWPurgeErasedFiles') ;
        if (@NWPurgeErasedFiles = nil) then
          exit ;
        nError := (NWPurgeErasedFiles(nServerHandle) ;
        if (nError <> 0) then
          lastError := nError
        else
          result := true ;
        exit ;
      {$ENDIF}
      screen.cursor := crHourglass   ;
      nserver := nServerHandle ;
      delList := getDeletedFileList(nServer, cPath) ;
      for nloop := 0 to (delList.count -1) do
        with pTNWDelFileInfo(delList.objects[nloop])^ do begin
          nError := NWPurgeDeletedFile(nServerHandle,
                                       dirHandle,
                                       iterHandle,
                                       volNum,
                                       dirBase,
                                       @pFileName) ;
          if (not result) and (nError <> 0) then
            begin
              lastError := nError ;
              exit ;
            end
          else if (nError <> 0) then
            exit
          else
            result := true ;
        end;
        delList.free ;
    finally
      screen.cursor := ncursor ;
    end;
  end;

function getDeletedFileList(nServerHandle : TNWConnHandle;
                            cpath : string) : TStringList ;
  { Return a TStringList w/objects of Deleted Files/Info in Directory }
  var
    dirHandle,
    outHandle    : TNWDirHandle ;
    dirEntry     : TNWDirEntry  ;
    sequence     : TNWSequence  ;
    volume       : TNWVol       ;
    effRights    : TNWAccessRights ;
    cpPath,
    cpRelative   : TNWPath         ;
    deletedInfo  : NWDELETED_INFO ;
    pDelFileInfo : ^TNWDelFileInfo ;
  begin
    result := TStringList.Create ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      @NWAllocTemporaryDirectoryHandle := getProcAddress(hNWCalls,'NWAllocTemporaryDirectoryHandle') ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      exit ;
    if (@NWScanForDeletedFiles = nil) then
      @NWScanForDeletedFiles := getProcAddress(hNWCalls,'NWScanForDeletedFiles') ;
    if (@NWScanForDeletedFiles = nil) then
      exit ;
    if (@NWDeallocateDirectoryHandle = nil) then
      @NWDeallocateDirectoryHandle := getProcAddress(hNWCalls,'NWDeallocateDirectoryHandle') ;
    if (@NWDeallocateDirectoryHandle = nil) then
      exit ;
    if (nServerHandle = 0) then
      nServerHandle := getPrimaryServerID ;
    sequence := -1 ;
    strPCopy(cpPath,strTran(upperCase(cPath),'\','/')) ;
    if (NWParseNetWarePath(@cpPath,
                           nServerHandle,
                           dirHandle,
                           @cpRelative)=0) and
       (NWAllocTemporaryDirectoryHandle(nServerHandle,
                                        0,
                                        @cpRelative,
                                        outHandle,
                                        @effRights)=0) then
      try
        while (NWScanForDeletedFiles(nServerHandle,
                                     outHandle,
                                     @sequence,
                                     @volume,
                                     @dirEntry,
                                     deletedInfo) = 0) do begin
          try
            new(pDelFileInfo) ;
            with pDelFileInfo^ do begin
              nServer    := nServerHandle ;
              dirHandle  := outHandle ;
              iterHandle := sequence  ;
              volNum     := volume    ;
              dirBase    := dirEntry  ;
              strLCopy(pFileName,deletedInfo.name,deletedInfo.nameLength) ;
            end;

            {string[x] = file name.  objects[x] = TNWDelFileInfo pointer}
            result.addObject(
                copy(strPas(deletedinfo.name),1,deletedInfo.nameLength), {yes, you hafta do it this way.}
                TObject(pDelFileInfo)) ;
          except
            on exception do
              break ;
          end;
        end;
      finally
        NWDeallocateDirectoryHandle(nServerHandle,outHandle) ;
      end;
  end;

function salvage(nServer : TNWConnHandle;
                 inFile   : string ;
                 outFile  : string) : boolean ;
  { Restore Deleted File }
  var
    dirHandle  : TNWDirHandle    ;
    outHandle  : TNWDirHandle    ;
    dirEntry   : TNWDirEntry     ;
    cpRelative : TNWPath         ;
    sequence   : TNWSequence     ;
    volume     : TNWVol          ;
    deletedInfo: NWDELETED_INFO  ;
    effRights  : TNWAccessRights ;
    cpInfile   : TRetBuff        ;
    cpOutfile  : TRetBuff        ;
    cpPath     : TNWPath         ;
  begin
    result := false ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      @NWAllocTemporaryDirectoryHandle := getProcAddress(hNWCalls,'NWAllocTemporaryDirectoryHandle') ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      exit ;
    if (@NWScanForDeletedFiles = nil) then
      @NWScanForDeletedFiles := getProcAddress(hNWCalls,'NWScanForDeletedFiles') ;
    if (@NWScanForDeletedFiles = nil) then
      exit ;
    if (@NWRecoverDeletedFile = nil) then
      @NWRecoverDeletedFile := getProcAddress(hNWCalls,'NWRecoverDeletedFile') ;
    if (@NWRecoverDeletedFile = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    sequence  := -1    ;
    strPCopy(cpOutFile,extractFileName(outfile)) ;
    strPCopy(cpPath,strTran(extractFilePath(infile),'\','/')) ;
    strPCopy(cpInfile,extractFileName(infile))   ;
    if (NWParseNetWarePath(@cpPath,
                           nServer,
                           dirHandle,
                           @cpRelative)=0) and
       (NWAllocTemporaryDirectoryHandle(nServer,
                                        0,
                                        @cpRelative,
                                        outHandle,
                                        @effRights)=0) then
      begin
        while (NWScanForDeletedFiles(nServer,
                                     outHandle,
                                     @sequence,
                                     @volume,
                                     @dirEntry,
                                     deletedInfo) = 0) do begin
          if (strPos(deletedinfo.name,cpInFile) <> nil) then
            begin
              result := (NWRecoverDeletedFile(nServer,
                                              outHandle,
                                              sequence,
                                              volume,
                                              dirEntry,
                                              @cpInFile,
                                              @cpOutFile)=0) ;
              break ;
            end;
        end;
      end;
  end;



function getServerStats(nServer : TNWConnHandle; { Some Items Netware 4.0 Only }
                        var serverStats : TNWServerInfo) : boolean ;
  var
    ttsLevel,
    sftLevel          : TNWSupportLevel ;
    maxConns,
    connsInUse,
    maxConnsUsed,
    numVolumes        : word            ;
    majorVer,
    minorVer,
    revision          : TNWServiceVer   ;
    serverName        : TObjName        ;
    fileServerInfo    : NWFSE_FILE_SERVER_INFO ;
  begin
    result := false ;
    if (@NWGetFileServerInfo = nil) then
      @NWGetFileServerInfo := getProcAddress(hNWCalls,'NWGetFileServerInfo') ;
    if (@NWGetFileServerInfo = nil) then
      exit ;
    if (@NWGetFileServerInformation = nil) then
      @NWGetFileServerInformation := getProcAddress(hNWCalls,'NWGetFileServerInformation') ;
    if (@NWGetFileServerInformation = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    if (NWGetFileServerInformation(nServer,
                                   @serverName,
                                   @majorVer,
                                   @minorVer,
                                   @revision,
                                   @maxConns,
                                   @maxConnsUsed,
                                   @connsInUse,
                                   @numVolumes,
                                   @sftLevel,
                                   @ttsLevel) = 0) then
      begin
        if (majorVer > 3) then
          result := (NWGetFileServerInfo(nServer,FileServerInfo) = 0)
        else
          result := true ; { cannot do serverstats under 3.x, 2.x }

        { Integrate into our Server Info Structure }
        serverStats.serverUpTime := fileServerInfo.serverTimeAndVConsoleInfo.currentServerTime ;
        serverStats.processor    := 0 ;
        serverStats.utilization  := fileServerInfo.serverUtilization ;
        serverStats.totalmemory  := 0 ; {// netware 2.2 only?!?!}
        serverStats.freememory   := 0 ; {// ditto - no api call found.}
        serverStats.serverName   := strPas(serverName) ;
        serverStats.version      := intToStr(majorVer) + '.' +
                                    intToStr(minorVer) ;
        serverStats.maxConns     := maxConns     ;
        serverStats.connsinUse   := connsInUse   ;
        serverStats.maxConnsUsed := maxConnsUsed ;
        serverStats.numVolumes   := numVolumes   ;
        serverStats.sftLevel     := sftLevel     ;
        serverStats.ttsLevel     := ttsLevel     ;
        serverStats.packetsIn    := fileServerInfo.fileServerCounters.totalPacketsServiced ;
        serverStats.packetsOut   := fileServerInfo.fileServerCounters.totalPacketsRouted ;
      end;
  end;


function getServerInformation(nServer : TNWConnHandle; { Netware 4.x Only }
                              var serverStats : NWFSE_FILE_SERVER_INFO) : boolean ;
  begin
    {same as getServerStats, but returns Native-Novell structures}
    result := false ;
    if (@NWGetFileServerInfo = nil) then
      @NWGetFileServerInfo := getProcAddress(hNWCalls,'NWGetFileServerInfo') ;
    if (@NWGetFileServerInfo = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWGetFileServerInfo(nServer,serverStats) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function getCacheInfo(nServer : TNWConnHandle;  { Netware 4.0 Only }
                      var cacheInfo : TNWMemCacheInfo) : boolean ;
  var
    fseCacheInfo : NWFSE_CACHE_INFO  ;
  begin
    result := false ;
    @NWGetCacheInfo := getProcAddress(hNWCalls,'NWGetCacheInfo') ;
    if (@NWGetCacheInfo = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWGetCacheInfo(nServer, fseCacheInfo) ;
    if (nError <> 0) then
      lastError := nError
    else
      with cacheInfo do begin
        result := true ;
        serverUpTime             := fseCacheInfo.serverTimeAndVConsoleInfo.currentServerTime  ;
        writeBlockCount          := fseCacheInfo.cacheCounters.writeBlockCount   ;
        diskWriteCount           := fseCacheInfo.cacheCounters.diskWriteCount    ;
        writeErrorCount          := fseCacheInfo.cacheCounters.writeErrorCount   ;
        numCacheHits             := fseCacheInfo.cacheTrendCounters.numCacheHits ;
        numDirtyCacheHits        := fseCacheInfo.cacheTrendCounters.numDirtyCacheHits ;
        lruSittingTime           := fseCacheInfo.cacheTrendCounters.lruSittingTime   ;
        cacheDirtyWaitTime       := fseCacheInfo.cacheInformation.cacheDirtyWaitTime ;
        cacheMaxConcurrentWrites := fseCacheInfo.cacheInformation.cacheMaxConcurrentWrites ;
        maxDirtyTime             := fseCacheInfo.cacheInformation.maxDirtyTime             ;
        DirCacheBuffers          := fseCacheInfo.cacheInformation.numOfDirCacheBuffers     ;
        maxByteCount             := fseCacheInfo.cacheInformation.maxByteCount             ;
        minCacheBuffers          := fseCacheInfo.cacheInformation.minNumofCacheBuffers     ;
        originalCacheBuffers     := fseCacheInfo.cacheMemCounters.originalNumOfCacheBuffers ;
        currentCacheBuffers      := fseCacheInfo.cacheMemCounters.currentNumOfCacheBuffers  ;
      end;
  end;


function setServerDateTime(nServer : TNWConnHandle ; dt : TDateTime) : boolean ;
  { Set Server's Date and Time }
  var
    nyear,nmonth,nday,
    nhour,nminute,nsecond,nMsec : word ;
  begin
    result := false ;
    if (@NWSetFileServerDateAndTime = nil) then
      @NWSetFileServerDateAndTime := getProcAddress(hNWCalls,'NWSetFileServerDateAndTime') ;
    if (@NWSetFileServerDateAndTime = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    decodeDate(dt,nYear,nMonth,nDay) ;
    decodeTime(dt,nhour,nMinute,nSecond,nMsec) ;
    if (nyear > 1900) then
      nyear := (nyear - 1900) ; {0-79=21st century. 80-99=20th century}
    nError := NWSetFileServerDateAndTime(nServer,
                                         byte(nYear),
                                         byte(nMonth),
                                         byte(nDay),
                                         byte(nHour),
                                         byte(nMinute),
                                         byte(nSecond)) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function disableLogins(nServer : TNWConnHandle) : boolean ;
  begin
    result := false ;
    if (@NWDisableFileServerLogin = nil) then
      @NWDisableFileServerLogin := getProcAddress(hNWCalls,'NWDisableFileServerLogin') ;
    if (@NWDisableFileServerLogin = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    if NWDisableFileServerLogin(nServer) = 0 then
      Result := True ;
  end;

function enableLogins(nServer : TNWConnHandle) : boolean ;
  begin
    result := false ;
    if (@NWEnableFileServerLogin = nil) then
      @NWEnableFileServerLogin := getProcAddress(hNWCalls,'NWEnableFileServerLogin') ;
    if (@NWEnableFileServerLogin = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    if NWEnableFileServerLogin(nServer) = 0 then
      Result := True ;
  end;

function downServer(nServer : TNWConnHandle) : boolean ;
  begin
    result := false ;
    if (@NWDownFileServer = nil) then
      @NWDownFileServer := getProcAddress(hNWCalls,'NWDownFileServer') ;
    if (@NWDownFileServer = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    if NWDownFileServer(nServer,0) = 0 then
      Result := True ;
  end;

function NWLogin(server,
                 userID,
                 password : string ;
                 objType  : TObjType) : boolean ;
  var
    nConn      : TNWConnHandle ;
    cpServer,
    cpUserID,
    cpPassword : TRetBuff       ;
  begin
    result  := false ;
    if (@NWAttachToFileServer = nil) then
      @NWAttachToFileServer := getProcAddress(hNWCalls,'NWAttachToFileServer') ;
    if (@NWAttachToFileServer = nil) then
      exit ;
    if (@NWLoginToFileServer = nil) then
      @NWLoginToFileServer := getProcAddress(hNWCalls,'NWLoginToFileServer') ;
    if (@NWLoginToFileServer = nil) then
      exit ;
    if (objType < 1) then
      objType := nw_user ;
    strPCopy(cpServer,upperCase(server)) ;
    strPCopy(cpUserID,upperCase(userID)) ;
    strPCopy(cpPassword,upperCase(password)) ;
    nError := 0 ;
    nConn  := getServerHandle(server) ;
    if (nConn < 1) then
      nError := NWAttachToFileServer(@cpServer,0,nConn) ;
    if (nError = 0) or
       (nError = NWERROR_ALREADY_ATTACHED) then
      begin
        nError := NWLoginToFileServer(nConn,
                                      @cpUserID,
                                      objType,
                                      @cpPassword) ;
        if (nError <> 0) and
           (nError <> NWERROR_ALREADY_ATTACHED) then
          lastError := nError
        else
          result := true ;
      end;
  end;

function NWLogout(nServer : TNWConnHandle) : boolean ;
  begin
    result := false ;
    if (@NWLogoutFromFileServer = nil) then
      @NWLogoutFromFileServer := getProcAddress(hNWCalls,'NWLogoutFromFileServer') ;
    if (@NWLogoutFromFileServer = nil) then
      exit ;
    if (@NWDetachFromFileServer = nil) then
      @NWDetachFromFileServer := getProcAddress(hNWCalls,'NWDetachFromFileServer') ;
    if (@NWDetachFromFileServer = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWLogoutFromFileServer(nServer) ;
    if (nError = 0) then
      nError := NWDetachFromFileServer(nServer) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function getServerSerial(cserver : string) : TNWSerial ;
  { get a server's serial number }
  var
    appno   : word ;
    nSerial : TNWSerial ;
  begin
    result := 0 ;
    if (@NWGetNetworkSerialNumber = nil) then
      @NWGetNetworkSerialNumber := getProcAddress(hNWCalls,'NWGetNetworkSerialNumber') ;
    if (@NWGetNetworkSerialNumber = nil) then
      exit ;
    if (@NWLongSwap = nil) then
      @NWLongSwap := getProcAddress(hNWCalls,'NWLongSwap') ;
    if (@NWLongSwap = nil) then
      exit ;
    nError := NWGetNetworkSerialNumber(
                          getServerHandle(cserver),
                          @nSerial,
                          @appno) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := NWLongSwap(nSerial) ;
  end;

function getServerHandle(cServer : string) : TNWConnHandle ;
  { get a server's connection handle }
  var
    nServer   : TNWConnHandle ;
    cpServer  : TObjName ;
  begin
    result  := 0 ;
    nServer := 0 ;
    if (@NWCCOpenConnByName = nil) then
      @NWCCOpenConnByName := getProcAddress(hNWClx,'NWCCOpenConnByName') ;
    if (@NWGetConnectionHandle = nil) then
      @NWGetConnectionHandle := getProcAddress(hNWCalls,'NWGetConnectionHandle') ;
    strPCopy(cpServer,upperCase(cServer)) ;
    if (@NWCCOpenConnByName <> nil) then
      nError := NWCCOpenConnByName(0,@cpServer,
                                 NWCC_NAME_FORMAT_BIND,
                                 NWCC_OPEN_LICENSED,
                                 0,
                                 @nServer) ;
    {try old pre 1/97 call on any type of failure}
    if ((nError <> 0) or (@NWCCOpenConnByName = nil) or (nServer = 0)) and  
       (@NWGetConnectionHandle <> nil) then
      nError  := NWGetConnectionHandle(@cpServer,
                                       0,
                                       nServer,
                                       nil) ;  
    if (nError <> 0) then
      lastError := nError
    else
      result := nServer;
  end;

function getFileSysStats(nServer : TNWConnHandle ;
  { Return File System Statistics - Netware 2.2 Only }
                         var FileSysInfo : FILESYS_STATS) : boolean ;
  begin
    result := false ;
    if (@NWGetFileSystemStats = nil) then
      @NWGetFileSystemStats := getProcAddress(hNWCalls,'NWGetFileSystemStats') ;
    if (@NWGetFileSystemStats = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWGetFileSystemStats(nServer, fileSysInfo);
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function getDiskCacheStats(nServer : TNWConnHandle;
                           var cacheInfo : DSK_CACHE_STATS) : boolean ;
  { Full Disk Cache Statistics - Netware 2.2 Only }
  begin
    result := false ;
    if (@NWGetDiskCacheStats = nil) then
      @NWGetDiskCacheStats := getProcAddress(hNWCalls,'NWGetDiskCacheStats') ;
    if (@NWGetDiskCacheStats = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWGetDiskCacheStats(nServer,cacheinfo) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function volInfo(nServer : TNWConnHandle ;
                 volName  : string ;
                 var volumeInfo : VOL_STATS) : boolean ;
  { Return Structure with Volume Information }
  var
    volNum    : nuint8 ;
    cpVolName : TObjName ;
  begin
    result := false ;
    if (@NWGetVolumeNumber = nil) then
      @NWGetVolumeNumber := getProcAddress(hNWCalls,'NWGetVolumeNumber') ;
    if (@NWGetVolumeNumber = nil) then
      exit ;
    if (@NWGetVolumeStats = nil) then
      @NWGetVolumeStats := getProcAddress(hNWCalls,'NWGetVolumeStats') ;
    if (@NWGetVolumeStats = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpVolName,volName) ;
    nError := NWGetVolumeNumber(nServer,@cpVolName,@volNum) ;
    if (nError <> 0) then
      lastError := nError
    else
      begin
        nError := NWGetVolumeStats(nServer, volNum, volumeInfo) ; {Netware 2.2 only}
        if (nError <> 0) then
          begin
            @NWGetVolumeInfoWithNumber := getProcAddress(hNWCalls,'NWGetVolumeInfoWithNumber');
            if (@NWGetVolumeInfoWithNumber = nil) then
              exit ;
            with volumeInfo do begin
              nError := NWGetVolumeInfoWithNumber(nServer,volNum,@cpVolName,
                                                  @totalBlocks,
                                                  @sectorsPerBlock,
                                                  @availableBlocks,
                                                  @totalDirectorySlots,
                                                  @availableDirectorySlots,
                                                  @isRemovable) ;
              volumeNumber := volNum ;
              strPCopy(volumeName,strPas(cpVolName)) ;
            end;
            if (nError <> 0) then
              lastError := nError
            else
              result := true
          end
        else
          result := true ;
      end;
  end;

function getVolumes(nServer : TNWConnHandle) : TStringList ;
  { get List of Netware Volumes }
  var
    volNum  : TNWVolNum  ;
    volName : TObjName ;
  begin
    result := TStringList.create ;
    if (@NWGetVolumeName = nil) then
      @NWGetVolumeName := getProcAddress(hNWCalls,'NWGetVolumeName') ;
    if (@NWGetVolumeName = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    volNum := 0 ;
    while (NWGetVolumeName(nServer,volNum,@volName) = 0) do begin
      if (strLen(volName) > 0) then
        result.addObject(volName,TObject(volNum)) ;
      inc(volNum) ;
    end;
  end;

function NWDateTimeToTDateTime(datein : TNWDateTime) : TDateTime ;
  { Novell Packed DateTime to TDateTime }
  var
    ndate : NW_DATE ;
    ntime : NW_TIME ;
  begin
    result := encodeDate(1980,1,1) ;
    if (@NWUnpackDateTime = nil) then
      @NWUnpackDateTime := getProcAddress(hNWCalls,'NWUnpackDateTime') ;
    if (@NWUnpackDateTime = nil) then
      exit ;
    with ndate do begin
      year  := 1980 ;
      month := 1 ;
      day   := 1 ;
    end;
    with ntime do begin
      hours   := 12 ;
      minutes := 0 ;
      seconds := 0 ;
    end;
    NWUnpackDateTime(datein,ndate,ntime) ;
    if ((ndate.year > 1979) and (ndate.year < 2112)) and
       ((ndate.month > 0) and (ndate.month < 13)) and
       ((ndate.day > 0) and (ndate.day <= lastDay(encodeDate(ndate.year,ndate.month,1))) ) then
      result := encodeDate(ndate.year,
                           ndate.month,
                           ndate.day) ;
    if (ntime.hours < 25) and
       (ntime.minutes < 60) and
       (ntime.seconds < 60) then
      result := result + encodeTime(ntime.hours,
                                    ntime.minutes,
                                    ntime.seconds,
                                    0) ;
  end;


function serverLoginOK(nServer : TNWConnHandle) : boolean ;
  {Are server Logins Disabled?}
  var
    loginOK : uint8 ;
  begin
    result := false ;
    if (@NWGetFileServerLoginStatus = nil) then
      @NWGetFileServerLoginStatus := getProcAddress(hNWCalls,'NWGetFileServerLoginStatus') ;
    if (@NWGetFileServerLoginStatus = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWGetFileServerLoginStatus(nServer,@loginOK) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := (loginOK <> 0) ;
  end;

function attach(serverName : string) : TNWConnHandle ;
  var
    connID : TNWConnHandle ;
    cpServerName : TObjName ;
  begin
    result := 0 ;
    connID := 0 ;
    strPCopy(cpServerName,upperCase(serverName)) ;
    if (@NWCCOpenConnByName = nil) then
      @NWCCOpenConnByName := getProcAddress(hNWClx,'NWCCOpenConnByName') ;
    if (@NWCCOpenConnByName <> nil) then
      nError := NWCCOpenConnByName(0,
                                   @cpServerName,
                                   NWCC_NAME_FORMAT_BIND,
                                   NWCC_OPEN_LICENSED,
                                   NWCC_RESERVED,
                                   @connID)
    else {pre 1/97 - obsolete function}
      begin
        @NWAttachToFileServer := getProcAddress(hNWCalls,'NWAttachToFileServer') ;
        if (@NWAttachToFileServer <> nil) then
          nError := NWAttachToFileServer(@cpServerName,0,connID) ;
      end;
    if (connID <> 0) or (nError = 0) or (nError = NWERROR_ALREADY_ATTACHED) then
      result := connID
    else
      lastError := nError ;
  end;

function getDiskUtilization(nServer : TNWConnHandle ;
                            userID  : string ;
                            volNum  : byte ;
                            var diskStats : TNWDiskUsageStats) : boolean ;
  var
    nDir,
    nfiles,
    nBlocks    : word ;
    nObjID     : TObjID ;
    nObjServer : TNWConnHandle ;
  begin
    result := false ;
    if (@NWGetDiskUtilization = nil) then
      @NWGetDiskUtilization := getProcAddress(hNWCalls,'NWGetDiskUtilization') ;
    if (@NWGetDiskUtilization = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nObjServer := nServer ;
    nObjID := getObjNumber(nObjServer,userID,nw_user) ;
    nError := NWGetDiskUtilization(nServer,
                                   nObjID,
                                   volNum,
                                   @nDir,
                                   @nfiles,
                                   @nblocks) ;

    if (nError <> 0) then
      lastError := nError
    else
      begin
        result := true ;
        with diskStats do begin
          usedDirectories := nDir ;
          usedFiles       := nFiles ;
          usedBlocks      := nblocks ;
        end;
      end;
  end;

function getDirTrustees(nServer : TNWConnHandle ; cPath : string) : TStringList ;
  var
    nloop,
    augmentFlag,
    nEntries    : word ;
    nRights     : TNWRightsMask ;
    iteration   : TNWSequence ;
    dirHandle   : TNWDirHandle ;
    cRights,
    cUserName   : string ;
    cpRelative,
    cpPath      : TNWPath   ;
    trusteeInfo : NWET_INFO ;
  begin
    result := TStringList.create ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWIntScanForTrustees = nil) then
      @NWIntScanForTrustees := getProcAddress(hNWCalls,'NWIntScanForTrustees') ;
    if (@NWIntScanForTrustees = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    iteration := 0 ;     
    augmentFlag := 0 ;
    strPCopy(cpPath,cPath) ;
    if (NWParseNetWarePath(@cpPath,
                           nServer,
                           dirHandle,
                           @cpRelative)=0) then
      begin
        while true do begin
          nError := NWIntScanForTrustees(nServer,
                                         dirHandle,
                                         @cpRelative,
                                         @iteration,
                                         @nEntries,
                                         trusteeInfo,
                                         augmentFlag) ;
          if (nError <> 0) then
            begin
              lastError := nError ;
              break ;
            end
          else
            begin
              for nloop := 0 to (nEntries-1) do begin
                cUserName := getObjName(nServer,trusteeInfo.trusteeList[nloop].objectID) ;
                cRights   := '' ;
                nRights   := trusteeInfo.trusteeList[nloop].objectRights ;
                if (length(cUserName) > 0) then
                  begin
                    if ((nRights and TA_ALL) > 0) then
                      cRights := 'SFRWCDMA'
                    else
                      begin
                        crights := '-' ;
                        if ((nRights and TA_SEARCH) > 0) then
                          cRights := cRights + 'F'
                        else
                          cRights := cRights + '-';
                        if ((nRights and TA_READ) > 0) then
                          cRights := cRights + 'R'
                        else
                          cRights := cRights + '-';
                        if ((nRights and TA_WRITE) > 0) then
                          cRights := cRights + 'W'
                        else
                          cRights := cRights + '-';
                        if ((nRights and TA_CREATE) > 0) then
                          cRights := cRights + 'C'
                        else
                          cRights := cRights + '-';
                        if ((nRights and TA_DELETE) > 0) then
                          cRights := cRights + 'D'
                        else
                          cRights := cRights + '-';
                        if ((nRights and TA_MODIFY) > 0) then
                          cRights := cRights + 'M'
                        else
                          cRights := cRights + '-';
                        if ((nRights and TA_OWNERSHIP) > 0) then
                          cRights := cRights + 'A'
                        else
                          cRights := cRights + '-';
                      end;
                    result.add(cUserName + '  [' + cRights + ']') ;
                  end;
              end;
            end;
        end;
      end;
  end;

function getVolBlockInfo(nServer : TNWConnHandle ;
                         volNum  : TNWVolNum ;
                         var volBlockInfo : DIR_SPACE_INFO) : boolean ;
  begin
    result := false ;
    if (@NWGetDirSpaceInfo = nil) then
      @NWGetDirSpaceInfo := getProcAddress(hNWCalls,'NWGetDirSpaceInfo') ;
    if (@NWGetDirSpaceInfo = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWGetDirSpaceInfo(nServer,
                                0,
                                volNum,
                                volBlockInfo) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function ticksToTimeString(ticks : longInt) : string ;
  {convert time ticks to time string}
  var
    nDays,
    nHours,
    nMins,
    nSecs  : word ;
    dtemp  : TDateTime ;
  begin
    ticks  := trunc(ticks / 18.20648) ;
    nDays  := (ticks div 86400) ;  {86400 = secs per day}
    if (ndays > 0) then
      ticks  := ticks - ((nDays-1) * 86400) ;
    nSecs  := (ticks mod 60)   ;
    nMins  := ((ticks div 60) mod 60) ;
    nHours := (((ticks div 60) div 60) mod 24) ;
    nSecs  := minLong(nSecs,59) ;
    nMins  := minLong(nMins,59) ;
    nHours := minLong(nHours,23) ;
    dtemp  := encodeTime(nHours,nMins,nSecs,0) ;
    result := intToStr(nDays) + ':' +
              formatDateTime('hh:mm:ss',dtemp) ;
  end;


function getVolNum(nServer : TNWConnHandle ;
                   volName : string ;
                   var volNum : integer) : boolean ;
  var
   cpVolName : TObjName ;
  begin
    result := false ;
    if (@NWGetVolumeNumber = nil) then
      @NWGetVolumeNumber := getProcAddress(hNWCalls,'NWGetVolumeNumber') ;
    if (@NWGetVolumeNumber = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;  
    strPCopy(cpVolName,upperCase(volName)) ;
    nError := NWGetVolumeNumber(nServer,
                                @cpVolName,
                                @volNum) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function getVolExtendedInfo(nServer        : TNWConnHandle ;
                            volName        : string ;
                            var volExtInfo : TNWVolExtendedInfo) : boolean ;
  var
    volNum : integer ;
  begin
    result := false ;
    if (@NWGetExtendedVolumeInfo = nil) then
      @NWGetExtendedVolumeInfo := getProcAddress(hNWCalls,'NWGetExtendedVolumeInfo') ;
    if (@NWGetExtendedVolumeInfo = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    if (not getVolNum(nServer,volName,volNum)) then
      exit ;
    fillChar(volExtInfo,sizeOf(TNWVolExtendedInfo),0) ;
    nError := NWGetExtendedVolumeInfo(nServer,volNum,volExtInfo) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function setFileFlags(nServer   : TNWConnHandle ; {renamed from flagFile}
                      fileName  : string        ;
                      fileFlags : TNWFileFlags) : boolean ;
  var
    newAttrs    : nuint8   ;
    dirHandle   : TNWDirHandle ;
    cpRelative,
    cpFileName  : TRetBuff ;
  begin
    result := false ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWSetFileAttributes = nil) then
      @NWSetFileAttributes := getProcAddress(hNWCalls,'NWSetFileAttributes') ;
    if (@NWSetFileAttributes = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpFileName,upperCase(fileName)) ;
    newAttrs := fa_normal ;
    with fileFlags do begin
      if readOnly then
        newAttrs := (newAttrs or fa_readOnly) ;
      if shareable then
        newAttrs := (newAttrs or fa_shareAble) ;
      if hidden then
        newAttrs := (newAttrs or fa_hidden) ;
      if system then
        newAttrs := (newAttrs or fa_system) ;
      if directory then
        newAttrs := (newAttrs or fa_directory) ;
      if needsArchive then
        newAttrs := (newAttrs or fa_needsArchive) ;
      if executeOnly then
        newAttrs := (newAttrs or fa_executeOnly) ;
    end;
    if (NWParseNetWarePath(@cpfileName,
                           nserver,
                           dirHandle,
                           @cpRelative)=0) then
      begin
        nError := NWSetFileAttributes(nServer,
                                      dirHandle,
                                      @cpRelative,
                                      (fa_normal or (fa_system or fa_hidden)),
                                      newAttrs) ;
        if (nError <> 0) then
          lastError := nError
        else
          result := true ;
     end;
  end;

function getFileFlags(nServer       : TNWConnHandle ;
                      fileName      : string ;
                      var fileFlags : TNWFileFlags) : boolean ;
  var
    dirHandle   : TNWDirHandle ;
    iteration   : TNWSequence ;
    cpRelative,
    cpFileName  : TRetBuff ;
    fileInfo    : TNWFileInfo2 ;
  begin
    result := false ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWIntScanFileInformation2 = nil) then
      @NWIntScanFileInformation2 := getProcAddress(hNWCalls,'NWIntScanFileInformation2') ;
    if (@NWIntScanFileInformation2 = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpFileName,upperCase(fileName)) ;
    iteration := -1 ;
    if (NWParseNetWarePath(@cpfileName,
                           nserver,
                           dirHandle,
                           @cpRelative)=0) then
      begin
        nError := NWIntScanFileInformation2(nServer,
                                            dirHandle,
                                            @cpRelative,
                                            ((fa_normal) or (fa_system or fa_hidden)),
                                            @iteration,
                                            fileInfo,0) ;
        if (nError <> 0) then
          lastError := nError
        else
          begin
            with fileFlags do begin
              normal       := ((fileInfo.fileAttributes and FA_NORMAL) > 0);
              readOnly     := ((fileInfo.fileAttributes and FA_READONLY) > 0);
              hidden       := ((fileInfo.fileAttributes and FA_HIDDEN) > 0);
              system       := ((fileInfo.fileAttributes and FA_SYSTEM) > 0);
              executeOnly  := ((fileInfo.fileAttributes and FA_EXECUTEONLY) > 0);
              directory    := ((fileInfo.fileAttributes and FA_DIRECTORY) > 0);
              needsArchive := ((fileInfo.fileAttributes and FA_NEEDSARCHIVE) > 0);
              shareable    := ((fileInfo.fileAttributes and FA_SHAREABLE) > 0);
            end;
            result := true ;
          end;  
      end;
  end;


function getObjDiskRestrictions(nServer        : TNWConnHandle ;
                                volName,
                                objName        : string;
                                var byteLimit  : longint;
                                var bytesInUse : longint) : boolean ;
                                { function courtesy of
                                  Mark Mazelin of Cedarville College }
  var
    volNum     : integer ;
    nObjID     : TObjID ;
    nObjServer : TNWConnHandle ;
    
  begin
    result := false ;
    if (@NWGetObjDiskRestrictions = nil) then
      @NWGetObjDiskRestrictions := getProcAddress(hNWCalls,'NWGetObjDiskRestrictions') ;
    if (@NWGetObjDiskRestrictions = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nObjServer := nServer ;  
    if (not getVolNum(nServer,volName,volNum)) then
      exit ;
    nObjID := getObjNumber(nObjServer,objName,getObjType(nServer,objName)) ;
    if (nObjID = 0) then
      exit ;
    nError := NWGetObjDiskRestrictions(nServer,
                                       volNum,
                                       nObjID,
                                       @byteLimit,
                                       @bytesInUse) ;
    if (nError <> 0) then
      lastError := nError
    else
      begin
        bytesInUse := (bytesInUse * 4096) ;
        if (byteLimit >= $7FFFFFFF) then
          byteLimit := -1 {no restrictions in place for object}
        else
          byteLimit := (byteLimit * 4096);
        result := true ;
      end;
  end;

function setObjDiskRestrictions(nServer : TNWConnHandle ;
                                volName,
                                userID  : string;
                                bytesRestrict: longInt): boolean;
                                { function courtesy of
                                  Mark Mazelin of Cedarville College }
  var
    volNum      : integer ;
    nObjID      : TObjID  ;
    restriction : longInt ;
    nObjServer  : TNWConnHandle ;
    
  begin
    result  := false;
    if (@NWSetObjectVolSpaceLimit = nil) then
      @NWSetObjectVolSpaceLimit := getProcAddress(hNWCalls,'NWSetObjectVolSpaceLimit') ;
    if (@NWSetObjectVolSpaceLimit = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nObjServer := nServer ;  
    nObjID     := getObjNumber(nObjServer,userID,nw_user) ;
    if (nObjID = 0) then
      exit ;
    if getVolNum(nServer, volName, volNum) then
      begin
        restriction := (bytesRestrict div 4096); {convert bytes to blocks}
        nError := NWSetObjectVolSpaceLimit(nServer,volNum,
                                           nObjID,restriction) ;
        if (nError <> 0) then
          lastError := nError
        else
          result := true ;
      end;
  end;

function removeObjDiskRestrictions(nServer : TNWConnHandle ;
                                   volName, userID: string): boolean ;
                                   { function courtesy of
                                    Mark Mazelin of Cedarville College }
  var
    volNum     : integer ;
    nObjID     : TObjID  ;
    nObjServer : TNWConnHandle ;
    
  begin
    result := false;
    if (@NWRemoveObjectDiskRestrictions = nil) then
      @NWRemoveObjectDiskRestrictions :=
        getProcAddress(hNWCalls,'NWRemoveObjectDiskRestrictions') ;
    if (@NWRemoveObjectDiskRestrictions = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nObjServer := nServer ;  
    nObjID     := getObjNumber(nObjServer,userID,nw_user) ;
    if (nObjID = 0) then
      exit ;
    if getVolNum(nServer, volName, volNum) then
      begin
        nError := NWRemoveObjectDiskRestrictions(nServer,volNum,nObjID) ;
        if (nError <> 0) then
          lastError := nError
        else
          result := true ;
      end;
  end;

function clearConnection(nServer : TNWConnHandle; connID : word) : boolean ;
  begin
    result := false ;
    if (@NWClearConnectionNumber = nil) then
      @NWClearConnectionNumber := getProcAddress(hNWCalls,'NWClearConnectionNumber') ;
    if (@NWClearConnectionNumber = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    nError := NWClearConnectionNumber(nServer,connID) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function openConnByName(serverName : string ; ndsFormat, licensed : boolean) : TNWConnHandle ;
  var
    openState,
    nameFormat   : word ;
    nConn        : TNWConnHandle ;
    cpServerName : TObjName ;
  begin
    result := 0 ;
    if (@NWCCOpenConnByName = nil) then
      @NWCCOpenConnByName := getProcAddress(hNWClx,'NWCCOpenConnByName') ;
    if (@NWCCOpenConnByName = nil) then
      exit ;
    strPCopy(cpServerName,serverName) ;
    openState := NWCC_OPEN_UNLICENSED ;
    nameFormat := NWCC_NAME_FORMAT_BIND ;
    if ndsFormat then
      nameFormat := NWCC_NAME_FORMAT_NDS_TREE ;
    if licensed then
      openState := NWCC_OPEN_LICENSED ;
    nError := NWCCOpenConnByName(0,@cpServerName,nameFormat,openState,NWCC_RESERVED,@nConn) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := nConn ;
  end;


function getNLMList(nServer : TNWConnHandle) : TStringList ;
  var
    startNum,
    nloop      : integer ;
    nlmInfo    : TNWNLMInfo ;
    loadedList : NWFSE_NLM_LOADED_LIST ;
  begin
    result   := TStringList.create ;
    startNum := 0 ;
    if (@NWGetNLMLoadedList = nil) then
      @NWGetNLMLoadedList := getProcAddress(hNWCalls,'NWGetNLMLoadedList') ;
    if (@NWGetNLMLoadedList = nil) then
      exit ;
    if (@NWGetNLMInfo = nil) then
      @NWGetNLMInfo := getProcAddress(hNWCalls,'NWGetNLMInfo') ;
    if (@NWGetNLMInfo  = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    loadedList.nlmsInList := 128 ;  
    while (loadedList.nlmsInList = 128) do begin  
      nError := NWGetNLMLoadedList(nServer,startNum,loadedList) ;  {gets max 128 nlms}
      if (nError <> 0) then
        begin
          if (result.count < 1) then
            lastError := nError ;
          break ;  
        end  
      else
        for nloop := 0 to (loadedList.nlmsInList-1) do begin
          if getNLMInfo(nServer,loadedList.nlmNums[nloop],nlmInfo) then
            result.addObject(nlmInfo.nlmFileName,TObject(nlmInfo.nlmID)) ;
        end;    
      startNum := startNum + (loadedList.numberLNMsLoaded-1) ;
    end;    
            
  end;


function getNLMID(nServer : TNWConnHandle ; NLMName : string) : longint ;
  var
    index    : integer ;
    tempList : TStringList ;
  begin
    result := 0 ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    tempList := getNLMList(nServer) ;
    if (tempList.count < 1) then
      exit ;
    index := tempList.indexOf(upperCase(nlmName)) ;
    if (index < 0) then
      exit ;
    result := longint(tempList.objects[index]) ;
  end;  
    
function getNLMInfo(nServer : TNWConnHandle ;
                    NLM_ID  : longint ;
                    var nlmInfo : TNWNLMInfo) : boolean ;
  var
    cpCopyright,
    cpFileName,
    cpNLMName  : TRetBuff ;
    fseNLMInfo : NWFSE_NLM_INFO ;                  
  begin
    result := false ;
    try
      if (@NWGetNLMInfo = nil) then
        @NWGetNLMInfo := getProcAddress(hNWCalls,'NWGetNLMInfo') ;
      if (@NWGetNLMInfo = nil) then
        exit ;
      if (nServer = 0) then
        nServer := getPrimaryServerID ; 
      nError := NWGetNLMInfo(nServer,NLM_ID,@cpFileName,@cpNLMName,@cpCopyright,fseNLMInfo) ;
      if (nError <> 0) then
        lastError := nError 
      else
        begin
          with nlmInfo,fseNLMInfo do begin
            NLMFileName     := strPas(cpFileName) ;
            NLMName         := strPas(cpNLMName) ;
            version         := intToStr(NLM_INFO.majorVersion) + '.' +
                               intToStr(NLM_INFO.minorVersion) + '.' +
                               intToStr(NLM_INFO.revision) ;
            copyRight       := strPas(cpCopyright) ;
            nlmDate         := encodeDate(NLM_INFO.year,
                                          NLM_INFO.month,
                                          NLM_INFO.day) ;
            NLMID           := NLM_ID ;
            NLMFlags        := NLM_INFO.flags   ;
            NLMType         := NLM_INFO.NLMType ;
            parentID        := NLM_INFO.parentID ;
            allocFreeBytes  := NLM_INFO.allocAvailableBytes ;
            allocFreeCount  := NLM_INFO.allocFreeCount ;
            messageLanguage := NLM_INFO.messageLanguage;
            numReferPublics := NLM_INFO.numOfReferencedPublics ; 
          end;  
          result := true ;
        end;  
    except
      on e:exception do
        alertBox(e.message) ;
    end;   
  end;

function serverExecNCFFile(nServer : TNWConnHandle ; ncfFileName : string) : boolean ;
  var
    pathInfo   : TNWPathInfo ;
    cpFileName : TRetBuff ;
  begin
    result := false ;
    if (@NWSMExecuteNCFFile = nil) then
      @NWSMExecuteNCFFile := getProcAddress(hNWCalls,'NWSMExecuteNCFFile') ;
    if (@NWSMExecuteNCFFile = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    if parseNetwarePath(nServer,ncfFileName,pathInfo) then
      strPCopy(cpFileName,pathInfo.relativePath) 
    else
      strPCopy(cpFileName,ncfFileName) ;  
    nError := NWSMExecuteNCFFile(nServer,@cpFileName) ;
    if (nError <> 0) then
      lastError := nError 
    else
      result := true ;  
  end;  

function nlmLoad(nServer : TNWConnHandle ; nlmFileName, parameters : string) : boolean ;  
  var
    cpFileName : TRetBuff ;
    pathInfo   : TNWPathInfo ;
  begin
    result := false ;
    if (@NWSMLoadNLM = nil) then
      @NWSMLoadNLM := getProcAddress(hNWCalls,'NWSMLoadNLM') ;
    if (@NWSMLoadNLM = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;  
    if (not parseNetwarePath(nServer,nlmFileName,pathInfo)) then
      pathInfo.relativePath := upperCase(nlmFileName) ;
    strPCopy(cpFileName,pathInfo.relativePath) ;  
    nError := NWSMLoadNLM(nServer,@cpFileName) ;
    if (nError <> 0) then
      lastError := nError 
    else
      result := true ;
  end;

function nlmUnload(nServer : TNWConnHandle ; nlmFileName : string) : boolean ;
  var
    cpFileName : TRetBuff ;
  begin
    result := false ;
    nlmFileName := extractFileName(nlmFileName) ;
    if (@NWSMUnloadNLM = nil) then
      @NWSMUnloadNLM := getProcAddress(hNWCalls,'NWSMUnloadNLM') ;
    if (@NWSMUnloadNLM = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;  
    strPCopy(cpFileName,upperCase(nlmFileName)) ;  
    nError := NWSMUnloadNLM(nServer,@cpFileName) ;  
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;  
  end;


function mountVolume(nServer : TNWConnHandle ; volName : string) : integer ;
  var
    volNum    : longint    ;
    cpVolName : TNWVolName ;
  begin
    result := 0 ;
    if (@NWSMMountVolume = nil) then
      @NWSMMountVolume := getProcAddress(hNWCalls,'NWSMMountVolume') ;
    if (@NWSMMountVolume = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;  
    strpCopy(cpVolName,upperCase(volName)) ; 
    nError := NWSMMountVolume(nServer,@cpVolName,@volnum) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := volNum ;  
  end;

function dismountVolume(nServer : TNWConnHandle ; volName : string) : boolean ;
  var
    cpVolName : TNWVolName ;
  begin
    result := false ;
    if (@NWSMDismountVolumeByName = nil) then
      @NWSMDismountVolumeByName := getProcAddress(hNWCalls,'NWSMDismountVolumeByName') ;
    if (@NWSMDismountVolumeByName = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;
    strPCopy(cpVolName,upperCase(volName));
    nError := NWSMDismountVolumeByName(nServer,@cpVolName) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;
  end;

function closeConnection(nServer : TNWConnHandle) : boolean ;
  begin
    result := false ;
    if (@NWCCCloseConn = nil) then
      @NWCCCloseConn := getProcAddress(hNWCalls,'NWCCCloseConn') ;
    if (@NWCCCloseConn = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;  
    nError := NWCCCLoseConn(nServer) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;    
  end;  


function serverSetStringValue(nServer : TNWConnHandle ;
                              setParmName,
                              setParmString : string) : boolean ;
  var
    cpSetValue,
    cpSetName   : TObjName ;                            
  begin
    result := false ;      
    if (@NWSMSetDynamicCmdStrValue = nil) then
      @NWSMSetDynamicCmdStrValue := getProcAddress(hNWCalls,'NWSMSetDynamicCmdStrValue') ;
    if (@NWSMSetDynamicCmdStrValue = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ;  
    strPCopy(cpSetName,setParmName) ;
    strPCopy(cpSetValue,upperCase(setParmString)) ;
    nError := NWSMSetDynamicCmdStrValue(nServer,@cpSetName,@cpSetValue) ;
    if (nError <> 0) then
      lastError := nError
    else
      result := true ;  
  end;                                          
                       
function serverSetIntegerValue(nServer : TNWConnHandle ;
                              setParmName : string ;
                              setParmInt  : longint) : boolean ;
  var
    cpSetName : TObjName ;                            
  begin      
    result := false ;      
    if (@NWSMSetDynamicCmdIntValue = nil) then
      @NWSMSetDynamicCmdIntValue := getProcAddress(hNWCalls,'NWSMSetDynamicCmdIntValue') ;
    if (@NWSMSetDynamicCmdIntValue = nil) then
      exit ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ; 
    strPCopy(cpSetName,setParmName) ;   
    nError := NWSMSetDynamicCmdIntValue(nServer,@cpSetName,setParmInt) ;
    if (nError <> 0) then
      lastError := nError 
    else
      result := true ;  
  end;                              

function setExtendedFileFlags(inFile : string ;
                              const extFileFlags : TNWExtFileFlags ;
                              delFileFlags : boolean) : boolean ; 
                              {thanks to Gunnar Prien for help 
                               in getting this to work properly!}
  var
    nServer    : TNWConnHandle ;
    dirHandle,
    outHandle  : TNWDirHandle ;
    effRights  : TNWAccessRights ;
    iterHandle : array[1..9] of nuint8 ;
    fileInfo   : TNWFileInfo2 ;
    cpInfile   : TRetBuff;
    cpPath,
    cpRelative : TNWPath ;
  begin
    result    := false ;
    outhandle := 0 ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      @NWAllocTemporaryDirectoryHandle := getProcAddress(hNWCalls,'NWAllocTemporaryDirectoryHandle');
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      exit ;
    if (@NWDeallocateDirectoryHandle = nil) then
      @NWDeallocateDirectoryHandle := getProcAddress(hNWCalls,'NWDeallocateDirectoryHandle') ;
    if (@NWDeallocateDirectoryHandle = nil) then
      exit ;
    if (@NWIntScanFileInformation2 = nil) then
      @NWIntScanFileInformation2 := getProcAddress(hNWCalls,'NWIntScanFileInformation2');
    if (@NWIntScanFileInformation2 = nil) then
      exit ;
    if (@NWSetFileInformation2 = nil) then
      @NWSetFileInformation2 := getProcAddress(hNWCalls,'NWSetFileInformation2');
    if (@NWSetFileInformation2 = nil) then
      exit ;
    if (@NWSetExtendedFileAttributes2 = nil) then                                    
      @NWSetExtendedFileAttributes2 := getProcAddress(hNWCalls,'NWSetExtendedFileAttributes2');
    if (@NWSetExtendedFileAttributes2 = nil) then                                    
      exit ;
    strPCopy(cpInfile,extractFileName(infile)) ;
    infile := extractFilePath(inFile) ;
    if (length(inFile) < 1) then
      inFile := '\' ;
    strPCopy(cpPath,inFile) ;  
    if (NWParseNetwarePath(@cpPath,nServer,dirHandle,@cpRelative) = 0) and
       (NWAllocTemporaryDirectoryHandle(nServer,0,@cpRelative,outHandle,@effRights) = 0) then
      try
        fillChar(iterHandle,sizeOf(iterHandle),$FF) ;
        nError := NWIntScanFileInformation2(nServer,outHandle,@cpInfile,
                                            (fa_normal or fa_System or fa_hidden),
                                            @iterhandle,
                                            fileInfo,
                                            0) ;
        if (nError = 0) then
          with fileInfo do 
            if delFileFlags then
              begin
                if extFileFlags.readOnly then
                  fileAttributes := (fileAttributes AND NOT fa_readOnly) ;
                if extFileFlags.hidden then
                  fileAttributes := (fileAttributes AND NOT fa_hidden) ;
                if extFileFlags.system then
                  fileAttributes := (fileAttributes AND NOT fa_system) ;
                if extFileFlags.executeOnly then
                  fileAttributes := (fileAttributes AND NOT fa_executeOnly) ;
                if extFileFlags.directory then
                  fileAttributes := (fileAttributes AND NOT fa_directory) ;
                if extFileFlags.needsArchive then
                  fileAttributes := (fileAttributes AND NOT fa_needsArchive) ;
                if extFileFlags.shareable then
                  fileAttributes := (fileAttributes AND NOT fa_shareable) ;
                if extFileFlags.transaction then
                  fileAttributes := (fileAttributes AND NOT fa_transaction) ;
                if extFileFlags.indexed then
                  fileAttributes := (fileAttributes AND NOT fa_indexed) ;
                if extFileFlags.readAudit then
                  fileAttributes := (fileAttributes AND NOT fa_readAudit) ;
                if extFileFlags.writeAudit then
                  fileAttributes := (fileAttributes AND NOT fa_writeAudit) ;
              end
            else
              begin
                if extFileFlags.readOnly then
                  fileAttributes := (fileAttributes or fa_readOnly) ;
                if extFileFlags.hidden then
                  fileAttributes := (fileAttributes or fa_hidden) ;
                if extFileFlags.system then
                  fileAttributes := (fileAttributes or fa_system) ;
                if extFileFlags.executeOnly then
                  fileAttributes := (fileAttributes or fa_executeOnly) ;
                if extFileFlags.directory then
                  fileAttributes := (fileAttributes or fa_directory) ;
                if extFileFlags.needsArchive then
                  fileAttributes := (fileAttributes or fa_needsArchive) ;
                if extFileFlags.shareable then
                  fileAttributes := (fileAttributes or fa_shareable) ;
                if extFileFlags.transaction then
                  fileAttributes := (fileAttributes or fa_transaction) ;
                if extFileFlags.indexed then
                  fileAttributes := (fileAttributes or fa_indexed) ;
                if extFileFlags.readAudit then
                  fileAttributes := (fileAttributes or fa_readAudit) ;
                if extFileFlags.writeAudit then
                  fileAttributes := (fileAttributes or fa_writeAudit) ;
              end;  

            nError := NWSetFileInformation2(nServer,outHandle,
                                            @cpInFile,
                                            (fa_normal or fa_system or fa_hidden) ,
                                            fileInfo) ;
            if (nError = 0) then
              nError := NWSetExtendedFileAttributes2(nServer,
                                                       outHandle,
                                                       @cpInFile,
                                                       fileInfo.extendedFileAttributes) ;                                
                                                       
            if (nError = 0) then
              result := true
            else  
              lastError := nError ;
      finally    
        {clean up directory handle}
        if (outHandle > 0) then
          NWDeallocateDirectoryHandle(nServer,outHandle) ;  
      end;
  end;          

function TTSEnable(nServer : TNWConnHandle) : boolean ;
  begin
    result := false ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ; 
    if (@NWEnableTTS = nil) then
      @NWEnableTTS := getProcAddress(hNWCalls,'NWEnableTTS') ;
    if (@NWEnableTTS <> nil) then
      result := (NWEnableTTS(nServer) = 0);  
  end;
  
function TTSDisable(nServer : TNWConnHandle) : boolean ;
  begin
    result := false ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ; 
    if (@NWDisableTTS = nil) then
      @NWDisableTTS := getProcAddress(hNWCalls,'NWDisableTTS') ;
    if (@NWDisableTTS <> nil) then
      result := (NWDisableTTS(nServer) = 0);  
  end;
  
function TTSBeginTransaction(nServer : TNWConnHandle) : boolean ;
  begin
    result := false ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ; 
    if (@NWTTSBeginTransaction = nil) then
      @NWTTSBeginTransaction := getProcAddress(hNWCalls,'NWTTSBeginTransaction') ;
    if (@NWTTSBeginTransaction <> nil) then
      result := (NWTTSBeginTransaction(nServer) = 0) ;
  end;                

function TTSEndTransaction(nServer : TNWConnHandle) : longint ;
  var
    transactionID : longint ;
  begin
    result := 0 ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ; 
    if (@NWTTSEndTransaction = nil) then
      @NWTTSEndTransaction := getProcAddress(hNWCalls,'NWTTSEndTransaction') ;
    if (@NWTTSEndTransaction <> nil) and
       (NWTTSEndTransaction(nServer,@transactionID) = 0) then
      result := transactionID ; 
  end;    
  
function TTSAbortTransaction(nServer : TNWConnHandle) : boolean ;
  begin
    result := false ;
    if (nServer = 0) then
      nServer := getPrimaryServerID ; 
    if (@NWTTSAbortTransaction = nil) then
      @NWTTSAbortTransaction := getProcAddress(hNWCalls,'NWTTSAbortTransaction') ;
    if (@NWTTSAbortTransaction <> nil) then
      result := (NWTTSAbortTransaction(nServer) = 0) ;
  end;

function setDirSpaceLimit(cPath : string ;   {rounds to 4K Boundaries}
                          KByteLimit : longint) : boolean ;        
  var
    nServer    : TNWConnHandle ;
    dirHandle,
    outHandle  : TNWDirHandle ;
    cpPath,
    cpRelative : TNWPath ;
  begin      
    result := false ;
    outhandle := 0 ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      @NWAllocTemporaryDirectoryHandle := getProcAddress(hNWCalls,'NWAllocTemporaryDirectoryHandle');
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      exit ;
    if (@NWDeallocateDirectoryHandle = nil) then
      @NWDeallocateDirectoryHandle := getProcAddress(hNWCalls,'NWDeallocateDirectoryHandle') ;
    if (@NWDeallocateDirectoryHandle = nil) then
      exit ;
    if (@NWSetDirSpaceLimit = nil) then
      @NWSetDirSpaceLimit := getProcAddress(hNWCalls,'NWSetDirSpaceLimit') ;
    if (@NWSetDirSpaceLimit = nil) then
      exit ;
    strPCopy(cpPath,cPath) ;
    if (NWParseNetwarePath(@cpPath,nServer,dirHandle,@cpRelative) = 0) and
       (NWAllocTemporaryDirectoryHandle(nServer,0,@cpRelative,outHandle,nil) = 0) then
      try
        nError := NWSetDirSpaceLimit(nServer,outHandle,(KByteLimit div 4)) ; {rounds to 4K Blocks}
        if (nError = 0) then
          result := true
        else  
          lastError := nError ;
      finally
        {clean up directory handle}
        if (outHandle > 0) then
          NWDeallocateDirectoryHandle(nServer,outHandle) ;  
      end;
  end;                          

function getDirectoryInfo(pathName : string;
                          var fileInfo : TNWFileInfo) : boolean ;
  { Retrieve General Directory Information }
  var
    nServer    : TNWConnHandle   ;
    augmentFlag: word            ;
    attrs      : TNWAttributes   ;
    dirHandle,
    outHandle  : TNWDirHandle    ;
    sequence   : TNWSequence     ;
    effRights  : TNWAccessRights ;
    cpPath,
    cpDirName,
    cpRelative : TNWPath         ;
    srchPattern: TRetBuff        ;
    entryInfo  : NWENTRY_INFO    ;
  begin
    result := false ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      @NWAllocTemporaryDirectoryHandle := getProcAddress(hNWCalls,'NWAllocTemporaryDirectoryHandle') ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      exit ;
    if (@NWDeallocateDirectoryHandle = nil) then
      @NWDeallocateDirectoryHandle := getProcAddress(hNWCalls,'NWDeallocateDirectoryHandle') ;
    if (@NWDeallocateDirectoryHandle = nil) then
      exit ;
    if (@NWIntScanDirEntryInfo = nil) then
      @NWIntScanDirEntryInfo := getProcAddress(hNWCalls,'NWIntScanDirEntryInfo') ;
    if (@NWIntScanDirEntryInfo = nil) then
      exit ;
    attrs       := (FA_DIRECTORY OR FA_HIDDEN OR FA_SYSTEM)  ;
    augmentFlag := 0 ;
    sequence    := -1    ;
    strPCopy(srchPattern,'*') ;
    strPCopy(cpPath,upperCase(extractFilePath(pathName))) ;
    strPCopy(cpDirName,upperCase(extractFileName(pathName))) ;
    if (NWParseNetWarePath(@cpPath,
                           nServer,
                           dirHandle,
                           @cpRelative)=0) and
       (NWAllocTemporaryDirectoryHandle(nServer,
                                        0,
                                        @cpRelative,
                                        outHandle,
                                        @effRights)=0) then
      try
        while (NWIntScanDirEntryInfo(nServer,
                                     outHandle,
                                     attrs,
                                     @sequence,
                                     @srchPattern,
                                     entryInfo,
                                     augmentFlag) = 0) do begin
          if (strPos(strUpper(entryinfo.name),cpDirName) <> nil) then
            begin
              with fileInfo.inheritedRights do
                begin
                  read          := ((entryInfo.dir.inheritedRightsMask and TR_READ) > 0) ;
                  write         := ((entryInfo.dir.inheritedRightsMask and TR_WRITE) > 0) ;
                  open          := ((entryInfo.dir.inheritedRightsMask and TR_OPEN) > 0) ;
                  create        := ((entryInfo.dir.inheritedRightsMask and TR_CREATE) > 0) ;
                  canDelete     := ((entryInfo.dir.inheritedRightsMask and TR_DELETE) > 0) ;
                  owner         := ((entryInfo.dir.inheritedRightsMask and TR_OWNER) > 0)  ;
                  accessControl := ((entryInfo.dir.inheritedRightsMask and TR_ACCESSCTRL) > 0) ;
                  fileScan      := ((entryInfo.dir.inheritedRightsMask and TR_FILESCAN) > 0) ;
                  search        := ((entryInfo.dir.inheritedRightsMask and TR_FILESCAN) > 0) ;
                  modify        := ((entryInfo.dir.inheritedRightsMask and TR_MODIFY) > 0) ;
                  all           := ((entryInfo.dir.inheritedRightsMask and TR_ALL) > 0) ;
                  supervisor    := ((entryInfo.dir.inheritedRightsMask and TR_SUPERVISOR) > 0) ;
                  normal        := ((entryInfo.dir.inheritedRightsMask and TR_NORMAL) > 0) ;
                end;
              fileInfo.lastModifyDate  := nwDateTimeToTDateTime(entryInfo.dir.lastModifyDateAndTime) ;
              fileInfo.maximumSpace    := entryInfo.dir.maximumSpace;
              fileInfo.attributes      := entryInfo.attributes   ;
              fileInfo.flags           := entryInfo.flags        ;  {reserved value}
              fileInfo.nameSpace       := entryInfo.namespace    ;
              fileInfo.nameLength      := entryInfo.nameLength   ;
              fileInfo.name            := strPas(entryInfo.name) ;
              fileInfo.creationdate    := nwDateTimeToTDateTime(entryInfo.creationDateAndTime) ;
              fileInfo.ownerID         := getObjName(nServer,entryInfo.ownerID)                ;
              fileInfo.lastArchiveDate := nwDateTimeToTDateTime(entryInfo.lastArchiveDateAndTime)     ;
              fileInfo.lastArchivedBy  := getObjName(nServer,entryInfo.lastArchiverID)         ;
              {attributes record}
              with fileInfo.attrsRec do begin
                readOnly         := ((entryInfo.attributes and fa_readOnly) > 0) ;
                hidden           := ((entryInfo.attributes and fa_hidden) > 0) ;
                system           := ((entryInfo.attributes and fa_system) > 0) ;
                executeOnly      := ((entryInfo.attributes and fa_executeOnly) > 0) ;
                directory        := ((entryInfo.attributes and fa_directory) > 0) ;
                needsArchive     := ((entryInfo.attributes and fa_needsArchive) > 0) ;
                shareable        := ((entryInfo.attributes and fa_shareable) > 0) ;
                transactional    := ((entryInfo.attributes and fa_transaction) > 0) ;
                indexed          := ((entryInfo.attributes and fa_indexed) > 0) ;
                readAudit        := ((entryInfo.attributes and fa_readAudit) > 0) ;
                writeAudit       := ((entryInfo.attributes and fa_writeAudit) > 0) ;
                immediatePurge    := ((entryInfo.attributes and fa_immediatePurge) > 0) ;
                deleteInhibit     := ((entryInfo.attributes and fa_deleteInhibit) > 0) ;
                renameInhibit     := ((entryInfo.attributes and fa_renameInhibit) > 0) ;
                copyInhibit       := ((entryInfo.attributes and fa_copyInhibit) > 0) ;
                cantCompress      := ((entryInfo.attributes and fa_cantCompress) > 0) ;
                dontCompress      := ((entryInfo.attributes and fa_dontCompress) > 0) ;
                immediateCompress := ((entryInfo.attributes and fa_immediateCompress) > 0) ;
                fileMigrated      := ((entryInfo.attributes and fa_fileMigrated) > 0) ;
                dontMigrate       := ((entryInfo.attributes and fa_dontMigrate) > 0) ;
              end; 
              result := true ;
              break;
            end;
        end;
      finally
        if (outHandle > 0) then
          NWDeallocateDirectoryHandle(nServer,outHandle) ;
      end;
  end;
  
function setDirFlags( pathName : string ; 
                      fileFlags : TNWAllFileFlags;
                      nameSpace : TNWNameSpaceSet) : boolean ;
  var
    nServer    : TNWConnHandle   ;
    dirHandle,
    outHandle  : TNWDirHandle    ;
    effRights  : TNWAccessRights ;
    cpPath,
    cpRelative : TNWPath         ;
    entryInfo  : NWENTRY_INFO    ;
  begin
    result := false ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      @NWAllocTemporaryDirectoryHandle := getProcAddress(hNWCalls,'NWAllocTemporaryDirectoryHandle') ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      exit ;
    if (@NWDeallocateDirectoryHandle = nil) then
      @NWDeallocateDirectoryHandle := getProcAddress(hNWCalls,'NWDeallocateDirectoryHandle') ;
    if (@NWDeallocateDirectoryHandle = nil) then
      exit ;
    if (@NWSetDirEntryInfo = nil) then
      @NWSetDirEntryInfo := getProcAddress(hNWCalls,'NWSetDirEntryInfo') ;
    if (@NWSetDirEntryInfo = nil) then
      exit ;
    
    strPCopy(cpPath,pathName) ;
    fillChar(entryInfo,sizeOf(entryInfo),0) ;
    if (NWParseNetWarePath(@cpPath,
                           nServer,
                           dirHandle,
                           @cpRelative)=0) and
       (NWAllocTemporaryDirectoryHandle(nServer,
                                        0,
                                        @cpRelative,
                                        outHandle,
                                        @effRights)=0) then
      with fileFlags do try
        strCopy(entryInfo.name,cpRelative) ;
        entryInfo.nameLength := 0 ;
        entryInfo.nameSpace  := ord(nameSpace) ;
        {set extended attributes}
        if (not normal) then
          begin
            if readOnly then
              entryInfo.attributes := (entryInfo.attributes or fa_readOnly) ;
            if hidden then
              entryInfo.attributes := (entryInfo.attributes or fa_hidden) ;
            if system then
              entryInfo.attributes := (entryInfo.attributes or fa_system) ;
            if executeOnly then
              entryInfo.attributes := (entryInfo.attributes or fa_executeOnly) ;
            if needsArchive then
              entryInfo.attributes := (entryInfo.attributes or fa_needsArchive) ;
            if shareable then
              entryInfo.attributes := (entryInfo.attributes or fa_shareable) ;
            if transactional then
              entryInfo.attributes := (entryInfo.attributes or fa_renameInhibit) ;
            if indexed then
              entryInfo.attributes := (entryInfo.attributes or fa_renameInhibit) ;
            if readAudit then
              entryInfo.attributes := (entryInfo.attributes or fa_renameInhibit) ;
            if writeAudit then
              entryInfo.attributes := (entryInfo.attributes or fa_renameInhibit) ;
            if immediatePurge then
              entryInfo.attributes := (entryInfo.attributes or fa_immediatePurge) ;
            if deleteInhibit then 
              entryInfo.attributes := (entryInfo.attributes or fa_deleteInhibit) ;
            if renameInhibit then 
              entryInfo.attributes := (entryInfo.attributes or fa_renameInhibit) ;
            if copyInhibit then       
              entryInfo.attributes := (entryInfo.attributes or fa_copyInhibit) ; 
            if cantCompress then      
              entryInfo.attributes := (entryInfo.attributes or fa_cantCompress) ;
            if dontCompress then      
              entryInfo.attributes := (entryInfo.attributes or fa_dontCompress) ;
            if immediateCompress then                                            
              entryInfo.attributes := (entryInfo.attributes or fa_immediateCompress) ;
            if fileMigrated then                                                      
              entryInfo.attributes := (entryInfo.attributes or fa_fileMigrated) ; 
            if dontMigrate then       
              entryInfo.attributes := (entryInfo.attributes or fa_dontMigrate) ;
          end;
        nerror := NWSetDirEntryInfo(nServer,outHandle,FA_DIRECTORY,0,$02,entryInfo) ;
        if (nError <> 0) then
          lastError := nError
        else
          result := true ;
      finally
        if (outHandle > 0) then
          NWDeallocateDirectoryHandle(nServer,outHandle) ;
      end;
  end;


function setFileOwner(fileName, ownerName : string) : boolean ;
  var
    nObjServer,
    nServer    : TNWConnHandle ;
    dirHandle,
    outHandle  : TNWDirHandle ;
    effRights  : TNWAccessRights ;
    iterHandle : array[1..9] of nuint8 ;
    fileInfo   : TNWFileInfo2 ;
    cpInfile   : TRetBuff;
    cpPath,
    cpRelative : TNWPath ;
  begin
    result    := false ;
    outhandle := 0 ;
    if (@NWParseNetWarePath = nil) then
      @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
    if (@NWParseNetWarePath = nil) then
      exit ;
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      @NWAllocTemporaryDirectoryHandle := getProcAddress(hNWCalls,'NWAllocTemporaryDirectoryHandle');
    if (@NWAllocTemporaryDirectoryHandle = nil) then
      exit ;
    if (@NWDeallocateDirectoryHandle = nil) then
      @NWDeallocateDirectoryHandle := getProcAddress(hNWCalls,'NWDeallocateDirectoryHandle') ;
    if (@NWDeallocateDirectoryHandle = nil) then
      exit ;
    if (@NWIntScanFileInformation2 = nil) then
      @NWIntScanFileInformation2 := getProcAddress(hNWCalls,'NWIntScanFileInformation2');
    if (@NWIntScanFileInformation2 = nil) then
      exit ;
    if (@NWSetFileInformation2 = nil) then
      @NWSetFileInformation2 := getProcAddress(hNWCalls,'NWSetFileInformation2');
    if (@NWSetFileInformation2 = nil) then
      exit ;
    strPCopy(cpPath,extractFilePath(fileName)) ;
    strPCopy(cpInFile,extractFileName(fileName)) ;
    if (NWParseNetwarePath(@cpPath,nServer,dirHandle,@cpRelative) = 0) and
       (NWAllocTemporaryDirectoryHandle(nServer,0,@cpRelative,outHandle,@effRights) = 0) then
      try
        fillChar(iterHandle,sizeOf(iterHandle),$FF) ;
        nError := NWIntScanFileInformation2(nServer,outHandle,@cpInfile,
                                            (fa_normal or fa_System or fa_hidden),
                                             @iterhandle,
                                             fileInfo,
                                             0) ;
        if (nError = 0) then
          begin
            nObjServer := nServer ;
            fileInfo.fileOwnerID := getObjNumber(nObjServer,ownerName,nw_user) ;                                    
            nError := NWSetFileInformation2(nServer,outHandle,
                                            @cpInFile,
                                            (fa_normal or fa_system or fa_hidden) ,
                                            fileInfo) ;
          end;                                  
        if (nError = 0) then
          result := true
        else  
          lastError := nError ;
      finally    
        {clean up directory handle}
        if (outHandle > 0) then
          NWDeallocateDirectoryHandle(nServer,outHandle) ;  
      end;
  end;                              
  

function setDirectoryInfo(dirName : string; dirInfo : TNWDirectoryInfo) : boolean ;
var
  rightsMask : byte ;
  nServer    : TNWConnHandle ;
  dirHandle  : TNWDirHandle ;
  ownerID    : TObjID ;
  dtint      : longint ;
  cpRelative,
  cpPath     : TNDSName ;
begin
  result := false ;
  if (@NWParseNetWarePath = nil) then
    @NWParseNetWarePath := getProcAddress(hNWCalls,'NWParseNetWarePath') ;
  if (@NWParseNetWarePath = nil) then
    exit ;
  if (@NWSetDirectoryInformation = nil) then
    @NWSetDirectoryInformation := getProcAddress(hNWCalls,'NWSetDirectoryInformation') ;
  if (@NWSetDirectoryInformation = nil) then
    exit ;
  rightsMask := 0 ;
  with dirInfo.rights do begin
    if (all or read) then 
      rightsMask := (rightsMask or TA_READ) ;
    if (all or write) then 
      rightsMask := (rightsMask or TA_WRITE) ;
    if (all or open) then 
      rightsMask := (rightsMask or TA_OPEN) ;
    if (all or create) then 
      rightsMask := (rightsMask or TA_CREATE) ;
    if (all or canDelete) then 
      rightsMask := (rightsMask or TA_DELETE) ;
    if (all or modify) then 
      rightsMask := (rightsMask or TA_MODIFY) ;
    if (all or filescan) then 
      rightsMask := (rightsMask or TA_SEARCH) ;
    if (all or owner) then 
      rightsMask := (rightsMask or TA_OWNERSHIP) ;
  end;
  dtint := (byte(year(dirInfo.dateTime)-1980) shl 32) ;
  dtint := dtint + (byte(month(dirInfo.dateTime)) shl 25) ;
  dtint := dtint + (byte(day(dirInfo.dateTime)) shl 21) ;
  strPCopy(cpPath,upperCase(dirName)) ;
  if (NWParseNetwarePath(@cpPath,nServer,dirHandle,@cpRelative) = 0) then
    begin
      ownerID := getObjNumber(nServer,dirInfo.ownerName,nw_user) ;
      nError := NWSetDirectoryInformation(nServer,dirHandle,@cpRelative,dtint,ownerid,rightsMask) ;
      if (nError <> 0) then
        lastError := nError
      else
        result := true ;
    end;
end;

function getLANConfigInfo(nServer : TNWConnHandle ;
                          boardNum : integer ;
                          var lanConfigInfo : LAN_CONFIG_INFO) : boolean ;
var
  fseLANConfigInfo : NWFSE_LAN_CONFIG_INFO ;                          
begin
  result := false ;
  if (@NWGetLANConfigInfo = nil) then
    @NWGetLANConfigInfo := getProcAddress(hNWCalls,'NWGetLANConfigInfo') ;
  if (@NWGetLANConfigInfo = nil) then
    exit ;
  nError := NWGetLanConfigInfo(nServer, boardNum, fseLANConfigInfo) ;  
  if (nError <> 0) then
    lastError := nError 
  else
    begin
      result := true ;
      with lanConfigInfo do begin
        DriverCFG_MajorVersion := fseLANConfigInfo.LANConfigInfo.DriverCFG_MajorVersion ;
        DriverCFG_MinorVersion := fseLANConfigInfo.LANConfigInfo.DriverCFG_MinorVersion ;
        DriverNodeAddress := fseLANConfigInfo.LANConfigInfo.DriverNodeAddress ;
        DriverModeFlags := fseLANConfigInfo.LANConfigInfo.DriverModeFlags ;
        DriverBoardNum := fseLANConfigInfo.LANConfigInfo.DriverBoardNum ;
        DriverBoardInstance := fseLANConfigInfo.LANConfigInfo.DriverBoardInstance ;
        DriverMaxSize := fseLANConfigInfo.LANConfigInfo.DriverMaxSize ;
        DriverMaxRecvSize := fseLANConfigInfo.LANConfigInfo.DriverMaxRecvSize ;
        DriverRecvSize := fseLANConfigInfo.LANConfigInfo.DriverRecvSize ;
        Reserved1 := fseLANConfigInfo.LANConfigInfo.Reserved1 ;
        DriverCardID := fseLANConfigInfo.LANConfigInfo.DriverCardID ;
        DriverMediaID := fseLANConfigInfo.LANConfigInfo.DriverMediaID ;
        DriverTransportTime := fseLANConfigInfo.LANConfigInfo.DriverTransportTime ;
        DriverReserved := fseLANConfigInfo.LANConfigInfo.DriverReserved ;
        DriverMajorVersion := fseLANConfigInfo.LANConfigInfo.DriverMajorVersion ;
        DriverMinorVersion := fseLANConfigInfo.LANConfigInfo.DriverMinorVersion ;
        DriverFlags := fseLANConfigInfo.LANConfigInfo.DriverFlags ;
        DriverSendRetries := fseLANConfigInfo.LANConfigInfo.DriverSendRetries ;
        DriverLink := fseLANConfigInfo.LANConfigInfo.DriverLink ;
        DriverSharingFlags := fseLANConfigInfo.LANConfigInfo.DriverSharingFlags ;
        DriverSlot := fseLANConfigInfo.LANConfigInfo.DriverSlot ;
        DriverIOPortsAndLengths := fseLANConfigInfo.LANConfigInfo.DriverIOPortsAndLengths ;
        DriverMemDecode0 := fseLANConfigInfo.LANConfigInfo.DriverMemDecode0 ;
        DriverLength0 := fseLANConfigInfo.LANConfigInfo.DriverLength0 ;
        DriverMemDecode1 := fseLANConfigInfo.LANConfigInfo.DriverMemDecode1 ;
        DriverLength1 := fseLANConfigInfo.LANConfigInfo.DriverLength1 ;
        DriverInterrupt := fseLANConfigInfo.LANConfigInfo.DriverInterrupt ;
        DriverDMAUsage := fseLANConfigInfo.LANConfigInfo.DriverDMAUsage ;
        Reserved2 := fseLANConfigInfo.LANConfigInfo.Reserved2 ;
        DriverLogicalName := fseLANConfigInfo.LANConfigInfo.DriverLogicalName ;
        DriverLinearMem := fseLANConfigInfo.LANConfigInfo.DriverLinearMem ;
        DriverChannelNum := fseLANConfigInfo.LANConfigInfo.DriverChannelNum ;
        DriverIOReserved := fseLANConfigInfo.LANConfigInfo.DriverIOReserved ;
      end;
    end;
end;

function getServerTransportAddr(nServer : TNWConnHandle; var ta : TNWTransportAddr) : boolean ;                         
var
  i    : integer ;
  nwTA : NWCCTranAddr ;
begin
  result := false ;
  if (@NWCCGetConnAddress = nil) then
    @NWCCGetConnAddress := getProcAddress(hNWCalls,'NWCCGetConnAddress') ;
  if (@NWCCGetConnAddress = nil) then
    exit ;  
  nError := NWCCGetConnAddress(nServer, sizeOf(nwTA), @nwTA) ;
  if (nError <> 0) then
    lastError := nError 
  else
    with ta do begin
      address := '' ;
      if (nwTA.tranType =  NWCC_TRAN_TYPE_IPX) then
        begin
          transportType := ttIPX ;
          for i := 0 to nwTA.len do begin
            address := address + intToHex(nwTA.buffer[i], 2); 
            if (i = 3) then
              address := address + ':' ;
          end;
        end
      else if (nwTA.tranType =  NWCC_TRAN_TYPE_DDP) then
        transportType := ttDDP
      else if (nwTA.tranType =  NWCC_TRAN_TYPE_ASP) then
        transportType := ttASP
      else if (nwTA.tranType =  NWCC_TRAN_TYPE_UDP) then
        transportType := ttUDP
      else if (nwTA.tranType =  NWCC_TRAN_TYPE_TCP) then
        begin
          transportType := ttTCP ;
          if (nwTA.buffer[0] = $02) and (nwTA.buffer[1] = $0C) then
            {Windows NT}
            for i := 2 to 5 do begin
              address := address + iif(i=2,'','.') + intToHex(nwTA.buffer[i], 2); 
            end
          else
            {Win95/98/NLM}
            for i := 0 to 3 do begin
              address := address + iif(i=0,'','.') + intToHex(nwTA.buffer[i], 2); 
            end;  
        end 
      else if (nwTA.tranType =  NWCC_TRAN_TYPE_UDP6) then
        transportType := ttUDP6
      else if (nwTA.tranType =  NWCC_TRAN_TYPE_WILD) then
        transportType := ttWild ;
    end;      
end;


end.


