#ifndef __fcntl_h__
#define __fcntl_h__
/*============================================================================
=  Novell Software Development Kit
=
=  Copyright (C) 1999-2002 Novell, Inc. All Rights Reserved.
=
=  This work is subject to U.S. and international copyright laws and treaties.
=  Use and redistribution of this work is subject  to  the  license  agreement
=  accompanying  the  software  development kit (SDK) that contains this work.
=  However, no part of this work may be revised and/or  modified  without  the
=  prior  written consent of Novell, Inc. Any use or exploitation of this work
=  without authorization could subject the perpetrator to criminal  and  civil
=  liability. 
=
=  Source(s): ISO/IEC (POSIX) 9845:1996
=
=  fcntl.h
==============================================================================
*/
#include <stddef.h>

/* 'cmd' values for fcntl()...                                              */
#define F_GETFL      1        /* get file status flags                      */
#define F_SETFL      2        /* set file status flags                      */
#define F_DUPFD      3        /* duplicate file descriptor                  */
#define F_GETFD      4        /* get file descriptor flags                  */
#define F_SETFD      5        /* set file descriptor flags                  */
#define F_SETLK      6        /* set record locking info                    */
#define F_SETLK64    16       /* set record locking info (64-bit)           */
#define F_GETLK      7        /* get record locking info                    */
#define F_GETLK64    17       /* get record locking info (64-bit)           */
#define F_SETLKW     8        /* get record locking info; wait if blocked   */
#define F_SETLKW64   18       /* get record locking info (64-bit)           */
#define F_CLOEXEC    9        /* close on execute                           */

/* values for 'l_type' field of 'struct flock'...                           */
#define F_RDLCK      1        /* shared or read lock                        */
#define F_WRLCK      2        /* exclusive or write lock                    */
#define F_UNLCK      3        /* unlock                                     */

/* values for 'oflag' in open()...                                          */
#define O_RDONLY     0x00000000 /* open for read only                       */
#define O_WRONLY     0x00000001 /* open for write only                      */
#define O_RDWR       0x00000002 /* open for read and write                  */
#define O_ACCMODE    0x00000003 /* access flags mask                        */
#define O_reserved1  0x00000004 /* reserved                                 */
#define O_reserved2  0x00000008 /* reserved                                 */
#define O_APPEND     0x00000010 /* writes done at end of file               */
#define O_CREAT      0x00000020 /* create new file                          */
#define O_TRUNC      0x00000040 /* truncate existing file                   */
#define O_EXCL       0x00000080 /* exclusive open                           */
#define O_NOCTTY     0x00000100 /* no controlling terminal--unsupported     */
#define O_BINARY     0x00000200 /* binary file--all files                   */
#define O_NDELAY     0x00000400 /* nonblocking flag                         */
#define O_reserved3  0x00000800 /* reserved                                 */
#define O_SYNC       0x00001000 /* synchronized I/O file integrity          */
#define O_DSYNC      0x00002000 /* synchronized I/O data integrity          */
#define O_RSYNC      0x00004000 /* synchronized read I/O                    */
#define O_NONBLOCK   O_NDELAY   /* alias                                    */
#define FD_CLOEXEC   0x00008000 /* parent closes after call to process()    */
#define O_FIFO       0x00100000 /* opening one end of a FIFO [non-standard] */

/* value for third argument when 'cmd' is F_SETFL in fcntl()...             */
#define FNDELAY      O_NDELAY   /* fcntl() non-blocking I/O                 */

/* 'shflag' values for sopen()...                                           */
#define SH_DENYRW    0x00000010 /* deny read/write mode                     */
#define SH_DENYWR    0x00000020 /* deny write mode                          */
#define SH_DENYRD    0x00000030 /* deny read mode                           */
#define SH_DENYNO    0x00000040 /* deny none mode                           */

#include <pshpack1.h>

/*
** Advisory locking. File locking with fcntl() is constrained to per-process
** and not shared across processes. As long as this remains the case, 'l_pid'
** will merely be a copy of the value in 'l_tid.'
*/
struct flock
{
   pid_t   l_pid;             /* process ID of owner, get with F_GETLK      */
   pid_t   l_tid;             /* thread ID of owner, get with F_GETLK       */
   short   l_type;            /* F_RDLCK or F_WRLCK                         */
   short   l_whence;          /* flag for starting offset                   */
   long    ________spare;     /* unused                                     */
   long    ________reserved1; /* padding to resemble 64-bit structure       */
   off_t   l_start;           /* relative offset in bytes                   */
   long    ________reserved2; /* padding to resemble 64-bit structure       */
   off_t   l_len;             /* size; if 0, then until EOF                 */
};

struct flock64                /* with F_GETLK64, F_SETLK64, FSETLKW64       */
{
   pid_t   l_pid;             /* process ID of owner, get with F_GETLK      */
   pid_t   l_tid;             /* thread ID of owner, get with F_GETLK       */
   short   l_type;            /* F_RDLCK or F_WRLCK                         */
   short   l_whence;          /* flag for starting offset                   */
   long    ________spare;     /* unused                                     */
   off64_t l_start;           /* relative offset in bytes                   */
   off64_t l_len;             /* size; if 0, then until EOF                 */
};

#include <poppack.h>


#ifdef __cplusplus
extern "C"
{
#endif

int   creat( const char *path, mode_t mode );
int   fcntl( int fildes, int cmd, ... );
int   open( const char *path, int oflag, ... );
int   pipe_open( const char *path, int oflag );
int   sopen( const char *path, int oflag, int shflag, ... );

#ifdef __cplusplus
}
#endif

#endif
