#ifndef __nks_fsio_h__
#define __nks_fsio_h__
/*============================================================================
=  Novell Software Development Kit
=
=  Copyright (C) 1999-2002 Novell, Inc. All Rights Reserved.
=
=  This work is subject to U.S. and international copyright laws and treaties.
=  Use and redistribution of this work is subject  to  the  license  agreement
=  accompanying  the  software  development kit (SDK) that contains this work.
=  However, no part of this work may be revised and/or  modified  without  the
=  prior  written consent of Novell, Inc. Any use or exploitation of this work
=  without authorization could subject the perpetrator to criminal  and  civil
=  liability.
=
=  Source(s): Novell Header
=
=  nks/fsio.h
==============================================================================
*/
#include <unilib.h>
#include <nks/defs.h>
#include <nks/time.h>
#include <nks/thread.h>
#include <nks/netware.h>

#define NX_MAX_NAME_LEN          255
#define NX_MAX_FILESIZE          /* (big! --this isn't used yet) */

/* values and masks for file modes... */
#define NX_O_RDONLY              0x00000000
#define NX_O_WRONLY              0x00000001
#define NX_O_RDWR                0x00000002
#define NX_O_ACCMODE             0x00000003

/* values and masks for file operations... */
#define NX_O_APPEND              0x00000010
#define NX_O_CREAT               0x00000020
#define NX_O_TRUNC               0x00000040
#define NX_O_EXCL                0x00000080
#define NX_O_TRANS               0x00000100
#define NX_O_NONBLOCK            0x00000400
#define NX_O_OPMODE              0x000005E0

/* aliases for above... */
#define NX_O_CREATE              NX_O_CREAT
#define NX_O_TRUNCATE            NX_O_TRUNC
#define NX_O_EXCLUSIVE           NX_O_EXCL

/* open flags... */
#define NX_OFLAG_DIRECTIO        0x00010000
#define NX_OFLAG_CRONLY          0x00020000
#define NX_OFLAG_BACKUP          0x00040000
#define NX_OFLAG_RESTORE         0x00080000
#define NX_OFLAG_EXTEND          0x00100000
#define NX_OFLAG_SYNCWR          0x00200000
#define NX_OFLAG_ATOMIC_RW       0x00400000
#define NX_OFLAG_NOTRAVERSE_LINK 0x00800000
#define NX_OFLAG_MASK            0x00FF0000

/* sharing flags... */
#define NX_SHARE_DENYNO          0x00000000
#define NX_SHARE_DENYRD          0x00100000
#define NX_SHARE_DENYWR          0x00200000
#define NX_SHARE_DENYALL         0x00400000
#define NX_SHARE_MASK            0x00700000

/* delegation type... */
#define NX_DELEG_NONE            0x00000000

/* flush flags... */
#define NX_FLUSH_DATA            0x00000001
#define NX_FLUSH_METADATA        0x00000002
#define NX_FLUSH_ASYNC           0x00000004

/* values and masks for access flags... */
#define NX_R_OK                  0x00000001
#define NX_W_OK                  0x00000002
#define NX_X_OK                  0x00000004
#define NX_F_OK                  0x00000008

/* values for setting file length... */
#define NX_FOP_RETURN_EXTEND     0x00000001
#define NX_FOP_RETURN_TRUNC_FREE 0x00000002
#define NX_FOP_RETURN_SPARSE     0x00000004

/* values for file byte-range locking flags... */
#define NX_RANGE_LOCK_SHARED     0x00000001
#define NX_RANGE_LOCK_EXCL       0x00000002
#define NX_RANGE_LOCK_CHECK      0x00000004
#define NX_RANGE_LOCK_TRYLOCK    0x00000008
#define NX_RANGE_LOCK_COURTESY   0x00000010
#define NX_RANGE_LOCK_CANCEL     0x00000020
#define NX_RANGE_LOCK_POSIX      0x00000040

#include <pshpack1.h>

struct                  fsio;
typedef unsigned long   NXMode_t;
typedef unsigned long   NXOFlags_t;
typedef unsigned long   NXShareMode_t;
typedef uint64_t        NXOffset_t;    /* (file offsets and lengths) */
typedef int64_t         NXSOffset_t;
typedef uint64_t        NXLockToken_t;
typedef int             NXHandle_t;
typedef int             NXPathCtx_t;
typedef int             NXDelegType_t;
typedef void           *NXAsyncId_t;

typedef enum                 /* I/O objects supported... */
{
   NX_OBJ_UNKNOWN = 0xFFFFFFFF,
   NX_OBJ_DEFAULT = 0xFFFFFFFE,
   NX_OBJ_FILE    = 0xFFFFFFFD,
   NX_OBJ_DIR     = 0xFFFFFFFC,
   NX_OBJ_FIFO    = 0xFFFFFFFB,
   NX_OBJ_DEVICE  = 0xFFFFFFFA,
   NX_OBJ_CONSOLE = 0xFFFFFFF9,
   NX_OBJ_SYMLINK = 0xFFFFFFF8
} NXObjType_t;

typedef union
{
   uint64_t    guid_field[2];
} NXGuid_t;

typedef struct
{
   NXGuid_t    fidFsId;
   uint64_t    fidFileId[2];
} NXFid_t;

typedef enum
{
   NX_UPCALL_UNKNOWN        = 0xFFFFFFFF,
   NX_UPCALL_DELEGRECALL    = 0x00000000,
   NX_UPCALL_BACKUPIMMINENT = 0x00000001
} NXUpCallReason_t;

typedef void   (*NXFsUpCall_t)( NXHandle_t fileHandle, NXUpCallReason_t reason,
                                 void *parm );
typedef union
{
   void    *name;               /* generic filesystem name if used           */
   uint32_t macintoshId;        /* Macintosh data stream (data or resource)  */
} NXDataStream_t;

typedef struct
{
   NXPathCtx_t    fosPathCtx;   /* based on which 'fosPathname' is understood*/
   const void    *fosPathname;  /* relative to 'fosPathCtx'                  */
   NXMode_t       fosMode;      /* NX_O_RDONLY, NX_O_RDWR, etc.              */
   NXOFlags_t     fosOFlags;    /* NX_OFLAG_SYNCWR, NX_OFLAG_DIRECTIO, etc.  */
   NXShareMode_t  fosShareMode; /* NX_SHARE_DENYRD, NX_DENY_ALL, etc.        */
   size_t         fosExtentSize;/* contiguous file block allocation hint     */
   uint32_t          reserved1; /* contiguous file block allocation hint     */
   NXDelegType_t  fosDelegType; /* delegation type: NX_DELEG_NONE            */
   NXDataStream_t fosDataStream;/* data stream (if not part of fosPathname)  */
   NXFsUpCall_t   fosUpCallFunc;/* called to warn of delegation revocation   */
   struct
   {
      NXMode_t      actionTaken;/* description of successful conclusion      */
      NXDelegType_t delegType;  /* actual grant of delegation                */
      NXFid_t       fid;        /* returns FID of opened file                */
      NXTime_t      accessTime; /* time of last access                       */
      NXTime_t      modifyTime; /* time of last modification                 */
      NXTime_t      creationTime;/* time of file creation                    */
      NXTime_t      archiveTime;/* time of last back-up                      */
      NXOffset_t    length;     /* length of file at open                    */
   }              fosResult;    /* create/open operation results             */
   uint32_t          reserved2;
   uint32_t          reserved3;
} NXFileOpenSpec_t;

typedef int    (*NXIoComp_t)( struct fsio *ioInfo );

typedef struct
{
   size_t        ivLength;      /* length of I/O buffer                      */
	uint32_t      reserved;
   void         *ivBuffer;      /* pointer to data for I/O operation         */
   void         *ivOpRef;       /* application-maintained reference          */
} NXIoVec_t;

typedef enum                 /* I/O objects supported... */
{
   NX_LOCK_RANGE_UNKNOWN  = INT_MAX,  /* For Internal Use Only               */
   NX_LOCK_RANGE_FORWARD  = 1,
   NX_LOCK_RANGE_BACKWARD = 2
} NXLockRange_t;

typedef struct fsio
{
   void         *ioAppRef;      /* application-maintained reference          */
   NXHandle_t    ioHandle;      /* file, FIFO, console, device, etc. handle  */
   unsigned long ioFlags;       /* I/O flags and hints                       */
   uint32_t      reserved;
   NXOffset_t    ioRemainingHint;/* hint as to remaining number of bytes     */
   NXLockToken_t ioLockToken;   /* from NXFileRangeLockEx()                  */
   NXIoComp_t    ioCompletion;  /* completion function if asynchronous       */
   unsigned long ioTimeOut;     /* only for non-file system operations       */
   NXOffset_t    ioOffset;      /* offset at which to begin I/O              */
   int           ioVecCount;    /* number of records in vector               */
   NXIoVec_t    *ioVector;      /* IN/OUT: I/O operation record vector       */
   NXAsyncId_t   ioAsyncID;     /* OUT: asynchronous transaction ID          */
   int           ioStatus;      /* OUT: for asynchronous use                 */
   NXOffset_t    ioProcessed;   /* OUT: total bytes written or read          */
#define          ioBytesWritten  ioProcessed
#define          ioBytesRead     ioProcessed
} NXIo_t;

typedef struct
{
   NXOffset_t    ldOffset;       /* in file (0-based)                        */
   NXOffset_t    ldLength;       /* magnitude of range                       */
   NXLockRange_t ldDirection;    /* NX_LOCK_RANGE_FORWARD, ...               */
   NXLockToken_t ldToken;        /* usually 0; reserved to server apps       */
	NXHandle_t    ldHandle;       /* in use only with NX_RANGE_LOCK_NETWARE   */
   uint32_t      reserved;
} NXFileLockDesc_t;

typedef void (*NXLockUpCall_t)( NXHandle_t fileHandle, void *appRef,
                          NXFileLockDesc_t *lockDesc );

typedef struct
{
   void             *rlsAppRef;  /* application-maintained reference         */
   NXHandle_t        rlsHandle;  /* handle on which file was opened          */
   NXBool_t          rlsMandatory;/* mandatory (TRUE) or merely advisory?    */
   unsigned long     rlsFlags;   /* NX_RANGE_LOCK_EXCL, etc.                 */
   unsigned long     rlsTimeOut; /* maximum time to wait for lock            */
   NXLockUpCall_t    rlsUpCall;  /* call-back by file system to release lock */
   NXVmId_t          rlsConflictLockVm;/* info. on existing conflicting lock */
   NXFileLockDesc_t  rlsConflictLock;/* info. for VM holding conflicting lock*/
   int               rlsVecCount;/* count of locks described in array        */
   NXFileLockDesc_t *rlsDescVec; /* array of lock descriptions               */
} NXFileRangeLockSpec_t;

#include <poppack.h>

#ifdef __cplusplus
extern "C"
{
#endif


/* File I/O... */
int NXClose(NXHandle_t handle);
int NXFileAllocExtentWithHandle(NXHandle_t fileHandle, NXOffset_t offset,
       NXOffset_t length, unsigned long flags);
int NXFileCancelIo(NXAsyncId_t async_id, NXBool_t wait, NXBool_t *status);
int NXFileFlushBuffers(NXHandle_t fileHandle, unsigned long flags,
       NXOffset_t offset, NXOffset_t length);
int NXFileGetLength(NXPathCtx_t pathCtx, const void *pathname,
       NXOffset_t *length);
int NXFileGetLengthWithHandle(NXHandle_t fileHandle, NXOffset_t *length);
int NXFileOpen(NXPathCtx_t pathCtx, const void *pathname, NXMode_t mode,
       NXHandle_t *fileHandle);
int NXFileOpenEx(NXFileOpenSpec_t *openSpec, NXHandle_t *fileHandle);
int NXFileRangeLock(NXHandle_t fileHandle, unsigned long flags,
       NXOffset_t offset, NXSOffset_t length);
int NXFileRangeLockEx(NXFileRangeLockSpec_t lockSpec);
int NXFileRangeUnlock(NXHandle_t fileHandle, unsigned long flags,
       NXOffset_t offset, NXSOffset_t length);
int NXFileRangeUnlockEx(NXFileRangeLockSpec_t lockSpec);
int NXFileRemoveWithHandle(NXHandle_t handle);
int NXFileRenameWithHandle(NXHandle_t fileHandle, NXPathCtx_t targetPathCtx,
       const void *newname, NXBool_t overwrite);
int NXFileSetLength(NXPathCtx_t pathCtx, void *pathname, NXOffset_t length,
       unsigned long flags);
int NXFileSetLengthWithHandle(NXHandle_t fileHandle, NXOffset_t length,
       unsigned long flags);
int NXRead(NXHandle_t handle, NXOffset_t offset, size_t length,
       void *address, unsigned long flags, size_t *bytesRead);
int NXReadEx(NXIo_t *ioInfo, NXAsyncId_t *ioAsyncId);
int NXRemove(NXPathCtx_t pathCtx, const void *pathname);
int NXRename(NXPathCtx_t sourcePathCtx, const void *oldname,
       NXPathCtx_t targetPathCtx, const void *newname, NXBool_t overwrite);
int NXWrite(NXHandle_t handle, NXOffset_t offset, size_t length,
       const void *address, unsigned long flags, size_t *bytesWritten);
int NXWriteEx(NXIo_t *ioInfo, NXAsyncId_t *ioAsyncId);

/* Additional I/O (including FIFOs, devices, etc.)... */
int NXDeviceOpen(NXPathCtx_t pathCtx, const void *name, NXMode_t mode,
       NXShareMode_t shareMode, unsigned long flags, size_t *ioBlockSize,
       NXHandle_t *deviceHandle);
int NXFifoOpen(NXPathCtx_t pathCtx, const void *pathname, NXMode_t mode,
       size_t fifoSize, NXHandle_t *fifoHandle);
int NXIoSetBlockingState(NXHandle_t handle, NXBool_t blocking);
int NXIoGetOpenMode(NXHandle_t handle, NXMode_t *mode);
int NXLinkCreate(NXPathCtx_t srcPathCtx, const void *source,
       NXPathCtx_t *tgtPathCtx, const void *target, int type);
int NXLinkCreateSymbolic(NXPathCtx_t srcPathCtx, const void *linkname,
       const void *target);

#ifdef __cplusplus
}
#endif

/* macro for useful alias... */
#define NXConsoleOpen(name, mode, flags, handle)                        \
            NXDeviceOpen(NX_OBJ_CONSOLE, name, mode, NX_SHARE_DENYNO,   \
            flags, (size_t *) NULL, handle)
#define NXFileCreate(pC, p, fH)                                         \
            NXFileOpen(pC, p, NX_O_WRONLY|NX_O_CREAT|NX_O_TRUNC, fH)

#endif
