/* menu.c
 * Win32 console menu
 * 
 * Copyright (C) 2002-2003 Paul Pergamenshchik <pp64@cornell.edu>
 *
 * $Id: menu.c 1.8 Fri, 30 Aug 2002 20:13:05 -0400 pahan $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include<winsock2.h>
#include<windows.h>
#include<stdlib.h>
#include<ctype.h>
#include"rcon.h"

// static strings
static const char sPress[] = "Press Enter to continue...";

void menu_draw(menu_t mn) {
    DWORD temp;
    COORD c;
    unsigned int i;
    char *line = malloc(mn->xsize);
	SetConsoleTitle("Remote Console IP");
    // draw top border
    line[0] = (char)201; // top left double corner
    memset(line + 1, 205, mn->xsize - 2); // horizontal double line
    line[mn->xsize - 1] = (char)187; // top right double corner
    c.X = mn->x;
    c.Y = mn->y - 2;
    WriteConsoleOutputCharacter(hconout, line, mn->xsize, c, &temp);
    FillConsoleOutputAttribute(hconout, FOGR_YELLOW | BACKGR_BLUE, mn->xsize, c, &temp);
    // draw title
    line[0] = (char)186; // vertical double line
    line[mn->xsize - 1] = (char)186; // vertical double line
    memset(line + 1, 32, mn->xsize - 2);
    memcpy(line + 1, mn->title, MIN(mn->xsize - 2, strlen(mn->title)));
    c.X = mn->x;
    c.Y = mn->y - 1;
    WriteConsoleOutputCharacter(hconout, line, mn->xsize, c, &temp);
    FillConsoleOutputAttribute(hconout, FOGR_YELLOW | BACKGR_BLUE, mn->xsize, c, &temp);
    // draw separator
    line[0] = (char)204; // middle left double corner
    memset(line + 1, 205, mn->xsize - 2); // horizontal double line
    line[mn->xsize - 1] = (char)185; // middle right double corner
    c.X = mn->x;
    c.Y = mn->y;
    WriteConsoleOutputCharacter(hconout, line, mn->xsize, c, &temp);
    FillConsoleOutputAttribute(hconout, FOGR_YELLOW | BACKGR_BLUE, mn->xsize, c, &temp);
    // draw bottom border
    line[0] = (char)200; // bottom left double corner
    line[mn->xsize - 1] = (char)188; // bottom right corner
    c.Y = mn->y + mn->ysize;
    WriteConsoleOutputCharacter(hconout, line, mn->xsize, c, &temp);
    FillConsoleOutputAttribute(hconout, FOGR_YELLOW | BACKGR_BLUE, mn->xsize, c, &temp);
    // draw entries
    line[0] = (char)186; // vertical double line
    line[1] = 32;
    line[2] = (char)179; // vertical single line
    line[mn->xsize - 1] = (char)186; // vertical double line
    for(i = mn->top; i < MIN(mn->ysize + mn->top - 1, mn->len); i++) {
        memset(line + 3, 32, mn->xsize - 4);
        memcpy(line + 3, mn->data[i], MIN(mn->xsize - 4, strlen(mn->data[i])));
        c.Y = i - mn->top + mn->y + 1;
        WriteConsoleOutputCharacter(hconout, line, mn->xsize, c, &temp);
        FillConsoleOutputAttribute(hconout, FOGR_YELLOW | BACKGR_BLUE, 1, c, &temp);
        c.X += 1;
        FillConsoleOutputAttribute(hconout, FOGR_WHITE | BACKGR_BLUE, mn->xsize - 2, c, &temp);
        c.X += mn->xsize - 2;
        FillConsoleOutputAttribute(hconout, FOGR_YELLOW | BACKGR_BLUE, 1, c, &temp);
        c.X -= mn->xsize - 1;
    }
    // draw selection
    c.X = mn->x + 3;
    c.Y = mn->cur - mn->top + mn->y + 1;
    if(mn->len > 0) {
        FillConsoleOutputAttribute(hconout, FOGR_BLUE | BACKGR_GREY, mn->xsize - 4, c, &temp);
    }
    c.X = strlen(mn->comp) + 3 + mn->x;
    SetConsoleCursorPosition(hconout, c);
    c.X = mn->x;
    // draw empty entries
    for(; i < mn->ysize + mn->top - 1; i++) {
        memset(line + 3, 32, mn->xsize - 4);
        c.Y = i - mn->top + mn->y + 1;
        WriteConsoleOutputCharacter(hconout, line, mn->xsize, c, &temp);
        FillConsoleOutputAttribute(hconout, FOGR_YELLOW | BACKGR_BLUE, 1, c, &temp);
        c.X += 1;
        FillConsoleOutputAttribute(hconout, FOGR_WHITE | BACKGR_BLUE, mn->xsize - 2, c, &temp);
        c.X += mn->xsize - 2;
        FillConsoleOutputAttribute(hconout, FOGR_YELLOW | BACKGR_BLUE, 1, c, &temp);
        c.X -= mn->xsize - 1;
    }
    // draw arrows
    if(mn->top > 0) {
        c.X = mn->x + 1;
        c.Y = mn->y + 1;
        FillConsoleOutputCharacter(hconout, (char)30, 1, c, &temp);
    }
    if(mn->top + mn->ysize - 1 < mn->len) {
        c.X = mn->x + 1;
        c.Y = mn->y + mn->ysize - 1;
        FillConsoleOutputCharacter(hconout, (char)31, 1, c, &temp);
    }
    free(line);
    c.X = 0;
    c.Y = 0;
}

char *inputbox_get(char *prompt, int x, int y, int len, int pwd) {
    DWORD temp;
    COORD c;
    INPUT_RECORD ir;
	CONSOLE_CURSOR_INFO cinfo;
    int cur = 0;
    char *inp = malloc(len + 1);
    int l, prolen = strlen(prompt);
    int max = len + prolen + 3;
    char *line = malloc(max);
    // draw top border
	GetConsoleCursorInfo(hconout,&cinfo);
	cinfo.bVisible=TRUE;
	SetConsoleCursorInfo(hconout,&cinfo);
    line[0] = (char)201; // top left double corner
    memset(line + 1, 205, max - 2); // horizontal double line
    line[max - 1] = (char)187; // top right double corner
    c.X = x;
    c.Y = y;
    WriteConsoleOutputCharacter(hconout, line, max, c, &temp);
    FillConsoleOutputAttribute(hconout, FOGR_YELLOW | BACKGR_BLUE, max, c, &temp);
    // draw bottom border
    line[0] = (char)200; // bottom left double corner
    line[max - 1] = (char)188; // bottom right corner
    c.Y = y + 2;
    WriteConsoleOutputCharacter(hconout, line, max, c, &temp);
    FillConsoleOutputAttribute(hconout, FOGR_YELLOW | BACKGR_BLUE, max, c, &temp);
    // draw prompt
    line[0] = (char)186; // vertical double line
    line[max - 1] = (char)186; // vertical double line
    memset(line + 1, 32, max - 2);
    memcpy(line + 1, prompt, prolen);
    c.Y = y + 1;
    WriteConsoleOutputCharacter(hconout, line, max, c, &temp);
    FillConsoleOutputAttribute(hconout, FOGR_YELLOW | BACKGR_BLUE, max, c, &temp);
    c.X = x + 1 + prolen;
    SetConsoleCursorPosition(hconout, c);
    free(line);
    while(1) {
        ReadConsoleInput(hconin, &ir, 1, &temp);
        if(ir.EventType == KEY_EVENT && ir.Event.KeyEvent.bKeyDown) {
            for(l = 0; l < ir.Event.KeyEvent.wRepeatCount; l++) {
                if(isprint(ir.Event.KeyEvent.uChar.AsciiChar)) {
                    if(cur < len) {
                        inp[cur] = ir.Event.KeyEvent.uChar.AsciiChar;
                        c.X = x + 1 + prolen + cur;
                        c.Y = y + 1;
                        if(pwd) {
                            FillConsoleOutputCharacter(hconout, '*', 1, c, &temp);
                        } else {
                            FillConsoleOutputCharacter(hconout, ir.Event.KeyEvent.uChar.AsciiChar, 1, c, &temp);
                        }
                        FillConsoleOutputAttribute(hconout, FOGR_WHITE | BACKGR_BLUE, 1, c, &temp);
                        cur++;
                        c.X++;
                        SetConsoleCursorPosition(hconout, c);
                    }
                } else {
                    switch(ir.Event.KeyEvent.wVirtualKeyCode) {
                        case VK_BACK:
                            if(cur > 0) {
                                cur--;
                                c.X = x + 1 + prolen + cur;
                                c.Y = y + 1;
                                FillConsoleOutputCharacter(hconout, ' ', 1, c, &temp);
                                FillConsoleOutputAttribute(hconout, FOGR_WHITE | BACKGR_BLUE, 1, c, &temp);
                                SetConsoleCursorPosition(hconout, c);
                            }
                            break;
                        case VK_ESCAPE:
                            free(inp);
                            return NULL;
                        case VK_RETURN:
                            inp[cur] = 0;
                            return inp;
                    }
                }
            }
        }
    }
}

int menu_curnotinview(menu_t mn) {
    return (mn->cur < mn->top) || (mn->cur > mn->top + mn->ysize - 2);
}

void menu_movedown(menu_t mn, unsigned int by) {
	if (mn->len != 0)
	{
		mn->cur += by;
	    if(menu_curnotinview(mn)) {
			mn->top += MIN(by, mn->len - (mn->top + mn->ysize - 1));
		}
    }
}

void menu_moveup(menu_t mn, unsigned int by) {
	if (mn->len != 0)
	{
	    mn->cur -= by;
		if(menu_curnotinview(mn)) {
	        mn->top -= MIN(by, mn->top);
	    }
	}
}

void menu_scrollto(menu_t mn, unsigned int k) {
    if(k != -1) {
        if(k > mn->cur) {
            menu_movedown(mn, k - mn->cur);
        } else {
            menu_moveup(mn, mn->cur - k);
        }
    }
}

unsigned int menu_handlekey(menu_t mn, KEY_EVENT_RECORD keyevent) {
    unsigned int k = -1, i = 0, l;
    char c;
    switch(keyevent.wVirtualKeyCode) {
    case VK_DOWN:
    case VK_NUMPAD2:
        mn->comp[0] = '\0';
        menu_movedown(mn, MIN(1, mn->len - mn->cur - 1));
        break;
    case VK_UP:
    case VK_NUMPAD8:
        mn->comp[0] = '\0';
        menu_moveup(mn, MIN(1, mn->cur));
        break;
    case VK_NEXT:
    case VK_NUMPAD3:
        mn->comp[0] = '\0';
        menu_movedown(mn, MIN(mn->ysize - 1, mn->len - mn->cur - 1));
        break;
    case VK_PRIOR:
    case VK_NUMPAD9:
        mn->comp[0] = '\0';
        menu_moveup(mn, MIN(mn->ysize - 1, mn->cur));
        break;
    case VK_END:
    case VK_NUMPAD1:
        mn->comp[0] = '\0';
        menu_movedown(mn, mn->len - mn->cur - 1);
        break;
    case VK_HOME:
    case VK_NUMPAD7:
        mn->comp[0] = '\0';
        menu_moveup(mn, mn->cur);
        break;
    case VK_BACK:
        if(mn->comp[0] != '\0') {
            mn->comp[strlen(mn->comp) - 1] = '\0';
        }
        if(mn->comp[0] == '\0') {
            menu_scrollto(mn, 0);
            break;
        }
        l = strlen(mn->comp);
        while(i < mn->len) {
            if(!_strnicmp(mn->data[i], mn->comp, l)) {
                k = i;
                break;
            }
            i++;
        }
        menu_scrollto(mn, k);
        break;
    case VK_INSERT:
    case VK_NUMPAD0:
        return 4;
    case VK_RETURN:
        if(mn->len > 0) {
            return 3;
        }
        break;
    case VK_ESCAPE:
        return 2;
    default:
        if(isprint(keyevent.uChar.AsciiChar)) {
            c = toupper(keyevent.uChar.AsciiChar);
            l = strlen(mn->comp);
            if(l == mn->xsize) {
                break;
            }
            if(mn->comp[0] == '\0') {
                while(i < mn->len) {
                    if(toupper(mn->data[i][0]) == c) {
                        k = i;
                        break;
                    }
                    i++;
                }
            } else {
                i = mn->cur;
                while(i < mn->len) {
                    if(!_strnicmp(mn->data[i], mn->comp, l) && toupper(*(mn->data[i] + l)) == c) {
                        k = i;
                        break;
                    }
                    i++;
                }
            }
            menu_scrollto(mn, k);
            if(k != -1) {
                mn->comp[l] = c;
                mn->comp[l + 1] = '\0';
            }
            break;
        }
        return 0;
    }
    menu_draw(mn);
    return 1;
}

// 10th line - top border
// 11-12 - message
// 13th line - bottom border
// width 70
void menu_disperr() {
    DWORD temp;
    COORD c;
    INPUT_RECORD ir;
    char *line;
    unsigned int width;
    if(globerr[0] == '\0') {
        return;
    }
    globerr[ERRMSG_WIDTH - 5] = '\0';
    width = MAX(strlen(sPress), strlen(globerr)) + 4;
    line = malloc(width);
    // draw top border
    line[0] = (char)201; // top left double corner
    memset(line + 1, 205, width - 2); // horizontal double line
    line[width - 1] = (char)187; // top right double corner
    c.X = (80 - width)/2;
    c.Y = 10;
    WriteConsoleOutputCharacter(hconout, line, width, c, &temp);
    FillConsoleOutputAttribute(hconout, FOGR_BLUE | BACKGR_GREY, width, c, &temp);
    // draw bottom border
    line[0] = (char)200; // bottom left double corner
    line[width - 1] = (char)188; // bottom right corner
    c.Y = 13;
    WriteConsoleOutputCharacter(hconout, line, width, c, &temp);
    FillConsoleOutputAttribute(hconout, FOGR_BLUE | BACKGR_GREY, width, c, &temp);
    // draw error message
    line[0] = (char)186; // vertical double line
    line[width - 1] = (char)186; // vertical double line
    memset(line + 1, 32, width - 2);
    memcpy(line + 2, globerr, strlen(globerr));
    c.Y = 11;
    WriteConsoleOutputCharacter(hconout, line, width, c, &temp);
    FillConsoleOutputAttribute(hconout, FOGR_BLUE | BACKGR_GREY, width, c, &temp);
    // draw "press any key" message
    memset(line + 1, 32, width - 2);
    memcpy(line + (width - strlen(sPress))/2, sPress, strlen(sPress));
    c.Y = 12;
    WriteConsoleOutputCharacter(hconout, line, width, c, &temp);
    FillConsoleOutputAttribute(hconout, FOGR_BLUE | BACKGR_GREY, width, c, &temp);
    free(line);
    while(1) {
        ReadConsoleInput(hconin, &ir, 1, &temp);
        if(ir.EventType == KEY_EVENT && ir.Event.KeyEvent.bKeyDown && ir.Event.KeyEvent.wVirtualKeyCode == VK_RETURN) {
            break;
        }
    }
    memset(globerr, 0, width);
    c.X = 0;
    c.Y = 10;
    FillConsoleOutputCharacter(hconout, ' ', 80 * 4, c, &temp);
    FillConsoleOutputAttribute(hconout, BACKGR_BLUE, 80 * 4, c, &temp);
}
