/* rcon.c
 * rconip network communication
 * 
 * Copyright (C) 2002-2003 Paul Pergamenshchik <pp64@cornell.edu>
 *
 * $Id: rcon.c 1.12 Fri, 30 Aug 2002 20:13:05 -0400 pahan $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include<winsock2.h>
#include<windows.h>
#include<process.h>
#include<stdlib.h>
#include"rcon.h"
#include"md5.h"

int rabort;

session_t sess;
char globerr[ERRMSG_WIDTH];

request_t rcon_newrequest(int code, int screenid, int datalen) {
    request_t request = malloc(sizeof(struct request));
//  char s[1024];
//  sprintf(s, "request %d sent\n", code);
//  OutputDebugString(s);
    request->datalen = 0;
    switch(code) {
    case REQ_DIGEST:
        request->datalen = 16;
        break;
    case REQ_SCREEN_INPUT:
        request->datalen = 2;
        break;
    case REQ_PROXY_CONNECT:
        request->datalen = screenid == 1 ? 12 : 6;
        break;
    case REQ_SCREEN_ACK:
        request->datalen = 4;
        break;
    }
    if(datalen != -1) {
        request->datalen = datalen;
    }
    request->data = malloc(8 + request->datalen);
    request->data[0] = (char)(request->datalen >> 8);
    request->data[1] = (char)request->datalen;
    request->data[2] = (char)(code >> 8);
    request->data[3] = (char)code;
    request->data[4] = (char)(screenid >> 24);
    request->data[5] = (char)(screenid >> 16);
    request->data[6] = (char)(screenid >> 8);
    request->data[7] = (char)screenid;
    return request;
}

void rcon_reqactivate(unsigned int id) {
    request_t request = rcon_newrequest(REQ_SCREEN_ACTIVATE, id, 0);
#ifdef USE_SSL
	if (useSSL)
		SSL_write_RT (sess->ssl, request->data, 8 + request->datalen);
	else
#endif
		send(sess->s, request->data, 8 + request->datalen, 0);
    free(request->data);
    free(request);
}

void rcon_reqproxyconnect(char *host, u_short port, u_short spx) {
    request_t request = rcon_newrequest(REQ_PROXY_CONNECT_NAME, 0, strlen(host)+1);

	request->data[4] = (char)spx;
    request->data[5] = (char)(spx >> 8);
    request->data[6] = (char)(port >> 8);
    request->data[7] = (char)port;

    memcpy(request->data + 8, host, strlen(host));
	request->data[strlen(host)+8] = '\0';

    send(sess->s, request->data, 8 + request->datalen, 0);
    free(request->data);
    free(request);
}

void rcon_reqdigest(char *hash) {
    request_t request = rcon_newrequest(REQ_DIGEST, 0, -1);
    memcpy(request->data + 8, hash, 16);
#ifdef USE_SSL
	if (useSSL)
		SSL_write_RT (sess->ssl, request->data, 8 + request->datalen);
	else
#endif
	    send(sess->s, request->data, 8 + request->datalen, 0);
    free(request->data);
    free(request);
}

void rcon_reqinput(unsigned int id, short inp) {
    request_t request = rcon_newrequest(REQ_SCREEN_INPUT, id, -1);
    request->data[8] = (char)(inp >> 8);
    request->data[9] = (char)inp;
#ifdef USE_SSL
	if (useSSL)
		SSL_write_RT (sess->ssl, request->data, 8 + request->datalen);
	else
#endif
	    send(sess->s, request->data, 8 + request->datalen, 0);
    free(request->data);
    free(request);
}

void rcon_reqack(unsigned int id, int stamp) {
    request_t request = rcon_newrequest(REQ_SCREEN_ACK, id, 4);
    request->data[8] = (char)(stamp >> 24);
    request->data[9] = (char)(stamp >> 16);
    request->data[10] = (char)(stamp >> 8);
    request->data[11] = (char)(stamp);
#ifdef USE_SSL
	if (useSSL)
		SSL_write_RT (sess->ssl, request->data, 8 + request->datalen);
	else
#endif
		send(sess->s, request->data, 8 + request->datalen, 0);
    free(request->data);
    free(request);
}

void rcon_reqopen(unsigned int id) {
    request_t request = rcon_newrequest(REQ_SCREEN_OPEN, id, -1);
#ifdef USE_SSL
	if (useSSL)
		SSL_write_RT (sess->ssl, request->data, 8 + request->datalen);
	else
#endif
	    send(sess->s, request->data, 8 + request->datalen, 0);
    free(request->data);
    free(request);
}

void rcon_reqclose(unsigned int id) {
    request_t request = rcon_newrequest(REQ_SCREEN_CLOSE, id, -1);
#ifdef USE_SSL
	if (useSSL)
		SSL_write_RT (sess->ssl, request->data, 8 + request->datalen);
	else
#endif
		send(sess->s, request->data, 8 + request->datalen, 0);
    free(request->data);
    free(request);
}

void rcon_reqreset(unsigned int id) {
    request_t request = rcon_newrequest(REQ_SCREEN_RESET, id, -1);
#ifdef USE_SSL
	if (useSSL)
		SSL_write_RT (sess->ssl, request->data, 8 + request->datalen); 
	else
#endif
	    send(sess->s, request->data, 8 + request->datalen, 0);
    free(request->data);
    free(request);
}

int recv_full(char *buf, int len) {
    int res;
    int received = 0;

    while(received < len) {
#ifdef USE_SSL
	if (useSSL)
		res = SSL_read_RT (sess->ssl, buf + received, len - received);
	else
#endif
        res = recv(sess->s, buf + received, len - received, 0);

		// disconnected gracefully
        if(res == 0) {
            break;
        }

        if(res == SOCKET_ERROR) {
            return SOCKET_ERROR;
        }
        received += res;
    }
    return received;
}

// slurp 8 bytes into a struct reply
void munch_header(char *buf1, reply_t reply) {
    unsigned char *buf = buf1;
    reply->datalen = (buf[0] << 8) + buf[1];
    reply->code = (buf[2] << 8) + buf[3];
    reply->screenid = (buf[4] << 24) + (buf[5] << 16) + (buf[6] << 8) + buf[7];
}

unsigned __stdcall rcon_sockthread(void *arg) {
    char header[HEADER_SIZE];
    reply_t reply;
    int res;
//	char s[1024];

	while(1) {
        fd_set rfd;
        FD_ZERO(&rfd);
        FD_SET(sess->s, &rfd);
        res = select(1, &rfd, NULL, NULL, NULL);
        if(res == SOCKET_ERROR) {
            res = WSAGetLastError();
            if(res == 10058 || res == 10054 || res == 10038) { // shutdown or reset
//              PERR "Connection terminated (%d)", res);
            } else {
                PERR "select: Unknown socket error (%d)", res);
            }
            return 1;
        }

        res = recv_full(header, HEADER_SIZE);
        if(res == SOCKET_ERROR) {
            res = WSAGetLastError();
            if(res == 10058 || res == 10054 || res == 10038) { // shutdown or reset
//              PERR "Connection terminated (%d)", res);
            } else {
                PERR "recv: Unknown socket error (%d)", res);
            }
			if (res == 10054)
				ReleaseSemaphore(sess->qusem, 1, NULL);
			rabort=1;
            return 1;
        }
//        if(res < HEADER_SIZE) {
//            PERR "Connection timed out");
//            return 1;
//        }

        reply = malloc(sizeof(struct reply));
        munch_header(header, reply);

//		sprintf(s, "reply %d received\n", reply->code);
//		OutputDebugString(s);

        reply->data = malloc(reply->datalen);
        if(reply->datalen != 0) {
            res = recv_full(reply->data, reply->datalen);
            if(res == SOCKET_ERROR) {
                free(reply->data);
                free(reply);
                res = WSAGetLastError();
            if(res == 10058 || res == 10054 || res == 10038) { // shutdown or reset
//                  PERR "Connection terminated (%d)", res);
                } else {
                    PERR "recv: Unknown socket error (%d)", res);
                }
                break;
            }
            if(res < reply->datalen) {
                free(reply->data);
                free(reply);
//                PERR "Connection terminated by peer");
                break;
            }
        }

        if(queue_push(reply)) {
			//Dropped packet
            free(reply->data);
            free(reply);
        }
    }
    return 0;
}

u_short rcon_plookup(char *port) {
    u_short rport = htons((u_short)atoi(port));
    if(rport == 0) {
        struct servent *se = getservbyname(port, "tcp");
        if(se != NULL) {
            rport = se->s_port;
        }
    }
    return rport;
}

u_long rcon_alookup(char *hostname) {
    u_long raddr = inet_addr(hostname);
    if(raddr == INADDR_NONE) {
        struct hostent *he = gethostbyname(hostname);
        if(he != NULL) {
            raddr = *(u_long*)he->h_addr_list[0];
        }
    }
    return raddr;
}

int rcon_connect(u_long ip, u_short port) {
    struct sockaddr_in remotea;
    int tmp;
#ifdef USE_SSL
	SSL_METHOD *meth;
	int err;
#endif
    sess->s = socket(AF_INET, SOCK_STREAM, 0);
    if(sess->s == INVALID_SOCKET) {
        PERR "Error creating socket (%d)", WSAGetLastError());
        return 1;
    }
    remotea.sin_family = AF_INET;
    remotea.sin_port = port;
    remotea.sin_addr.s_addr = ip;
    if(connect(sess->s, (struct sockaddr*)&remotea, sizeof(struct sockaddr_in)) == SOCKET_ERROR) {
        tmp = WSAGetLastError();
        if(tmp == 10049) {
            PERR "Invalid address");
        } else if(tmp == 10061) {
			if (useSSL)
			{
	            PERR "SSL connection refused");
			}
			else
			{
	            PERR "Connection refused");
			}
        } else {
            PERR "Error connecting (%d)", tmp);
        }
        closesocket(sess->s);
        return 1;
    }
#ifdef USE_SSL
	if (useSSL)
	{
		SSL_library_init_RT();
		SSL_load_error_strings_RT();
		meth = SSLv3_client_method_RT();
		sess->ctx = SSL_CTX_new_RT (meth);                       
		if (sess->ctx == NULL) return 1;
		sess->ssl = SSL_new_RT (sess->ctx);                         
		if (sess->ssl == NULL) return 1;
		SSL_set_fd_RT (sess->ssl, sess->s);
		err = SSL_connect_RT (sess->ssl);
		if (err <= 0) PERR "Could not establish SSL connection");
	}
#endif
    return 0;
}

void rcon_init() {
    unsigned temp;
    sess->qusem = CreateSemaphore(NULL, 0, MAXQLEN, NULL);
    sess->netth = (HANDLE)_beginthreadex(NULL, 0, rcon_sockthread, NULL, 0, &temp);
    sess->qumut = CreateMutex(NULL, 0, NULL);
    sess->servername = NULL;
    sess->codepage = NULL;
    sess->fmenu = 3;
	sess->sync = syncScreen; // Global variable set from args
    sess->menuth = NULL;
}

int rcon_proxyconnect(char *host, u_short prt) {
    reply_t reply;
    u_short port = 0;
    char *sep;

    if(WaitForSingleObject(sess->qusem, INITIAL_TIMEOUT_SEC) == WAIT_TIMEOUT) {
        PERR "Proxy connect failure");
        return 1;
    }
    reply = queue_pop();
    if(reply->code != REP_PROXY) {
        PERR "Proxy connect failure");
        return 1;
    }

	port = prt;
	sep = strrchr(host, 58); // ':'
    if(sep) {
		sep[0] = '\0';
        port = rcon_plookup(sep + 1);
	}
    if(!port) {
		if (proxySPX)
			port = htons(16800);
		else
			port = htons(2034);
    }

	rabort=0;
    rcon_reqproxyconnect(host, port, proxySPX);

	if(WaitForSingleObject(sess->qusem, INITIAL_TIMEOUT_SEC) == WAIT_TIMEOUT) {
        PERR "Proxy connect timeout");
        return 1;
    }
	if (rabort) 
	{
        PERR "Proxy could not connect to %s",host);
		return 1;
	}
	return 0;
}
	
int rcon_authorize(char *password, int proxyConnected) {
    tstream_t stream;
    reply_t reply;
    char *nonce;
    char hash1[32];
    char hash[32];
    if(password == NULL) {
        return 1;
    }
	if(!proxyConnected)
	{
		if(WaitForSingleObject(sess->qusem, INITIAL_TIMEOUT_SEC) == WAIT_TIMEOUT) {
			PERR "Authorization failure");
	        return 1;
		}
    }
    reply = queue_pop();
    if(reply->code == REP_PROXY) {
        PERR "Authorization failure");
        return 1;
    }
    if(reply->code != REP_DIGEST_NONCE) {
        PERR "Authorization failure");
        return 1;
    }
    nonce = reply->data;
	free(reply);

    if(WaitForSingleObject(sess->qusem, INITIAL_TIMEOUT_SEC) == WAIT_TIMEOUT) {
		PERR "Authorization failure");
        return 1;
	}

    reply = queue_pop();
	if(reply->code != REP_SERVER_NAME) {
        PERR "Authorization failure");
		return 1;
    }

    stream = tstream_init(reply->data, reply->datalen);
    sess->servername = tstream_readstr(stream);
    sess->codepage = tstream_readstr(stream);
    tstream_destroy(stream);
    free(reply->data);
    free(reply);
    md5_buffer(password, MIN(strlen(password), MAXPWDLEN), hash1);
    memcpy(hash1 + 16, nonce, 4);
    md5_buffer(hash1, 20, hash);
    rcon_reqdigest(hash);
    free(nonce);

    if(WaitForSingleObject(sess->qusem, INITIAL_TIMEOUT_SEC) == WAIT_TIMEOUT) {
        PERR "Authorization failure");
        return 1;
    }
    reply = queue_pop();
    if(reply->code == REP_DIGEST_ERROR) {
        PERR "Authentication failure");
        return 1;
    }
    if(reply->code != REP_DIGEST_OK) {
        PERR "Authorization failure");
        return 1;
    }
    free(reply->data);
    free(reply);
    if(WaitForSingleObject(sess->qusem, INITIAL_TIMEOUT_SEC) == WAIT_TIMEOUT) {
        PERR "Authorization failure");
        return 1;
    }
    reply = queue_pop();
    if(reply->code != REP_SCREENLIST) {
        PERR "Authorization failure");
        return 1;
    }
    screen_init();
    if(screen_handle(reply)) {
        return 1;
    }
    return 0;
}

void rcon_mainloop() {
    HANDLE handles[3];
    int res, brk = 0;
    reply_t reply;
    handles[0] = sess->qusem;
    handles[1] = sess->netth;
    handles[2] = hconin;
    while(!brk) {
        res = WaitForMultipleObjects(sess->fmenu, handles, 0, INFINITE);
        switch(res) {
        case WAIT_OBJECT_0 + 2:
            screen_handlecon();
            break;
        case WAIT_OBJECT_0 + 0:
            reply = queue_pop();
            if(screen_handle(reply)) {
                screen_cleanup();
                brk = 1;
            }
            break;
        case WAIT_OBJECT_0 + 1:
            screen_cleanup();
            brk = 1;
            break;
        }
    }
}

void rcon_cleanup() {
#ifdef USE_SSL
	if (useSSL)
	{
		SSL_shutdown_RT (sess->ssl);
		shutdown (sess->s, SD_SEND);
		SSL_shutdown_RT (sess->ssl);  
		SSL_free_RT (sess->ssl);
		SSL_CTX_free_RT (sess->ctx);
	}
#endif
    closesocket(sess->s);
    free(sess->codepage);
    free(sess->servername);
    CloseHandle(sess->qumut);
    WaitForSingleObject(sess->netth, INFINITE);
    CloseHandle(sess->netth);
    CloseHandle(sess->qusem);
}

int rcon_session(char *host, char *pwd, u_short prt) {
	int useProxy=0;
    u_short port = 0;
    int res = 0;
    u_long ipaddr;
    char *sep;

	if (strlen(proxy)!=0) useProxy=1;
    
	if (!useProxy)
	{
		// use default port or get one from hostname or get one from argument
		port = prt;
		sep = strrchr(host, 58); // ':'
	    if(sep) {
			sep[0] = '\0';
	        port = rcon_plookup(sep + 1);
		}
	    if(!port) {
			if (useSSL)
				port = htons(2036);
			else
				port = htons(2034);
	    }
		ipaddr = rcon_alookup(host);
	    if(ipaddr == INADDR_NONE) {
			PERR "Unable to resolve hostname %s", host);
	        menu_disperr();
			return 1;
	    }
	}
	else
	{
 		// use default port or get one from hostname or get one from argument
		sep = strrchr(proxy, 58); // ':'
	    if(sep) {
			sep[0] = '\0';
	        port = rcon_plookup(sep + 1);
		}
	    if(!port) {
			port = htons(2035);
	    }
		ipaddr = rcon_alookup(proxy);
	    if(ipaddr == INADDR_NONE) {
			PERR "Unable to resolve proxy hostname %s", proxy);
	        menu_disperr();
			return 1;
	    }
	}

    memset(globerr, 0, ERRMSG_WIDTH);
    sess = malloc(sizeof(struct session));
    if(!rcon_connect(ipaddr, port))
	{
		rcon_init();
		if(useProxy) res = rcon_proxyconnect(host,prt);
		if (!res)
		{
	        if(!rcon_authorize(pwd, useProxy)) {
				rcon_mainloop();
	        } else {
				res = 1;
	        }
			rcon_cleanup();
		}
    } else {
        res = 1;
    }
    menu_disperr();
    free(sess);
    return res;
}
