#!/bin/bash
# config.sh - Gathers system configuration information for NTS.
#             Creates a tarball to attach to the SR or send to
#             support. Basic info is collected for SLES8. More
#             comprehensive info is collected for SLES9/OES.
#             jrecord@novell.com (c) 2005, Novell, Inc.
SVER=1.08
SDATE="2005 10 07"

# changes
##############################################################################
# 0.95 initial release, beta test
# 1.00 added SPident, tarball filename include hostname and timestamp
# 1.01 added lvm, and only 500 lines of messages
# 1.02 added software raid check
#      changed vmstat.txt to performance.txt
#      added free & ps to performance.txt
#      added evms_gather_info
# 1.03 added *-release files
# 1.05 added multipath I/O & boot services status
#      added root execution detection
# 1.06 added iscsi, script date
#      stderr was not specified for some commands
#      added network, PAM
#      optimized with functions
#      added OES components: ncp, nss
# 1.07 added sfisk -l to iscsi
#      added OES component: ncs
#      added and optimized with log_files, log_write
#      added function headers, ie ====
# 1.08 added echolog to create config.sh.txt
#      fixed mpio '\ls -lR /dev/disk'
#      changed vmstat 5 3 => vmstat 1 4
#      commented out NCS, hangs on some unstable nodes
#      modified log_write
#      updated rpm -qa display format
#      removed rm -rf /tmp/*


# To do list
##############################################################################
# Fix NCS on unstable nodes
#

title() {
	echo "==========================================================="
	echo "      Novell Technical Support - Information Files"
	echo "                  Script Version: $SVER"
	echo "                Script Date: $SDATE"
	echo "==========================================================="
	echo
}

# Input: logfilename logfiles...
conf_files() {
	LOGFILE=$LOG/$1
	shift
	for CONF in $@
	do
		echo "#==[ Configuration File ]===========================#" >> $LOGFILE
		if [ -f $CONF ]; then
			echo "# $CONF" >> $LOGFILE
			cat $CONF | sed -e '/^#/d' -e '/^;/d' -e '/^    #/d' -e 's/
//g' -e '/^$/d' >> $LOGFILE
		else
			echo "# $CONF - File not found" >> $LOGFILE
		fi
		echo >> $LOGFILE
	done
	echo >> $LOGFILE
}

# Input: logfilename lines logfiles...
log_files() {
	LOGFILE=$LOG/$1
	shift
	LOGLINES=$1
	shift
	for CONF in $@
	do
		echo "#==[ Log File ]=====================================#" >> $LOGFILE
		if [ -f $CONF ]; then
			echo "# $CONF - Last $LOGLINES Lines" >> $LOGFILE
			cat $CONF | tail -$LOGLINES | sed -e 's/
//g' >> $LOGFILE
		else
			echo "# $CONF - File not found" >> $LOGFILE
		fi
		echo >> $LOGFILE
	done
	echo >> $LOGFILE
}

# Input: logfilename "text"
log_write() {
	LOGFILE=$LOG/$1
	shift
	echo "$@" >> $LOGFILE
	echo >> $LOGFILE
}

echolog() {
	LOGFILE=$LOG/config.sh.txt
	echo "$@" | tee -a $LOGFILE
}

# Input: logfilename command
log_cmd() {
	LOGFILE=$LOG/$1
	shift
	CMD="$@"
	echo "#==[ Command ]======================================#" >> $LOGFILE
	echo "# $CMD" >> $LOGFILE
	$CMD >> $LOGFILE 2>&1
	echo >> $LOGFILE
}

# Input: logfilename rpm
# Assumes the rpm is installed
rpm_verify() {
	LOGFILE=$LOG/$1
	RPM=$2
	echo "#==[ Verification ]=================================#" >> $LOGFILE
	echo "# rpm -V $RPM" >> $LOGFILE
	rpm -V $RPM >> $LOGFILE 2>&1
	if [ $? -gt 0 ]; then
		echo "# Verification Status: Differences Found" >> $LOGFILE
	else
		echo "# Verification Status: Passed" >> $LOGFILE
	fi
	echo >> $LOGFILE
}

#############################################################################################
#  main 
#############################################################################################

CURRENTUID=$(id -u)
if [ $CURRENTUID -ne 0 ]; then
	clear
	title
	echo "ERROR: You must be logged in as root."
	echo "       $(id)"
	echo
	exit 1
fi

SRV=$(hostname)
if [ "$SRV" = "" ]; then
	SRV=novell
fi
TIME=$(date +"%y%m%d_%H%M")
BASE=nts_${SRV}_${TIME}
LOG=/tmp/$BASE

clear
mkdir -p $LOG

title | tee -a $LOG/release.txt
RELEASE=$(ls -1 /etc/*-release)
conf_files release.txt $RELEASE

echolog Gathering system information
echolog

# uname
log_cmd uname.txt 'uname -a'
echolog "- uname info done"

# performance data
log_cmd performance.txt 'uptime'
log_cmd performance.txt 'vmstat 1 4'
log_cmd performance.txt 'free -k'
log_cmd performance.txt 'ps aux'
echolog "- performance info done"

# rpm list of all files
log_write rpm.txt "#==[ Command ]======================================#"
log_write rpm.txt '# rpm -qa --queryformat "%-30{NAME} %-25{DISTRIBUTION} %{VERSION}-%{RELEASE}\n" | sort'
printf "%-30s %-25s %s\n" NAME DISTRIBUTION VERSION >> $LOG/rpm.txt
rpm -qa --queryformat "%-30{NAME} %-25{DISTRIBUTION} %{VERSION}-%{RELEASE}\n" | sort >> $LOGFILE 2>&1
echolog "- rpm info done"

# messages file
log_files messages.txt 500 /var/log/messages
echolog "- logfile info done"

# oes ncp
if rpm -q novell-ncpserv >/dev/null 2>&1
then
	rpm_verify ncp.txt novell-ncpserv
	log_cmd ncp.txt 'ncpcon volume'
	log_cmd ncp.txt 'ncpcon stats'
	log_cmd ncp.txt 'ncpcon config'
	log_cmd ncp.txt 'ncpcon connection'
	conf_files ncp.txt /etc/opt/novell/ncpserv.conf
	echolog "- OES ncp info done"
else
	log_write ncp.txt "# Missing novell-ncpserv rpm"
	echolog "- OES ncp info SKIPPED: Missing novell-ncpserv rpm"
fi

# oes nss
if rpm -q novell-nss >/dev/null 2>&1
then
	LOGFILE=$LOG/nss.txt
	rpm_verify nss.txt novell-nss
	log_cmd nss.txt 'nss /pools'
	echo "# mount | grep -i nss" >> $LOGFILE
	mount | grep -i nss >> $LOGFILE
	echo >> $LOGFILE
	conf_files nss.txt /opt/novell/nss/conf/nssstart.cfg
	echolog "- OES nss info done"
else
	log_write nss.txt "# Missing novell-nss rpm"
	echolog "- OES nss info SKIPPED: Missing novell-nss rpm"
fi

# NOTE: Gathering novell cluster stats on an unstable node
#       can cause the information gathering process to hang.
#
# oes ncs
#if rpm -q novell-cluster-services >/dev/null 2>&1
#then
#	SCRIPTS=$(find /etc/opt/novell/ncs -type f)
#	RESOURCELOGS=$(find /var/opt/novell/ncs -type f -name \*out)
#
#	rpm_verify ncs.txt novell-cluster-services
#	log_cmd ncs.txt 'sfdisk -l'
#	log_cmd ncs.txt 'sbdutil -f'
#	log_cmd ncs.txt 'sbdutil -v'
#	log_cmd ncs.txt 'cluster stats display'
#	log_cmd ncs.txt 'cluster view'
#	log_cmd ncs.txt 'cluster resources'
#	log_cmd ncs.txt 'cluster pools'
#	log_cmd ncs.txt 'cluster info all'
#	conf_files ncs.txt "$SCRIPTS"
#	log_files ncs.txt 500 "/_admin/Novell/Cluster/EventLog.xml $RESOURCELOGS"
#	echolog "- OES ncs info done"
#else
#	log_write ncs.txt "# Missing novell-cluster-services rpm"
#	echolog "- OES ncs info SKIPPED: Missing novell-cluster-services rpm"
#fi


# PAM
rpm_verify pam.txt pam
rpm_verify pam.txt pam-modules
FILES="/etc/nsswitch.conf $(find /etc/pam.d -type f)"
conf_files pam.txt $FILES
echolog "- PAM info done"

# system services
log_cmd chkconfig.txt 'chkconfig --list'
LOGFILE=$LOG/chkconfig.txt
echo "# individual boot services:" >> $LOGFILE
for BOOTSERVICE in $(find /etc/init.d -type f -name boot\*)
do
	NAME=$(basename $BOOTSERVICE)
	STATUS=$(chkconfig $NAME)
	printf "%25s:  %-4s\n" $(echo $STATUS | awk '{print $1}') $(echo $STATUS | awk '{print $2}') >> $LOGFILE
done
echolog "- system services info done"

# software raid 
conf_files softraid.txt /proc/mdstat
if rpm -q mdadm >/dev/null 2>&1
then
	rpm_verify softraid.txt mdadm
	log_cmd softraid.txt 'mdadm --detail --scan'
else
	log_write softraid.txt "# Missing mdadm rpm"
fi
if rpm -q raidtools >/dev/null 2>&1
then
	rpm_verify softraid.txt raidtools
	LOGFILE=$LOG/softraid.txt
	for MDEVICE in $(cat /proc/mdstat | grep ^md | awk '{print $1}')
	do
		echo "# lsraid -a /dev/$MDEVICE" >> $LOGFILE
		lsraid -a /dev/$MDEVICE >> $LOGFILE 2>&1
		echo >> $LOGFILE
	done
else
	log_write softraid.txt "# Missing raidtools rpm"
fi
conf_files softraid.txt /etc/mdadm.conf /etc/raidtab
echolog "- software raid info done"

# lvm
if rpm -q lvm2 >/dev/null 2>&1
then
	LOGFILE=$LOG/lvm.txt
	rpm_verify lvm.txt lvm2
	printf "\n\n%s\n\n" "##############################################################################################" >> $LOGFILE
	log_cmd lvm.txt 'pvs -vvvv'
	log_cmd lvm.txt 'vgs -vvvv'
	log_cmd lvm.txt 'lvs -vvvv'
	conf_files lvm.txt /etc/lvm/lvm.conf
	echo "# Run 'grep -v # lvm.txt' for a summary" >> $LOGFILE
	echolog "- LVM info done"
else
	log_write lvm.txt "# Missing lvm2 rpm"
	echolog "- LVM info SKIPPED: Missing lvm2 rpm"
fi

# multipath
if rpm -q multipath-tools >/dev/null 2>&1
then
	rpm_verify mpio.txt multipath-tools
	log_cmd mpio.txt 'multipath -v 3 -l'
	log_cmd mpio.txt 'ls -lR /dev/disk'
	echolog "- multipath info done"
else
	log_write mpio.txt "# Missing multipath-tools rpm"
	echolog "- multipath info SKIPPED: Missing multipath-tools rpm"
fi

# evms
if rpm -q evms >/dev/null 2>&1
then
	rpm_verify evms.txt evms
	log_cmd evms.txt 'evms_gather_info'
	test -f gather_info.qry && rm -f gather_info.qry
	echolog "- evms info done"
else
	log_write evms.txt "# Missing evms rpm"
	echolog "- evms info SKIPPED: Missing evms rpm"
fi

# SPident
if rpm -q SPident >/dev/null 2>&1
then
	rpm_verify spident.txt SPident
	log_cmd spident.txt 'SPident -vv'
	echolog "- SPident info done"
else
	log_write spident.txt "# Missing SPident rpm"
	echolog "- SPident info SKIPPED: Missing SPident rpm"
fi

# hwinfo
log_cmd hwinfo.txt 'hwinfo'
echolog "- hwinfo info done"

# networking
log_cmd network.txt 'ifconfig -a'
log_cmd network.txt 'netstat -nlp'
log_cmd network.txt 'netstat -nr'
conf_files network.txt /etc/resolv.conf
echolog "- network info done"

# iscsi
if rpm -q linux-iscsi >/dev/null 2>&1
then
	rpm_verify iscsi.txt linux-iscsi
	log_cmd iscsi.txt 'iscsi-ls -l'
	log_cmd iscsi.txt 'sfdisk -l'
	conf_files iscsi.txt /etc/iscsi.conf /etc/initiatorname.iscsi
	echolog "- iscsi info done"
else
	log_write iscsi.txt "# Missing linux-iscsi rpm"
	echolog "- iscsi info SKIPPED: Missing linux-iscsi rpm"
fi

# siga
if rpm -q siga >/dev/null 2>&1
then
	LOGFILE=$LOG/siga.txt
	echolog "Starting siga"
	rpm_verify siga.txt siga
	siga no | tee -a $LOG/config.sh.txt
	echo "# siga no" >> $LOGFILE
	echo >> $LOGFILE
	cat /tmp/siga/siga.txt >> $LOGFILE
	rm -rf /tmp/siga*
else
	log_write siga.txt "# Missing siga rpm"
	echolog "Starting siga SKIPPED: Missing siga rpm"
fi

# creating tarball for NTS
echo
echo Creating tarball
cd /tmp
tar zcf ${BASE}.tgz ${BASE}/*
rm -rf $LOG
LOGSIZE=$(ls -lh ${LOG}.tgz | awk '{print $5}')

echo
echo "==[ DONE ]================================================="
echo "  Log file tarball: ${LOG}.tgz"
echo "  Log file size:    $LOGSIZE"
echo
echo "  Please attach the log file tarball to your open Service"
echo "  Request at the following URL:"
echo "  https://secure-support.novell.com/eService_enu"
echo 
echo "  If you cannot attach the tarball to the SR, then email"
echo "  it to the engineer."
echo "==========================================================="
echo; echo; echo; echo; echo
exit 0
